package com.plusmpm.parser.translation.wrapper.form.datachooser;

import com.google.common.collect.HashMultimap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Multimap;
import com.plusmpm.parser.translation.utils.Constants;
import com.plusmpm.parser.translation.utils.PackageElementsExtractor;
import com.plusmpm.parser.translation.utils.XpdlKeyResolver;
import com.plusmpm.parser.translation.wrapper.ElementWrapper;
import com.plusmpm.parser.translation.wrapper.form.datachooser.exception.DataChooserSettingDoesNotExist;
import com.plusmpm.parser.wrapper.DataChooser;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.enhydra.shark.xpdl.elements.DataField;
import org.enhydra.shark.xpdl.elements.ExtendedAttribute;

import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Stream;

@Slf4j
@AllArgsConstructor
public abstract class AbstractDataChooserWrapper
                extends ElementWrapper
{
    protected DataField field;

    @Override
    public Map<String, String> getProperties()
    {
        Map<String, String> map = new LinkedHashMap<>();

        try
        {
            DataChooser dataChooserXpdlKey = XpdlKeyResolver.extractDataChooser( field );

            getTranslatedMappings().forEach( ( mappingName, mappingTranslation ) -> {
                map.put( dataChooserXpdlKey.getMapping( mappingName ), mappingTranslation );
            } );
        }
        catch ( Exception ex )
        {
            log.error( ex.getMessage(), ex );
        }

        return map;
    }

    protected abstract Map<String, String> getTranslatedMappings();

    protected static Multimap<String, String> findDataChooserSettings( DataField field )
    {
        Optional<ExtendedAttribute> optionalAttribute =
                        PackageElementsExtractor.getExtendedAttribute( field, Constants.DATA_CHOOSER );
        if ( optionalAttribute.isPresent() )
        {
            String attrValue = optionalAttribute.get().getVValue();

            Multimap<String, String> result = HashMultimap.create();
            Stream.of( attrValue.split( "\n" ) )
                            .map( dcOption -> dcOption.split( "=" ) )
                            .forEach( split -> {
                                String key = split[0].trim();
                                String value = StringUtils.EMPTY;

                                if ( split.length > 1 )
                                {
                                    value = split[1].trim();
                                }

                                result.put( key, value );
                            } );

            return result;
        }
        else
        {
            return HashMultimap.create();
        }
    }

    protected static String findDataChooserSetting( DataField field, String name )
                    throws DataChooserSettingDoesNotExist
    {
        Multimap<String, String> settings = findDataChooserSettings( field );
        if ( settings.containsKey( name ) )
        {
            return Iterables.get( settings.get( name ), 0 );
        }
        else
        {
            throw new DataChooserSettingDoesNotExist();
        }
    }

    @Override
    public String getKey()
    {
        return null;
    }

    @Override
    public String getRawName()
    {
        return null;
    }

    static boolean isDataChooser( DataField field )
    {
        return PackageElementsExtractor.getExtendedAttribute( field, Constants.DATA_CHOOSER ).isPresent();
    }
}
