package com.plusmpm.parser.translation.wrapper.general;

import com.plusmpm.parser.translation.utils.PackageElementsExtractor;
import com.plusmpm.parser.translation.utils.XpdlKeyResolver;
import com.plusmpm.parser.translation.wrapper.ElementWrapper;
import com.plusmpm.parser.translation.wrapper.form.DataFieldWrapper;
import com.plusmpm.parser.translation.wrapper.form.datachooser.AbstractDataChooserWrapper;
import com.plusmpm.parser.translation.wrapper.form.datachooser.ClassDataChooserWrapper;
import com.plusmpm.parser.translation.wrapper.form.datachooser.ComponentDataChooserWrapper;
import lombok.AllArgsConstructor;
import lombok.extern.slf4j.Slf4j;
import org.enhydra.shark.xpdl.elements.ExtendedAttribute;
import org.enhydra.shark.xpdl.elements.WorkflowProcess;

import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import java.util.stream.Stream;

@Slf4j
@AllArgsConstructor
public class ProcessWrapper
                extends ElementWrapper
{
    private WorkflowProcess process;

    @Override
    public String getKey()
    {
        return XpdlKeyResolver.extract( process ).getKey();
    }

    @Override
    public String getRawName()
    {
        return process.getName();
    }

    @Override
    public Map<String, String> getProperties()
    {
        Map<String, String> map = super.getProperties();
        map.put( XpdlKeyResolver.extract( process ).getKeyDesc(), process.getProcessHeader().getDescription() );

        List<ExtendedAttribute> extendedAttributes = PackageElementsExtractor.getExtendedAttributes( process );
        Optional<ExtendedAttribute> optionalProcessNameMask =
                        PackageElementsExtractor.getProcessNameMask( extendedAttributes );

        if ( optionalProcessNameMask.isPresent() )
        {
            String processNameMask = XpdlKeyResolver.extract( process ).forProcessNameMask().getKey();
            String nameMaskTranslation = optionalProcessNameMask.get().getVValue().split( "_" )[0].trim();
            map.put( processNameMask, nameMaskTranslation );
        }

        for ( ParticipantWrapper participant : findParticipants() )
        {
            map.putAll( participant.getProperties() );
        }

        for ( DataFieldWrapper dataField : findDataFields() )
        {
            map.putAll( dataField.getProperties() );
        }

        for ( AbstractDataChooserWrapper dataChooser : findDataChoosers() )
        {
            map.putAll( dataChooser.getProperties() );
        }

        for ( ActivityWrapper activity : findActivities() )
        {
            map.putAll( activity.getProperties() );
        }

        return map;
    }

    private List<DataFieldWrapper> findDataFields()
    {
        return PackageElementsExtractor.getDataFields( process ).stream()
                        .map( DataFieldWrapper::new )
                        .collect( Collectors.toList() );
    }

    private List<ParticipantWrapper> findParticipants()
    {
        return PackageElementsExtractor.getParticipants( process ).stream()
                        .map( ParticipantWrapper::new )
                        .collect( Collectors.toList() );
    }

    private List<ActivityWrapper> findActivities()
    {
        return PackageElementsExtractor.getActivities( process ).stream()
                        .map( ActivityWrapper::new )
                        .collect( Collectors.toList() );
    }

    private List<AbstractDataChooserWrapper> findDataChoosers()
    {
        return Stream.concat( findComponentDataChoosers(), findClassDataChoosers() )
                        .collect( Collectors.toList() );
    }

    private Stream<ComponentDataChooserWrapper> findComponentDataChoosers()
    {
        return PackageElementsExtractor.getDataFields( process ).stream()
                        .filter( ComponentDataChooserWrapper::isTypeOf )
                        .map( ComponentDataChooserWrapper::new );
    }

    private Stream<ClassDataChooserWrapper> findClassDataChoosers()
    {
        return PackageElementsExtractor.getDataFields( process ).stream()
                        .filter( ClassDataChooserWrapper::isTypeOf )
                        .map( ClassDataChooserWrapper::new );
    }
}
