package com.plusmpm.parser.translation.wrapper.form.datachooser;

import com.plusmpm.parser.translation.utils.Constants;
import com.plusmpm.parser.translation.utils.PackageElementsExtractor;
import com.plusmpm.parser.translation.wrapper.form.datachooser.exception.DataChooserSettingDoesNotExist;
import lombok.SneakyThrows;
import lombok.extern.slf4j.Slf4j;
import org.apache.commons.lang.StringUtils;
import org.enhydra.shark.xpdl.elements.DataField;
import org.enhydra.shark.xpdl.elements.ExtendedAttribute;
import org.json.JSONException;
import org.json.JSONObject;

import java.util.Iterator;
import java.util.LinkedHashMap;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;

@Slf4j
public class ComponentDataChooserWrapper
                extends AbstractDataChooserWrapper
{
    public ComponentDataChooserWrapper( DataField field )
    {
        super( field );
    }

    @Override
    @SneakyThrows
    protected Map<String, String> getTranslatedMappings()
    {
        Map<String, JSONObject> mappings = getMappings();
        return mappings.entrySet().stream()
                        .collect( Collectors.toMap(
                                        Map.Entry::getKey,
                                        entry -> {
                                            try
                                            {
                                                return entry.getValue().getString( "name" );
                                            }
                                            catch ( JSONException e )
                                            {
                                                throw new IllegalArgumentException(
                                                                "Cannot read name of mapping " + entry.getKey() );
                                            }
                                        } ) );
    }

    private Map<String, JSONObject> getMappings()
                    throws JSONException
    {
        Optional<ExtendedAttribute> optionalDefinitionAttribute =
                        PackageElementsExtractor.getExtendedAttribute( field, Constants.DATA_CHOOSER_DEFINITION );
        if ( !optionalDefinitionAttribute.isPresent() )
        {
            log.error( "Cannot find {} in extended attributes of variable {}", Constants.DATA_CHOOSER_DEFINITION,
                       field.getId() );
            return new LinkedHashMap<>();
        }
        else
        {
            ExtendedAttribute definitionAttribute = optionalDefinitionAttribute.get();
            return resolveMappings( definitionAttribute.getVValue() );
        }
    }

    private Map<String, JSONObject> resolveMappings( String jsonDefinition )
                    throws JSONException
    {
        JSONObject jsonObject = new JSONObject( jsonDefinition );
        JSONObject mappingsObject = jsonObject.getJSONObject( "mappings" );

        Map<String, JSONObject> mappings = new LinkedHashMap<>();
        Iterator<String> iterator = mappingsObject.sortedKeys();
        while ( iterator.hasNext() )
        {
            String mappingName = iterator.next();
            mappings.put( mappingName, mappingsObject.getJSONObject( mappingName ) );
        }

        return mappings;
    }

    public static boolean isTypeOf( DataField field )
    {
        return AbstractDataChooserWrapper.isDataChooser( field ) && isComponentDataChooser( field );
    }

    private static boolean isComponentDataChooser( DataField field )
    {
        try
        {
            String dataChooserClass = findDataChooserSetting( field, Constants.DATA_CHOOSER_CLASS );
            return StringUtils.isEmpty( dataChooserClass );
        }
        catch ( DataChooserSettingDoesNotExist dataChooserSettingDoesNotExist )
        {
            return false;
        }
    }

}
