package com.plusmpm.parser;

import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;
import javax.xml.parsers.FactoryConfigurationError;
import javax.xml.parsers.ParserConfigurationException;

import org.apache.log4j.Logger;
import org.w3c.dom.Document;
import org.w3c.dom.Node;
import org.xml.sax.SAXException;

import com.plusmpm.parser.printer.FilePrinter;
import com.plusmpm.parser.xpdlelements.Package;

/**
 * Klasa umożliwiająca tłumaczenie dokumentów XPDL w systemie PlusWorkflow Z założenia tłumaczone są wszystkie elementy,
 * używane w systemie. Z bliższą specyfikacją elementów przetłumaczonych można sie zapoznać przeglądając dokumentacje
 * klas zawartych w pakiecie "com.plusmpm.parser.xpdlelements"
 * 
 * @author Michał Bocian 25-08-2011
 */
public class XpdlParser
{
    /**
     * Logger log4j
     */
    private static Logger log = Logger.getLogger( XpdlParser.class );

    /**
     * Stała ustawiająca lokalizacje plików. Domyślnie jest to tylko "pl" Zostanie wygenerowany zatem plik
     * "nazwa_pl.properties".
     */
    public static final String[] LANGUAGE_CONSTANTS = { "pl" };

    /**
     * Funkcja odpowiadająca za uruchomienie procesu parsowania dokumentu.
     * 
     * @param xpdl Zawartość pliku xml
     * @author Michał Bocian 25-08-2011
     */
    public static Package parse( byte[] xpdl )
        throws Exception
    {

        DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
        factory.setNamespaceAware( true );

        DocumentBuilder builder = factory.newDocumentBuilder();

        ByteArrayInputStream inputStream = new ByteArrayInputStream( xpdl );
        Document document = builder.parse( inputStream );

        Node packNode = document.getChildNodes().item( 0 );
        return new Package( packNode );
    }

    /**
     * Funkcja odpowiadająca za uruchomienie procesu parsowania dokumentu. Uruchamiana ona funkcje
     * {@link XpdlParser#parse(String, String)} jednak jako parametr okreslający ścieżkę wyjściową podaje null, przez co
     * dokumenty zostaną wygenerowane w sposób domyślny, w katalogu w którym znajduje się jar. Pliki będa sie wówczas
     * nazywać xpdl.properties, xpdl_pl.properties, xpdl.utf8.properties
     * 
     * @param urlToXpdl ścieżka do pliku XPDL, dla którego mają zostać wygenerowane pliki .properties
     * @author Michał Bocian 25-08-2011
     */
    public static void parse( String[] urlToXpdl )
    {
        parse( urlToXpdl, null );
    }

    /**
     * Funkcja uruchamiająca proces parsowania dokumentow
     * 
     * @param urlToXpdl ścieżki do plików XPDL, dla których mają zostać wygenerowane pliki .properties
     * @param urlToProperties ścieżka do pliku, która jako ostatni argument zawiera nazwę podstawową pliku. Np
     *            /home/enhydra/xpdl. Dla takiej nazwy zostaną wygenerowane pliki: xpdl.properties, xpdl_pl.properties,
     *            xpdl.utf8.properties
     * @author Michał Bocian 25-08-2011
     */
    public static void parse( String[] urlToXpdl, String urlToProperties )
    {
        log.info( "Rozpoczynam parsowanie" );
        try
        {
            DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            factory.setNamespaceAware( true );

            DocumentBuilder builder = factory.newDocumentBuilder();

            Map<String, String> resultMap = new HashMap<String, String>();

            for ( String pathToXpdl : urlToXpdl )
            {
                Document document = builder.parse( pathToXpdl );

                Node packNode = document.getChildNodes().item( 0 );

                if ( !"Package".equals( packNode.getLocalName() ) )
                {
                    log.warn( "Namespace pakietu: '" + packNode.getNodeName() + "' jest inny niż 'Package'" );
                }

                Package pack = new Package( packNode );

                resultMap.putAll( pack.getProperties() );
            }

            FilePrinter filePrinter = new FilePrinter( new File( urlToProperties + "/xpdlMessages.i18n" ) );
            filePrinter.printProperties( resultMap );
        }
        catch ( FactoryConfigurationError e )
        {
            log.error( "Nie można stworzyć fabryki konfiguracji", e );
        }
        catch ( ParserConfigurationException e )
        {
            log.error( "XPDL zawiera błędy", e );
        }
        catch ( SAXException e )
        {
            log.error( "Błąd parsowania", e );
        }
        catch ( IOException e )
        {
            log.error( "Błąd wejścia/wyjścia", e );
        }
        catch ( Exception e )
        {
            log.error( "Nieznany błąd", e );
        }
        finally
        {
            log.info( "Zakonczono parsowanie" );
        }
    }

    /**
     * <p>
     * Metoda umożliwiająca odpalenie parsera z konsoli Paczkę można uruchomić podając ścieżkę z parametrem "-i" co
     * oznacza ścieżki do plików XPDL oddzielonych przecinkami i opcjonalnie z parametrem "-o" przez co można określić
     * ścieżkę dla plików wyjściowych.
     * </p>
     * Przykładowe wywołanie
     * <code>java -jar /home/enhydra/XpdlParser.jar -i shark.xpdl,shark2.xpdl -o /home/enhydra/xpdl</code>
     * 
     * @param args
     * @author Michał Bocian 25-08-2011
     */
    public static void main( String[] args )
    {
        String[] pathsToXpdl = null;
        String pathToProperties = null;

        if ( args.length == 0 )
        {
            log.error( "Nie podano ścieżki do XPDL" );
            log.error( "Możliwe parametry to:" );
            log.error( "\t -i <sciezka> ( obowiazkowe )  bezwzględna ścieżka do pliku XPDL" );
            log.error( "\t -o <sciezka> ( opcjonalne ) ścieżka do wygenerowanego pliku .properties" );

            return;
        }

        if ( args[0].equals( "-i" ) )
        {
            String argXpdlPaths = args[1];
            pathsToXpdl = argXpdlPaths.split( "," );
            for ( String pathToXpdl : pathsToXpdl )
            {
                File file = new File( pathToXpdl );
                if ( !file.exists() )
                {
                    log.error( "Plik: ".concat( pathToXpdl ).concat( " nie istnieje." ) );
                    return;
                }
            }
        }

        if ( args.length == 4 && args[2].equals( "-o" ) )
        {
            pathToProperties = args[3];
        }

        parse( pathsToXpdl, pathToProperties );

    }

}
