
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: Logger.java,v 1.1 2005/07/13 11:09:06 slobodan Exp $
 */
package com.lutris.logging;

import com.lutris.util.Config;
import com.lutris.util.ConfigException;

/**
 * A general-purpose logging facility.  It is modeled after
 * <CODE>syslogd</CODE>.  This is a base class from which an actual
 * implementation is derived.  It only defines how log message are written
 * by a client.  Where the log message is written and the mechanism for
 * controlling logging is left up to the implementation.  This class does
 * not define any of these mechanism and their definition is not necessary
 * for understand how to use this class. <P>
 *
 * Each log message is associate with a facility and has a level assigned
 * to it.  A facility is a symbolic (String) name that defines a class of
 * log messages.  A level is used to indicate the 
 *
 It is expected that the implementation can enable, disable and
 * direct log messages based on these attributes.  Facilities and levels
 * are defined symbolicly, with no restriction on the name, and form a tuple.
 * Several standard levels are defined as integer constants and their use
 * is expected to be higher performing than symbolic levels..<P>
 *
 *
 *Normally, a single,
 global instance of the object
 * is exists and is obtainable by a static method in this class.<P>
 *
 * Log messages are written via an object implementing <CODE>LogChannel</CODE>.
 * A channel is associated with a single facility, with the level being
 * specified when a message is written.  Normally, a <CODE>LogChannel</CODE>
 * is obtained once at initialization time and use repeatedly.  It is
 * permissible to obtain multiple references to the log channel for a facility,
 * but this is discouraged for performance reasons.<P>
 *
 * Log messages, even debugging ones, should be defined with care.  They
 * should be terse, but clear to someone who isn't intimately familiar with
 * the code.  Permanent debugging messages should be designed with the idea
 * of use when supportting a deployed product.<P>
 *
 * The central logging object needs to be configured very early in the startup
 * process.  If logging can't be configured, then the startup should be aborted
 * or a object created that does some simple form of logging, such as write
 * to <CODE>stderr<CODE>.  A client should never have to check if the global
 * logger object exists.<P>
 * 
 * @author Mark Diekhans
 * @see com.lutris.logging.LogChannel
 */
public abstract class Logger {

    /**
     * Standard level for urgent condition that requires immediate attention
     * and indicates that the system is no longer functioning.
     */
    public static final int EMERGENCY = 0;

    /**
     * A condition that should be corrected immediately
     */
    public static final int ALERT = 1;

    /**
     * Critical conditions.
     */
    public static final int CRITICAL = 2;

    /**
     * Errors that have been correctly handled.
     */
    public static final int ERROR = 3;

    /**
     * Warning messages.
     */
    public static final int WARNING = 4;

    /**
     * Conditions that are not error conditions, but should possi bly be
     * handled specially.
     */
    public static final int NOTICE = 5;

    /**
     * Informational messages.
     */
    public static final int INFO = 6;

    /**
     * Messages that contain information normally of use only when debugging.
     * This is the basic level of debugging.  Levels DEBUG1 through DEBUG9
     * are defined to allow for more detailed messages.
     */
    public static final int DEBUG = 7;

    /**
     * Debug detail level 1.
     */
    public static final int DEBUG1 = 8;

    /**
     * Debug detail level 2.
     */
    public static final int DEBUG2 = 9;

    /**
     * Debug detail level 3.
     */
    public static final int DEBUG3 = 10;

    /**
     * Debug detail level 4.
     */
    public static final int DEBUG4 = 11;

    /**
     * Debug detail level 5.
     */
    public static final int DEBUG5 = 12;

    /**
     * Debug detail level 6.
     */
    public static final int DEBUG6 = 13;

    /**
     * Debug detail level 7.
     */
    public static final int DEBUG7 = 14;

    /**
     * Debug detail level 8.
     */
    public static final int DEBUG8 = 15;

    /**
     * Debug detail level 9.
     */
    public static final int DEBUG9 = 16;

    /**
     * Temporary debugging; should not be left in shipping code.
     */
    public static final int DEBUGTMP = 17;

    /**
     * Largest fixed logging level.
     */
    public static final int MAX_STD_LEVEL = DEBUGTMP;

    /**
     * Global <CODE>Logger</CODE> object.
     */
    protected static Logger centralLogger;

    /**
     * Table of standard level names
     */
    protected static final String [] standardLevelNames = {
        "EMERGENCY", // 0
 "ALERT",     // 1
 "CRITICAL",  // 2
 "ERROR",     // 3
 "WARNING",   // 4
 "NOTICE",    // 5
 "INFO",      // 6
 "DEBUG",     // 7
 "DEBUG1",    // 8
 "DEBUG2",    // 9
 "DEBUG3",    // 10
 "DEBUG4",    // 11
 "DEBUG5",    // 12
 "DEBUG6",    // 13
 "DEBUG7",    // 14
 "DEBUG8",    // 15
 "DEBUG9",    // 16
 "DEBUGTMP" // 17
    };

    /**
     * Get the central (global) logging object.
     *
     * @return A reference the object.  If the facility has not been
     *  initialized <CODE>null</CODE> is returned.  However, this is
     *  considered a bug in the design of the initialization. Clients
     *  do not need to check for <CODE>null</CODE>.
     */
    public static Logger getCentralLogger() {
        return centralLogger;
    }

    /**
     * Get the log channel object for a facility.  For a given facility,
     * the same object is always returned.
     * 
     * @param facility Facility the channel is associated with.
     */
    abstract public LogChannel getChannel(String  facility);

    /**
     * Configure Logger with given config file, interpreting of config file is 
     * logger implementation specific.
     *
     * @param confFilePath Path to configuration file.
     */
    abstract public void configure(String  confFilePath) throws ConfigException;

    /**
     * Configure Logger with given config section
     *
     * @param logConfig containing parameters for configuring logger
     */
    abstract public void configure(Config logConfig) throws ConfigException;
}
