
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: Log4jLogChannel.java,v 1.4 2005/03/24 10:51:20 slobodan Exp $
 */
package com.lutris.logging;

import java.text.SimpleDateFormat ;

import org.apache.log4j.Level;
import org.apache.log4j.Logger;

/**
 * Standard implementation of a channel associated with a logging
 * facility.  All messages for the facility are written using a channel.
 * Care is take to avoid synchronization when possible for performance
 * reasons.
 *
 * @author Vladimir Puskas
 * @author Predrag Djoic
 * @author Sinisa Milosevic
 * @see com.lutris.logging.LogChannel
 * @see com.lutris.logging.Log4jLogger
 */
public class Log4jLogChannel implements LogChannel {
    
    private String  facility;
    
    private static final int INT_ERROR = -1;    

    /**
     * Format for the date.
     */
    private static final SimpleDateFormat  dateFormatter = new SimpleDateFormat ("yyyy.MM.dd HH:mm:ss");

    /**
     * Logger object with which we are associated.
     */
    private org.apache.log4j.Logger logger;

    /**
     * Construct a new log channel.
     *
     * @param chanFacility The facility that the channel is associate with.
     * @param loggerObj The logging object that this channel will be associated
     *  with.
     */
    protected Log4jLogChannel(String  chanFacility,
            org.apache.log4j.Logger loggerObj) {
        facility = chanFacility;
        logger = loggerObj;
 
    }

    public int getLevel(String  level) {
        if (logger != null) {
            if (logger.getLevel() != null) {
                return  logger.getLevel().toInt();
            }
        }
        return INT_ERROR;
    }

    /**
     * Determine if logging is enabled for the specified level.
     */
    public boolean isEnabled(int level) {
        return true;
//         return logger.isEnabledFor(intToLevel(level));
    }

    /**
     * Determine if logging is enabled for the specified level.
     */
    public boolean isEnabled(String  level) {
        return true;
//      return logger.isEnabledFor(stringToLevel(level));
    }

    /**
     * Write a string to the log file.
     */
    public void write(int level, String  msg) {
        Level lev;

        if (logger == null) {
            logger = Logger.getLogger(LogAdapter.class.getName());
        }
        lev = intToLevel(level);
        if (lev == null) {
            lev = Level.INFO;
        }
       
        logger.log(lev, msg + Log4jLogger.separatorLine);
    }

    /**
     * Write a string and exception to the log file.
     */
    public void write(int level, String  msg, Throwable  throwable) {
        Level lev;

        if (logger == null) {
            logger = Logger.getLogger(LogAdapter.class.getName());
        }
        lev = intToLevel(level);
        if (lev == null) {
            lev = Level.INFO;
        }
        logger.log(lev, msg + Log4jLogger.separatorLine, throwable);
    }

    /**
     * Write a string to the log file.
     */
    public void write(String  level, String  msg) {
        Level lev;

        if (logger == null) {
            logger = Logger.getLogger(LogAdapter.class.getName());
        }
        lev = stringToLevel(level);
        if (lev == null) {
            lev = Level.INFO;
        }
        logger.log(lev, msg + Log4jLogger.separatorLine);
    }

    /**
     * Write a string and exception to the log file.
     */
    public void write(String  level, String  msg, Throwable  throwable) {
        Level lev;

        if (logger == null) {
            logger = Logger.getLogger(LogAdapter.class.getName());
        }
        lev = stringToLevel(level);
        if (lev == null) {
            lev = Level.INFO;
        }
        logger.log(lev, msg + Log4jLogger.separatorLine, throwable);
    }

    private Level intToLevel(int level) {
        Level lev;

        switch (level) {
        case com.lutris.logging.Logger.EMERGENCY: 
        case com.lutris.logging.Logger.ALERT:
            lev = Level.FATAL;
            break;

        case com.lutris.logging.Logger.CRITICAL: 
        case com.lutris.logging.Logger.ERROR:
            lev = Level.ERROR;
            break;

        case com.lutris.logging.Logger.WARNING:
            lev = Level.WARN;
            break;

        case com.lutris.logging.Logger.NOTICE: 
        case com.lutris.logging.Logger.INFO:
            lev = Level.INFO;
            break;

        case com.lutris.logging.Logger.DEBUG: 
        case com.lutris.logging.Logger.DEBUG1: 
        case com.lutris.logging.Logger.DEBUG2: 
        case com.lutris.logging.Logger.DEBUG3: 
        case com.lutris.logging.Logger.DEBUG4: 
        case com.lutris.logging.Logger.DEBUG5: 
        case com.lutris.logging.Logger.DEBUG6: 
        case com.lutris.logging.Logger.DEBUG7: 
        case com.lutris.logging.Logger.DEBUG8: 
        case com.lutris.logging.Logger.DEBUG9: 
        case com.lutris.logging.Logger.DEBUGTMP:
            lev = Level.DEBUG;
            break;

        default:
            lev = Level.DEBUG;
        } // end switch (level)
 return lev;
    }

    private Level stringToLevel(String  level) {
        Level lev;

        if (level.equalsIgnoreCase("EMERGENCY")
                || level.equalsIgnoreCase("ALERT")) {
            lev = Level.FATAL;
        } else if (level.equalsIgnoreCase("CRITICAL")
                || level.equalsIgnoreCase("ERROR")) {
            lev = Level.ERROR;
        } else if (level.equalsIgnoreCase("WARNING")) {
            lev = Level.WARN;
        } else if (level.equalsIgnoreCase("NOTICE")
                || level.equalsIgnoreCase("INFO")) {
            lev = Level.INFO;
        } else if (level.equalsIgnoreCase("DEBUG")
                || level.equalsIgnoreCase("DEBUG1")
                || level.equalsIgnoreCase("DEBUG2")
                || level.equalsIgnoreCase("DEBUG3")
                || level.equalsIgnoreCase("DEBUG4")
                || level.equalsIgnoreCase("DEBUG5")
                || level.equalsIgnoreCase("DEBUG6")
                || level.equalsIgnoreCase("DEBUG7")
                || level.equalsIgnoreCase("DEBUG8")
                || level.equalsIgnoreCase("DEBUG9")
                || level.equalsIgnoreCase("DEBUGTMP")) {
            lev = Level.DEBUG;
        } else {// default
            lev = Level.DEBUG;
        }
        return lev;
    }
    
    /**
     * <b>NOT SUPPORTED</b>
     *
     * @param level Symbolic level that is to be checked.
     * @return A log writer object.
     */
    public LogWriter getLogWriter(String  level) {
//  throw new UnsupportedOperationException("Log4jLogChannel doesn't support this operation.");
 return new Log4jLogWriter(this, level);

    }

    /**
     * <b>NOT SUPPORTED</b>
     *
     * @param level Numeric level that is to be checked.
     * @return A log writer object.
     */
    public LogWriter getLogWriter(int level) {
//  throw new UnsupportedOperationException("Log4jLogChannel doesn't support this operation.");
 return new Log4jLogWriter(this, level);
    }
}
