
/*
 * @(#)NodeImpl.java    1.36 02/03/21
 *
 * Copyright 2002 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package org.enhydra.xml;

import org.w3c.dom.DOMException ;
import org.w3c.dom.Document ;
import org.w3c.dom.NamedNodeMap ;
import org.w3c.dom.Node ;
import org.w3c.dom.NodeList ;
import org.w3c.dom.UserDataHandler ;

/**
 * @author Tweety
 *
 * A class representing a node in a meta-data tree, which implements
 * the <a HREF="../../../../api/org/w3c/dom/Node.html">
 *
 * <p> Namespaces are ignored in this implementation.  The terms "tag
 * name" and "node name" are always considered to be synonymous.
 *
 * @version 1.0
 */
public class NodeImpl implements Node , NodeList  {

    /**
     * Owner document.
     */
    protected Document  ownerDocument;


    /**
     * The name (tag) of the node as a <code>String</code>.
     */
    protected String  nodeName = null;


    /**
     * The value of the node as a <code>String</code>.
     */
    protected String  nodeValue = null;


    /**
     * The type of the node as a <code>short</code>.
     */
    protected short type = ELEMENT_NODE;


    /**
     * The parent node of this node, or <code>null</code> if this node
     * forms the root of its own tree.
     */
    protected NodeImpl parent = null;


    /**
     * The number of child nodes.
     */
    protected int numChildren = 0;


    /**
     * The first (leftmost) child node of this node, or
     * <code>null</code> if this node is a leaf node.
     */
    protected NodeImpl firstChild = null;


    /**
     * The last (rightmost) child node of this node, or
     * <code>null</code> if this node is a leaf node.
     */
    protected NodeImpl lastChild = null;


    /**
     * The next (right) sibling node of this node, or
     * <code>null</code> if this node is its parent's last child node.
     */
    protected NodeImpl nextSibling = null;


    /**
     * The previous (left) sibling node of this node, or
     * <code>null</code> if this node is its parent's first child node.
     */
    protected NodeImpl previousSibling = null;


    /**
     * Constructs an empty <code>NodeImpl</code>.
     */
    public NodeImpl() {
    }


    /**
     * Constructs a <code>NodeImpl</code> from the given node,
     * without creating entire children subtree.
     *
     * @param node, as a <code>NodeImpl</code>.
     */
    public NodeImpl(NodeImpl node) {
        ownerDocument = node.ownerDocument;
        nodeName = node.nodeName;
        nodeValue = node.nodeValue;
        type = node.type;
        parent = node.parent;
        numChildren = node.numChildren;
        firstChild = node.firstChild;
        lastChild = node.lastChild;
        nextSibling = node.nextSibling;
        previousSibling = node.previousSibling;
    }


    /**
     * Constructs an <code>NodeImpl</code> from a given node (creates the children subtree too),
     * as a <code>Node</code>
     *
     * @param node, as a <code>Node</code>.
     */
    public NodeImpl(Node  node) {
        this(node, true);
    }


    /**
     * Constructs an <code>NodeImpl</code> from a given node, as a <code>Node</code>,
     * and deep as <code>boolean</code>.
     *
     * @param node, as a <code>Node</code>.
     * @param deep if <code>true</code>, recursively clone the subtree
     * under the specified node; if <code>false</code>, clone only the
     * node itself.
     */
    public NodeImpl(Node  node, boolean deep) {
        this.ownerDocument = node.getOwnerDocument();
        this.nodeName = node.getNodeName();
        this.type = node.getNodeType();
        this.nodeValue = node.getNodeValue();
        if (deep)
            this.initNodeImplChildren(node);
    }


    /**
     * Constructs a <code>NodeImpl</code> from the given document owner and node name.
     *
     * @param ownerDoc the document owner of the node, as a <code>Document</code>.
     * @param name the name of the node, as a <code>String</code>.
     */
    public NodeImpl(Document  ownerDoc, String  name) {
        this.ownerDocument = ownerDoc;
        this.nodeName = nodeName;
    }


    /**
     * Constructs an <code>NodeImpl</code> from a given document owner,
     * node name and node type.
     *
     * @param ownerDoc the document owner of the node, as a <code>Document</code>.
     * @param nodeName the name of the node, as a <code>String</code>.
     * @param type the type of the node, as a <code>short</code>.
     */
    public NodeImpl(Document  ownerDoc, String  nodeName, short type) {
        this.ownerDocument = ownerDoc;
        this.nodeName = nodeName;
        this.type = type;
    }


    /**
     * Constructs an <code>NodeImpl</code> from a given document owner,
     * node name, node type and node value.
     *
     * @param ownerDoc the document owner of the node, as a <code>Document</code>.
     * @param nodeName the name of the node, as a <code>String</code>.
     * @param type the type of the node, as a <code>short</code>.
     * @param value the value of the node, as a <code>String</code>.
     */
    public NodeImpl(Document  ownerDoc, String  nodeName, short type, String  value) {
        this.ownerDocument = ownerDoc;
        this.nodeName = nodeName;
        this.type = type;
        this.nodeValue = value;
    }



    /**
     * Creates the children subtree and adds to this node.
     * (this part had to be splited from the constructor)
     *
     * @param nodeas a <code>Node</code>.
     */
    protected void initNodeImplChildren(Node  node) {
        // add children
 Node  child = node.getFirstChild();
        while (child != null) {
            switch (child.getNodeType()) {
                case Node.ELEMENT_NODE : {
                    this.appendChild(newElementInstance(child));
                }; break;
                case Node.TEXT_NODE : {
                    this.appendChild(newTextInstance(child));
                }; break;
                case Node.COMMENT_NODE : {
                    this.appendChild(newCommentInstance(child));
                }; break;
                default : {
                    this.appendChild(newDefaultInstance(child));
                };
            }
            child = child.getNextSibling();
        }
    }


    /**
     * Creates new instance of the ElementImpl class.
     *
     * @param node, as a <code>Node</code>.
     * @return Node new instance of the ElementImpl class.
     */
    protected Node  newElementInstance(Node  node) {
        return new ElementImpl(node);
    }

    /**
     * Creates new instance of the TextImpl class.
     *
     * @param node, as a <code>Node</code>.
     * @return Node new instance of the TextImpl class.
     */
    protected Node  newTextInstance(Node  node) {
        return new TextImpl(node);
    }

    /**
     * Creates new instance of the CommentImpl class.
     *
     * @param node, as a <code>Node</code>.
     * @return Node new instance of the CommentImpl class.
     */
    protected Node  newCommentInstance(Node  node) {
        return new CommentImpl(node);
    }

    /**
     * Creates new instance of the NodeImpl class.
     *
     * @param node, as a <code>Node</code>.
     * @return Node new instance of the NodeImpl class.
     */
    protected Node  newDefaultInstance(Node  node) {
        return new NodeImpl(node);
    }


    /**
     * Check that the node is either <code>null</code> or an
     * <code>NodeImpl</code>.
     *
     * @exception DOMException if node is not an instance of <code>NodeImpl</code>.
     */
    private void checkNode(Node  node) throws DOMException  {
        if (node == null) {
            return;
        }
        if (!(node instanceof NodeImpl))
            throw new NodeDOMException(DOMException.WRONG_DOCUMENT_ERR, "Node not an NodeImpl!");
    }



    // Methods from Node

    /**
     * Returns the name associated with this node.
     *
     * @return the name, as a <code>String</code>.
     */
    public String  getNodeName() {
        return nodeName;
    }


    /**
     * Returns the value associated with this node.
     *
     * @return the node value, as a <code>String</code>.
     */
    public String  getNodeValue() {
        return nodeValue;
    }


    /**
     * Sets the node value of this node.
     *
     * @param nodeValue new node value, as a <code>String</code>.
     */
    public void setNodeValue(String  nodeValue) {
        this.nodeValue = nodeValue;
    }


    /**
     * Returns the node type.
     *
     * @return the <code>short</code> value node type.
     */
    public short getNodeType() {
        return type;
    }


    /**
     * Returns the parent of this node. A <code>null</code> value
     * indicates that the node is the root of its own tree.  To add a
     * node to an existing tree, use one of the
     * <code>insertBefore</code>, <code>replaceChild</code>, or
     * <code>appendChild</code> methods.
     *
     * @return the parent, as a <code>Node</code>.
     *
     * @see #insertBefore
     * @see #replaceChild
     * @see #appendChild
     */
    public Node  getParentNode() {
        return parent;
    }


    /**
     * Returns all child nodes of this node, or <code>null</code> if
     * the node has no children.
     *
     * @return all child nodes of this node, as a <code>Node</code>, or
     * <code>null</code>.
     */
    public NodeList  getChildNodes() {
        return this;
    }


    /**
     * Returns the first child of this node, or <code>null</code> if
     * the node has no children.
     *
     * @return the first child, as a <code>Node</code>, or
     * <code>null</code>
     */
    public Node  getFirstChild() {
        return firstChild;
    }


    /**
     * Returns the last child of this node, or <code>null</code> if
     * the node has no children.
     *
     * @return the last child, as a <code>Node</code>, or
     * <code>null</code>.
     */
    public Node  getLastChild() {
        return lastChild;
    }


    /**
     * Returns the previous sibling of this node, or <code>null</code>
     * if this node has no previous sibling.
     *
     * @return the previous sibling, as a <code>Node</code>, or
     * <code>null</code>.
     */
    public Node  getPreviousSibling() {
        return previousSibling;
    }


    /**
     * Returns the next sibling of this node, or <code>null</code> if
     * the node has no next sibling.
     *
     * @return the next sibling, as a <code>Node</code>, or
     * <code>null</code>.
     */
    public Node  getNextSibling() {
        return nextSibling;
    }


    /**
     * Returns <code>null</code>, since <code>NodeImpl</code>s
     * do not belong to any <code>Document</code>.
     *
     * @return document owner as <code>Document</code>.
     */
    public Document  getOwnerDocument() {
        return ownerDocument;
    }


    /**
     * Inserts the node <code>newChild</code> before the existing
     * child node <code>refChild</code>. If <code>refChild</code> is
     * <code>null</code>, insert <code>newChild</code> at the end of
     * the list of children.
     *
     * @param newChild the <code>Node</code> to insert.
     * @param refChild the reference <code>Node</code>.
     *
     * @return the node being inserted.
     *
     * @exception IllegalArgumentException if <code>newChild</code> is
     * <code>null</code>.
     */
    public Node  insertBefore(Node  newChild, Node  refChild) {
        if (newChild == null) {
            throw new IllegalArgumentException ("newChild == null!");
        }

        checkNode(newChild);
        checkNode(refChild);

        NodeImpl newChildNode = (NodeImpl) newChild;
        NodeImpl refChildNode = (NodeImpl) refChild;

        // Siblings, can be null.
 NodeImpl previous = null;
        NodeImpl next = null;

        if (refChild == null) {
            previous = this.lastChild;
            next = null;
            this.lastChild = newChildNode;
        } else {
            previous = refChildNode.previousSibling;
            next = refChildNode;
        }

        if (previous != null) {
            previous.nextSibling = newChildNode;
        }
        if (next != null) {
            next.previousSibling = newChildNode;
        }

        newChildNode.parent = this;
        newChildNode.previousSibling = previous;
        newChildNode.nextSibling = next;

        // N.B.: O.K. if refChild == null
 if (this.firstChild == refChildNode) {
            this.firstChild = newChildNode;
        }
        ++numChildren;
        return newChildNode;
    }


    /**
     * Replaces the child node <code>oldChild</code> with
     * <code>newChild</code> in the list of children, and returns the
     * <code>oldChild</code> node.
     *
     * @param newChild the <code>Node</code> to insert.
     * @param oldChild the <code>Node</code> to be replaced.
     *
     * @return the node replaced.
     *
     * @exception IllegalArgumentException if <code>newChild</code> is
     * <code>null</code>.
     */
    public Node  replaceChild(Node  newChild, Node  oldChild) {
        if (newChild == null) {
            throw new IllegalArgumentException ("newChild == null!");
        }

        checkNode(newChild);
        checkNode(oldChild);

        NodeImpl newChildNode = (NodeImpl) newChild;
        NodeImpl oldChildNode = (NodeImpl) oldChild;

        NodeImpl previous = oldChildNode.previousSibling;
        NodeImpl next = oldChildNode.nextSibling;

        if (previous != null) {
            previous.nextSibling = newChildNode;
        }
        if (next != null) {
            next.previousSibling = newChildNode;
        }

        newChildNode.parent = this;
        newChildNode.previousSibling = previous;
        newChildNode.nextSibling = next;

        if (firstChild == oldChildNode) {
            firstChild = newChildNode;
        }
        if (lastChild == oldChildNode) {
            lastChild = newChildNode;
        }

        oldChildNode.parent = null;
        oldChildNode.previousSibling = null;
        oldChildNode.nextSibling = null;

        return oldChildNode;
    }


    /**
     * Removes the child node indicated by <code>oldChild</code> from
     * the list of children, and returns it.
     *
     * @param oldChild the <code>Node</code> to be removed.
     *
     * @return the node removed.
     *
     * @exception IllegalArgumentException if <code>oldChild</code> is
     * <code>null</code>.
     */
    public Node  removeChild(Node  oldChild) {
        if (oldChild == null) {
            throw new IllegalArgumentException ("oldChild == null!");
        }
        checkNode(oldChild);

        NodeImpl oldChildNode = (NodeImpl) oldChild;

        NodeImpl previous = oldChildNode.previousSibling;
        NodeImpl next = oldChildNode.nextSibling;

        if (previous != null) {
            previous.nextSibling = next;
        }
        if (next != null) {
            next.previousSibling = previous;
        }

        if (this.firstChild == oldChildNode) {
            this.firstChild = next;
        }
        if (this.lastChild == oldChildNode) {
            this.lastChild = previous;
        }

        oldChildNode.parent = null;
        oldChildNode.previousSibling = null;
        oldChildNode.nextSibling = null;

        --numChildren;
        return oldChildNode;
    }


    /**
     * Adds the node <code>newChild</code> to the end of the list of
     * children of this node.
     *
     * @param newChild the <code>Node</code> to insert.
     *
     * @return the node added.
     *
     * @exception IllegalArgumentException if <code>newChild</code> is
     * <code>null</code>.
     */
    public Node  appendChild(Node  newChild) {
        if (newChild == null) {
            throw new IllegalArgumentException ("newChild == null!");
        }
        checkNode(newChild);

        // insertBefore will increment numChildren
 return insertBefore(newChild, null);
    }


    /**
     * Returns <code>true</code> if this node has child nodes.
     *
     * @return <code>true</code> if this node has children.
     */
    public boolean hasChildNodes() {
        return numChildren > 0;
    }


    /**
     * Returns a duplicate of this node.  The duplicate node has no
     * parent (<code>getParentNode</code> returns <code>null</code>).
     * If a shallow clone is being performed (<code>deep</code> is
     * <code>false</code>), the new node will not have any children or
     * siblings.  If a deep clone is being performed, the new node
     * will form the root of a complete cloned subtree.
     *
     * @param deep if <code>true</code>, recursively clone the subtree
     * under the specified node; if <code>false</code>, clone only the
     * node itself.
     *
     * @return the duplicate node.
     */
    public Node  cloneNode(boolean deep) {
        return new NodeImpl(this, deep);
    }


    /**
     * Does nothing, since <code>NodeImpl</code>s do not
     * contain <code>Text</code> children.
     */
    public void normalize() {
    }


    /**
     * Returns <code>false</code> since DOM features are not
     * supported.
     *
     * @return <code>false</code>.
     *
     * @param feature a <code>String</code>, which is ignored.
     * @param version a <code>String</code>, which is ignored.
     */
    public boolean isSupported(String  feature, String  version) {
        return false;
    }


    /**
     * Returns <code>null</code>, since namespaces are not supported.
     *
     * @return <code>null</code>.
     *
     * @see #setPrefix
     */
    public String  getPrefix() {
        return null;
    }


    /**
     * Does nothing, since namespaces are not supported.
     *
     * @param prefix a <code>String</code>, which is ignored.
     *
     * @see #getPrefix
     */
    public void setPrefix(String  prefix) {
    }


    /**
     * Equivalent to <code>getNodeName</code>.
     *
     * @return the node name, as a <code>String</code>.
     */
    public String  getLocalName() {
        return nodeName;
    }


    /**
     * Returns all attribute nodes of this node.
     *
     * @return all attribute nodes of this node.
     */
    public NamedNodeMap  getAttributes() {
        return null;
    }


    /**
     * Returns <code>true</code>, if this node has attributes, otherwise
     * <code>false</code>.
     *
     * @return <code>true</code> if node has attributes, otherwise <code>false</code>..
     */
    public boolean hasAttributes() {
        return false;
    }




    // Methods from NodeList


    /**
     * Returns number of child nodes.
     *
     * @return all number of child nodes.
     */
    public int getLength() {
        return numChildren;
    }


    /**
     * Returns child node with the given index.
     *
     * @return child node with the given index.
     */
    public Node  item(int index) {
        if (index < 0) {
            return null;
        }

        Node  child = getFirstChild();
        while (child != null && index-- > 0) {
            child = child.getNextSibling();
        }
        return child;
    }



    // String methodes

    /**
     * Returns <code>String</code> representation of this node.
     *
     * @return <code>String</code> representation of this node.
     */
    public String  toString() {
        return toString(Indent.DEFAULT_TAB);
    }


    /**
     * Returns <code>String</code> representation of this node.
     *
     * @param tab tab for node indentation.
     *
     * @return <code>String</code> representation of this node.
     */
    public String  toString(String  tab) {
        StringBuffer  sb = new StringBuffer ();
        this.allToString(sb, new Indent(0,tab));
        return sb.toString();
    }


    /**
     * Method beginToString should be redefined in extended classes.
     * Each type of node has its own <code>beginToString and
     * <code>endToString</code>. This was added to support
     * writing of the xml file. The <code>Element</code>
     * type of node: it writes the beginning tag, then calls
     * the child's <code>toString</code>, and then writes the ending tag.
     *
     * @param sb string buffer to add resulting string.
     * @param indent used in formating the output.
     */
    protected void beginToString(StringBuffer  sb, Indent indent) {
    }


    /**
     * Method endToString should be redefined in extended classes.
     * Each type of node has its own <code>beginToString and
     * <code>endToString</code>. This was added to support
     * writing of the xml file. The <code>Element</code>
     * type of node: it writes the beginning tag, then calls
     * the child's <code>toString</code>, and then writes the ending tag.
     *
     * @param sb string buffer to add resulting string.
     * @param indent used in formating the output.
     */
    protected void endToString(StringBuffer  sb, Indent indent) {
    }


    private void allToString(StringBuffer  sb, Indent indent) {
        this.beginToString(sb, indent);
        Node  child = getFirstChild();
        while (child != null) {
            ((NodeImpl) child).allToString(sb, indent);
            child = child.getNextSibling();
        }
        this.endToString(sb, indent);
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#getBaseURI()
     */
    public String  getBaseURI() {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#compareDocumentPosition(org.w3c.dom.Node)
     */
    public short compareDocumentPosition(Node  arg0) throws DOMException  {
        // TODO Auto-generated method stub
 return 0;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#getTextContent()
     */
    public String  getTextContent() throws DOMException  {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#setTextContent(java.lang.String)
     */
    public void setTextContent(String  arg0) throws DOMException  {
        // TODO Auto-generated method stub
 
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#isSameNode(org.w3c.dom.Node)
     */
    public boolean isSameNode(Node  arg0) {
        // TODO Auto-generated method stub
 return false;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#lookupPrefix(java.lang.String)
     */
    public String  lookupPrefix(String  arg0) {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#isDefaultNamespace(java.lang.String)
     */
    public boolean isDefaultNamespace(String  arg0) {
        // TODO Auto-generated method stub
 return false;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#lookupNamespaceURI(java.lang.String)
     */
    public String  lookupNamespaceURI(String  arg0) {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#isEqualNode(org.w3c.dom.Node)
     */
    public boolean isEqualNode(Node  arg0) {
        // TODO Auto-generated method stub
 return false;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#getFeature(java.lang.String, java.lang.String)
     */
    public Object  getFeature(String  arg0, String  arg1) {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#setUserData(java.lang.String, java.lang.Object, org.w3c.dom.UserDataHandler)
     */
    public Object  setUserData(String  arg0, Object  arg1, UserDataHandler  arg2) {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#getUserData(java.lang.String)
     */
    public Object  getUserData(String  arg0) {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#getNamespaceURI()
     */
    public String  getNamespaceURI() {
        // TODO Auto-generated method stub
 return null;
    }

}



