
package org.enhydra.xml;

import org.w3c.dom.CharacterData ;
import org.w3c.dom.DOMException ;
import org.w3c.dom.Node ;

/**
 * @author Tweety
 *
 * A class representing a node in a meta-data tree, which implements
 * the <a HREF="../../../../api/org/w3c/dom/CharacterData.html">
 *
 * @version 1.0
 */
public class CharacterDataImpl extends NodeImpl implements CharacterData  {
    
    
    /**
     * Constructs an empty <code>CharacterDataImpl</code>.
     */
    public CharacterDataImpl() {
    }
    
    /**
     * Constructs a <code>CharacterDataImpl</code> from the 
     * given node as <code>Node</code>.
     * 
     * @param node, as a <code>Node</code>.
     */
    public CharacterDataImpl(Node  node) {
        super(node);
    }


    /**
     * Returns node value.
     * 
     * @return node value, as <code>String</code>.
     * 
     * @see org.w3c.dom.CharacterData#getData().
     */
    public String  getData() throws DOMException  {
        return nodeValue;
    }

    /**
     * Sets the new value of this node.
     * 
     * @param data the new data
     * 
     * @see org.w3c.dom.CharacterData#setData(String).
     */
    public void setData(String  data) throws DOMException  {
        nodeValue = data;
    }


    /**
     * Returns the substring from the node's value.
     * 
     * @param offset the begin index of the substring.
     * @param count the number of characters.
     * 
     * @return substring of the node's value.
     * 
     * @see org.w3c.dom.CharacterData#substringData(int, int).
     */
    public String  substringData(int offset, int count) throws DOMException  {
        int length = nodeValue.length();
        if (count < 0 || offset < 0 || offset > length - 1)
            throw new DOMException (DOMException.INDEX_SIZE_ERR, "Index out of bounds");

        int tailIndex = length;
        if(offset + count < length)
            tailIndex = offset + count;
        return nodeValue.substring(offset, tailIndex);
    }


    /**
     * Appends data to the node's value.
     * 
     * @param arg the data to append to the node's value.
     * 
     * @see org.w3c.dom.CharacterData#appendData(String).
     */
    public void appendData(String  arg) {
        nodeValue += arg;
    }


    /**
     * Inserts substring into node's value string.
     * 
     * @param offset the begin index of the substring.
     * @param arg the <code>String</code> to insert.
     * 
     * @see org.w3c.dom.CharacterData#insertData(int, String).
     */
    public void insertData(int offset, String  arg) throws DOMException  {
        try {
            nodeValue = new StringBuffer (nodeValue).insert(offset, arg).toString();
        } catch (StringIndexOutOfBoundsException  e) {
            throw new DOMException (DOMException.INDEX_SIZE_ERR, "Index out of bounds");
        }
    }


    /**
     * Deletes characters from the node's value string.
     * 
     * @param offset the begin index of the substring.
     * @param count the number of characters.
     * 
     * @see org.w3c.dom.CharacterData#deleteData(int, int).
     */
    public void deleteData(int offset, int count) throws DOMException  {
        int tailLength = nodeValue.length() - count - offset;
        if(nodeValue.length() - count - offset < 0)
            tailLength = 0;
        try {
            nodeValue = nodeValue.substring(0, offset) +
                (tailLength > 0 ? nodeValue.substring(offset + count, offset + count + tailLength) : "");
        } catch (StringIndexOutOfBoundsException  e) {
            throw new DOMException (DOMException.INDEX_SIZE_ERR, "Index out of bounds");
        }
    }


    /**
     * Replaces characters in the node's value string.
     * 
     * @param offset the begin index of the substring.
     * @param count the number of characters.
     * @param arg the <code>String</code> to insert.
     * 
     * @see org.w3c.dom.CharacterData#replaceData(int, int, String).
     */
    public void replaceData(int offset, int count, String  arg) throws DOMException  {
        deleteData(offset, count);
        insertData(offset, arg);
    }


    /**
     * Returns the namespace of the node.
     * 
     * @return the namespace of the node.
     * 
     * @see org.w3c.dom.Node#getNamespaceURI().
     */
    public String  getNamespaceURI() {
        return super.getNamespaceURI();
    }
    

}
