

/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: LogWriter.java,v 1.1 2005/07/13 11:09:06 slobodan Exp $
 */

package com.lutris.logging;

import java.io.PrintWriter ;
import java.io.Writer ;

/**
 * Class use to write log output to a particular LogChannel and level.
 * This class is PrintWriter, with println() causing a write.
 * One should use println() with this rather than write, as with a
 * LogChannel, since write doesn't write a full line.
 */
public class LogWriter extends PrintWriter  {
    /*
     * Log channel and associated level.
     */
    private LogChannel channel;
    private int level;

    /*
     * Are we enabled.
     */
    private boolean enabled;

    /**
     * Constructor.
     * @param logChannel The log channel to write to.
     * @param logLevel The level associated with this channel.
     */
    protected LogWriter(LogChannel logChannel,
                        String  logLevel) {
        this(logChannel, logChannel.getLevel(logLevel));
    }

    /**
     * Constructor.
     * @param logChannel The log channel to write to.
     * @param logLevel The level associated with this channel.
     */
    protected LogWriter(LogChannel logChannel,
                        int logLevel) {
        // Flushing logic is in ChannelWriter, so don't buffer here
 super(new ChannelWriter(logChannel, logLevel), true);
        channel = logChannel;
        level = logLevel;
        enabled = logChannel.isEnabled(level);
    }

    /**
     * Get the associate channel.
     */
    public LogChannel getChannel() {
        return channel;
    }

    /**
     * Get the associate level.
     */
    public int getLevel() {
        return level;
    }

    /**
     * Determine if logging is enabled.  This is useful to prevent a series of
     * unnecessary logging calls, as often encountered with debug logging, or
     * a call where generating the message is expensive.
     *
     * @return <CODE>true</CODE> if enabled, <CODE>false</CODE> if not
     *         enabled.
     */
    public boolean isEnabled() {
        return enabled;
    }
    
    /**
     * Write a string and exception to the log file.
     *
     * @param msg The message to log.
     * @param throwable Exception or error to log.
     */
    public void println(String  msg, Throwable  throwable) {
        if (enabled) {
            flush();
            channel.write(level, msg, throwable);
        }
    }
}


/**
 * Internal Writer object that interfaces to the LogChannel.
 * Can't be inner class of LogWriter, it must be passed
 * to super constructor.
 */
class ChannelWriter extends Writer  {
    // Output collected here
 private StringBuffer  buffer = new StringBuffer ();

    // Log channel, etr.
 private LogChannel channel;
    private int level;
    private boolean enabled;

    /**
     * Construct a new writer.
     */
    public ChannelWriter(LogChannel logChannel,
                         int logLevel) {
        channel = logChannel;
        level = logLevel;
        enabled = logChannel.isEnabled(level);
    }

    /**
     * Main write method that transfers stuff to the buffer.
     */
    public void write(char[] cbuf, int off, int len) {
        buffer.append(cbuf, off, len);
    }

    /**
     * Flush the buffer to the log file.
     */
    public void flush() {
        if (enabled && (buffer.length() > 0)) {
            synchronized(lock) {
                channel.write(level, buffer.toString());
                buffer.setLength(0);
            }
        }
    }

    /**
     * Close is a no-op.
     */
    public void close() {
    }
}

