

/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: LocalDirResource.java,v 1.2 2005/03/24 10:51:25 slobodan Exp $
 */





package com.lutris.classloader;

// lutris packages
// v. strahinja, 24 sep 2002
import java.io.File ;
import java.io.FileInputStream ;
import java.io.FileNotFoundException ;
import java.io.IOException ;
import java.io.InputStream ;

import com.lutris.logging.LogChannel;
import com.lutris.util.FatalExceptionError;

/**
 * <P>A <CODE>Resource</CODE> that is a file on the local machine in
 * a specified directory.  The directory is represented by a
 * <CODE>ClassPathEntry</CODE>, and the filename is specified by a String.
 *
 * @author Kristen Pol, Lutris Technologies
 * @version $Revision : 1.1 $
 * @see com.lutris.classloader.ClassPathEntry
 * @see com.lutris.classloader.Resource
 * @see java.io.File
 */
public class LocalDirResource extends Resource {

    // private data members

    /** The file that represents this resource. */
    private File  file = null;

    // constructors

    /**
     * Constructs local directory resource with specified name and location.
     *
     * @param name The file name of the resource.
     * @param location The location of the resource.
     * @param loadLogChannel The log channel for logging.
     * @exception FileNotFoundException
     *  thrown if the desired file does not exist, does not have
     *  read permission or if the desired file exists above the
     *  relative root of the <code>LocalDirResource</code>.
     * @see Resource
     * @see ClassPathEntry
     */
    protected LocalDirResource(String          name,
                   ClassPathEntry location,
// v. strahinja, 24 sep 2002
 LogChannel     loadLogChannel)
//                   Logger loadLogger)
 throws FileNotFoundException  {
// v. strahinja, 24 sep 2002
 super(name, location, loadLogChannel);
//    super(name, location, loadLogger);
 String  locName = location.getName();
    if (locName == null) {
        throw new FileNotFoundException ("The name for location, "
                                            + location + ", is null");
    }
    file = new File (locName, name);
    if (!file.exists() || !file.canRead()) {
        File  tmpFile = file;
        file = null;
        throw new FileNotFoundException ("File, " +
                        tmpFile.getAbsolutePath() +
                                            ", does not exist or does not " +
                                            "have read permission");
    }

        String  path = null;
        String  parentPath = null;

        try {
            parentPath = new File (locName).getCanonicalPath();
        } catch (IOException  e) {
            file = null;
            throw new FileNotFoundException ("Classpath Directory " + locName +
                                            " cannot be resolved: " +
                        e.toString());
        }

        try {
            path = file.getCanonicalPath();
        } catch (IOException  e) {
            File  tmpFile = file;
            file = null;
            throw new FileNotFoundException ("File " +
                                            tmpFile.getAbsolutePath() +
                                            " cannot be resolved: " +
                        e.toString());
        }

        if (path.startsWith(parentPath) == false) {
            File  tmpFile = file;
            file = null;
            throw new FileNotFoundException ("File, " + tmpFile +
                                            " does not live under " + locName);
        }

    size = file.length();
    lastModifiedTime = file.lastModified();
    }

    // public methods

    /**
     * Gets input stream representing resource.
     *
     * @return the input stream that represents the resource.
     * @exception IOException if the input stream can not be constructed.
     * @see InputStream
     */
    public InputStream  getInputStream() throws IOException  {
    try {
        return new FileInputStream (file);
    } catch (FileNotFoundException  e) {
        throw new FatalExceptionError(e);
    }
    }

    /**
     * Get current last-modification time of resource.  This is the
     * time on the disk file the resource is associated with.
     *
     * @return the last-modified time of the permanent copy of the resource
     * in milliseconds.
     */
    public long getCurrentLastModifiedTime() throws FileNotFoundException  {
        return file.lastModified();
    }

    /**
     * Get the file associate with this resource.
     */
    public File  getFile() {
        return file;
    }
}
