

/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: ClassPath.java,v 1.3 2005/06/10 20:43:51 slobodan Exp $
 */

package com.lutris.classloader;

// lutris packages
import java.io.File ;
import java.net.URL ;
import java.util.Enumeration ;
import java.util.Vector ;

import com.lutris.logging.LogChannel;
import com.lutris.util.FatalExceptionError;

/**
 * <P>A class path that is composed of <CODE>ClassPathEntry</CODE> objects.
 * This class can be used in conjunction with a class loader to load
 * classes and resources.
 *
 * @author Kristen Pol, Lutris Technologies
 * @version $Revision : 1.0 $
 * @see com.lutris.classloader.ClassPathEntry
 * @see com.lutris.classloader.Resource
 */
class ClassPath {

    // private data members

    /** The class path Vector made up of ClassPathEntry objects. */
    private Vector  classPath = null;

    /** Is logging enabled? */
    private boolean loggingEnabled = false;

    /** Log channel to write messages to */
// v. strahinja, 23 sep 2002      
 private LogChannel logChannel;
//    private Logger logger;

    /** Numeric log level number for LOGLEVEL string */
// v. strahinja, 23 sep 2002
 private int logLevel;
//    private Level logLevel;

    // constructors

    /**
     * Constructs empty class path with no entries.
     *
     * @param loadLogChannel The log channel, maybe null.
     * @see #set
     * @see #add
     */
// v. strahinja, 23 sep 2002
 public ClassPath(LogChannel loadLogChannel) {
// v. strahinja, 23 sep 2002
 this((Vector )null, loadLogChannel);
//    public ClassPath(Logger loadLogger) {
//        this((Vector)null, loadLogger);
    }

    /**
     * Constructs class path with specified class path entries.  The
     * parameter is assumed to be an array of directories, URLs, and/or
     * zip files.
     *
     * @param entries The class path represented by a String array.
     * @param loadLogChannel The log channel, maybe null.
     */
// v. strahinja, 23 sep 2002
 public ClassPath(String [] entries, LogChannel loadLogChannel) {
// v. strahinja, 23 sep 2002
 this(convertArrayToVector(entries, loadLogChannel), loadLogChannel);
//    public ClassPath(String[] entries, Logger loadLogger) {
//        this(convertArrayToVector(entries, loadLogger), loadLogger);
    }

    /**
     * Constructs class path with specified class path entries.  The
     * parameter is assumed to be an array of zip files and/or directories.
     *
     * @param entries The class path represented by a File array.
     * @param loadLogChannel The log channel, maybe null.
     */
// v. strahinja, 23 sep 2002
 public ClassPath(File [] entries, LogChannel loadLogChannel) {
// v. strahinja, 23 sep 2002
 this(convertArrayToVector(entries, loadLogChannel), loadLogChannel);
//    public ClassPath(File[] entries, Logger loadLogger) {
//        this(convertArrayToVector(entries, loadLogger), loadLogger);
    }

    /**
     * Constructs class path with specified class path entries.  The URLs can
     * represent directories and/or zip files on the local machine and/or
     * on remote machines.
     *
     * @param entries The class path represented by a URL array.
     * @param loadLogChannel The log channel, maybe null.
     */
// v. strahinja, 23 sep 2002
 public ClassPath(URL [] entries, LogChannel loadLogChannel) {
// v. strahinja, 23 sep 2002
 this(convertArrayToVector(entries, loadLogChannel), loadLogChannel);
//    public ClassPath(URL[] entries, Logger loadLogger) {
//        this(convertArrayToVector(entries, loadLogger), loadLogger);
    }

    // private helper constructors

    /**
     * Constructs class path with specified class path entries.
     * Vector entries must be <CODE>ClassPathEntry</CODE> objects.
     *
     * @param entries The class path entries.
     * @param loadLogChannel The log channel, maybe null.
     * @see ClassPathEntry
     */
// v. strahinja, 23 sep 2002
 private ClassPath(Vector  entries, LogChannel loadLogChannel) {
//    private ClassPath(Vector entries, Logger loadLogger) {
 /*
     * This constructor actually does all the work, because all
     * other constructors call this one.
     */
        classPath = new Vector ();
        set(entries);

// v. strahinja, 23 sep 2002
          logChannel = loadLogChannel;
// v. strahinja, 23 sep 2002
 if (logChannel != null) {
// v. strahinja, 23 sep 2002
              logLevel = logChannel.getLevel(MultiClassLoader.LOG_LEVEL);
// v. strahinja, 23 sep 2002
              loggingEnabled = logChannel.isEnabled(logLevel);
// v. strahinja, 23 sep 2002
          }
//        logger = loadLogger;
//        if (logger != null) {
 //logLevel = logger.getLevel();
 //if (logLevel == null) {
//            logLevel = Level.DEBUG;  
 //}
//            loggingEnabled = logger.isEnabledFor(logLevel);
 //}
    }

    // public methods

    /**
     * Sets class path with specified class path entries.  The
     * parameter is assumed to be an array of directories, URLs, and/or
     * zip files.
     *
     * @param entries The class path represented by a String array.
     */
    public void set(String [] entries) {
// v. strahinja, 23 sep 2002
    set(convertArrayToVector(entries, logChannel));
//        set(convertArrayToVector(entries, logger));
    }

    /**
     * Sets class path with specified class path entries.  The
     * parameter is assumed to be an array of zip files and/or directories.
     *
     * @param entries The class path represented by a File array.
     */
    public void set(File [] entries) {
// v. strahinja, 23 sep 2002
        set(convertArrayToVector(entries, logChannel));
//        set(convertArrayToVector(entries, logger));
    }

    /**
     * Sets class path with specified class path entries.  The URLs can
     * represent directories and/or zip files on the local machine and/or
     * on remote machines.
     *
     * @param entries The class path represented by a URL array.
     */
    public void set(URL [] entries) {
// v. strahinja, 23 sep 2002
       set(convertArrayToVector(entries, logChannel));
//        set(convertArrayToVector(entries, logger));
    }

    /**
     * Adds specified class path entries to class path.  The
     * parameter is assumed to be an array of directories, URLs, and/or
     * zip files.
     *
     * @param entries The class path entries to add.
     */
    public void add(String [] entries) {
// v. strahinja, 23 sep 2002
       add(convertArrayToVector(entries, logChannel));
//        add(convertArrayToVector(entries, logger));
    }

    /**
     * Adds specified class path entries to class path.  The
     * parameter is assumed to be an array of zip files and/or directories.
     *
     * @param entries The class path entries to add.
     */
    public void add(File [] entries) {
// v. strahinja, 23 sep 2002
    add(convertArrayToVector(entries, logChannel));
//        add(convertArrayToVector(entries, logger));
    }

    /**
     * Adds specified class path entries to class path.  The URLs can
     * represent directories and/or zip files on the local machine and/or
     * on remote machines.
     *
     * @param entries The class path entries to add.
     */
    public void add(URL [] entries) {
// v. strahinja, 23 sep 2002
       add(convertArrayToVector(entries, logChannel));
//        add(convertArrayToVector(entries, logger));
    }

    /**
     * Clears class path by removing all entries.
     *
     * @see #set
     * @see #add
     */
    public void clear() {
        classPath.removeAllElements();
    }

    /**
     * Get the number of entries in the classpath.
     *
     * @return The length ot the class path.
     */
    public int getLength() {
    return classPath.size();
    }

    /**
     * Gets an Enumeration of class path entries.
     *
     * @return an Enumeration of ClassPathEntry objects.
     * @see ClassPathEntry
     * @see #set
     */
    public Enumeration  getPath() {
    return classPath.elements();
    }

    /**
     * Gets resource represented by specified file name.  The class path
     * entries are searched in order to find the desired resource.  If the
     * resource is not found in the class path, null is returned.
     *
     * @param name The file name of the resource.
     * @return the resource associated with the given file name, or null if
     *         it can not be found.
     * @see Resource
     */
    public Resource getResource(String  name) {

    if (name == null) {
        throw new NullPointerException ("Null resource name passed to " +
        "getResource() for class path, " + this);
    }
    Resource resource = null;
    for (int i = 0; i < classPath.size(); i++) {
        ClassPathEntry entry = null;
            entry = (ClassPathEntry)classPath.elementAt(i);

            if (loggingEnabled) {
// v. strahinja, 23 sep 2002
               logChannel.write(logLevel, "  checking: \"" + entry.getName()
//                logger.log(logLevel, "  checking: \"" + entry.getName()
                                 + "\"");
            }
        resource = entry.getResource(name);
        if (resource != null) {
                if (loggingEnabled) {
// v. strahinja, 23 sep 2002
                   logChannel.write(logLevel, "  found: " + name);
//                    logger.log(logLevel, "  found: " + name);
                }
        return resource;
        }
    }
        if (loggingEnabled) {
// v. strahinja, 23 sep 2002
            logChannel.write(logLevel, "  not found: " + name);
//            logger.log(logLevel, "  not found: " + name);
        }
    return null;
    }


    // private helper methods

    /**
     * Sets class path with specified class path entries.
     * Vector entries must be <CODE>ClassPathEntry</CODE> objects.
     * All null and duplicate entries are removed.
     *
     * @param entries The class path entries.
     * @see ClassPathEntry
     */
    private void set(Vector  entries) {
    /*
     * This set method actually does all the work,
     * since all the other set methods call this one.
     */
    classPath.removeAllElements();
    add(entries);
    }

    /**
     * Adds specified class path entries to class path.
     * Vector entries must be <CODE>ClassPathEntry</CODE> objects.
     * All null and duplicate entries are removed.
     *
     * @param entries The class path entries.
     * @see ClassPathEntry
     */
    private void add(Vector  entries) {
    /*
     * This add method actually does all the work,
     * since all the other add methods call this one.
     */
    if (entries != null) {
        for (int i = 0; i < entries.size(); i++) {
        classPath.insertElementAt(entries.elementAt(i), i);
        }
    }
    }

    /**
     * Converts array of Objects to Vector.  Converts all array objects
     * to ClassPathEntry objects and removes nulls and duplicates.
     *
     * @param array The array to convert.
     * @return the Vector representation of the array.
     * @see ClassPathEntry
     */
    private static Vector  convertArrayToVector(Object [] array,
// v. strahinja, 23 sep 2002
 LogChannel loadLogChannel) {
//                                               Logger loadLogger) {
 //FIXME: Is there really a need to support a nul array??
 if (array != null) {
        Vector  vector = new Vector ();
        for (int i = 0; i < array.length; i++) {
        Object  object = array[i];
        ClassPathEntry entry = null;
        if (object instanceof String ) {
// v. strahinja, 23 sep 2002
            entry = new ClassPathEntry((String )object, loadLogChannel);
//            entry = new ClassPathEntry((String)object, loadLogger);
        } else if (object instanceof File ) {
// v. strahinja, 23 sep 2002
            entry = new ClassPathEntry((File )object, loadLogChannel);
//            entry = new ClassPathEntry((File)object, loadLogger);
        } else if (object instanceof URL ) {
// v. strahinja, 23 sep 2002
            entry = new ClassPathEntry((URL )object, loadLogChannel);
//            entry = new ClassPathEntry((URL)object, loadLogger);
        } else {
            // This should not happen because the only public set
 // methods are for Strings, Files, and URLs
 throw new FatalExceptionError(new ClassCastException (
            "Type, " + object.getClass() + ", is not supported. " +
            "Expecting a String, File, or URL."));
        }
        if (entry != null && ! vector.contains(entry)) {
            vector.addElement(entry);
        }
        }
        return vector;
    }
    return null;
    }
}

