
/*
 * @(#)ElementImpl.java 1.36 02/03/21
 *
 * Copyright 2002 Sun Microsystems, Inc. All rights reserved.
 * SUN PROPRIETARY/CONFIDENTIAL. Use is subject to license terms.
 */
package org.enhydra.xml;

import java.util.ArrayList ;
import java.util.HashMap ;
import java.util.Iterator ;
import java.util.List ;

import org.w3c.dom.Attr ;
import org.w3c.dom.DOMException ;
import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.w3c.dom.NamedNodeMap ;
import org.w3c.dom.Node ;
import org.w3c.dom.NodeList ;
import org.w3c.dom.TypeInfo ;
import org.w3c.dom.UserDataHandler ;

/**
 * @author Tweety
 *
 * A class representing a node in a meta-data tree, which implements
 * the <a HREF="../../../../api/org/w3c/dom/Element.html">
 *
 * <p> Namespaces are ignored in this implementation.  The terms "tag
 * name" and "node name" are always considered to be synonymous.
 *
 * @version 1.0
 */
public class ElementImpl extends NodeImpl implements Element  {

    /**
     * A <code>HashMap</code> of <code>AttrImpl</code> nodes representing
     * attributes.
     */
    protected HashMap  attributes = null;


    /**
     * Constructs an empty <code>ElementImpl</code>.
     */
    public ElementImpl() {
        attributes = new HashMap ();
        type = ELEMENT_NODE;
    }


    /**
     * Constructs a <code>ElementImpl</code> from the given node,
     * without creating entire children subtree.
     *
     * @param element, as a <code>ElementImpl</code>.
     */
    public ElementImpl(ElementImpl element) {
        super(element);
        attributes = element.attributes;
        type = ELEMENT_NODE;
    }


    /**
     * Constructs an <code>ElementImpl</code> with the given
     * document owner and node name.
     *
     * @param ownerDoc the document owner of the node, as a <code>Document</code>.
     * @param nodeName the name of the node, as a <code>String</code>.
     */
    public ElementImpl(Document  ownerDoc, String  name) {
        super(ownerDoc,name,ELEMENT_NODE);
        this.attributes = new HashMap ();
    }


    /**
     * Constructs an <code>ElementImpl</code> with the given
     * document owner, node name, node type and node value.
     *
     * @param ownerDoc the document owner of the node, as a <code>Document</code>.
     * @param nodeName the name of the node, as a <code>String</code>.
     * @param type the type of the node, as a <code>short</code>.
     * @param value the value of the node, as a <code>String</code>.
     */
    protected ElementImpl(Document  ownerDoc, String  nodeName, short type, String  value) {
        super(ownerDoc, nodeName, type, value);
    }


    /**
     * Constructs an <code>ElementImpl</code> from a given node (creates the children subtree too),
     * as a <code>Node</code>
     *
     * @param node, as a <code>Node</code>.
     */
    public ElementImpl(Node  node) {
        this(node,true);
    }


    /**
     * Constructs an <code>ElementImpl</code> from a given node, as a <code>Node</code>,
     * and deep as <code>boolean</code>.
     *
     * @param node, as a <code>Node</code>.
     * @param deep if <code>true</code>, recursively clone the subtree
     * under the specified node; if <code>false</code>, clone only the
     * node itself.
     */
    public ElementImpl(Node  node, boolean deep) {
        super(node,false);
        attributes = new HashMap ();
        NamedNodeMap  attrs = node.getAttributes();
        if (attrs != null) {
            for (int i = 0; i < attrs.getLength(); i++) {
                Attr  attr = new AttrImpl((Attr ) attrs.item(i));
                attributes.put(attr.getName(), attr);
            }
        }
        if (deep)
            initNodeImplChildren(node);
    }


    /**
     * Creates new instance of <code>ElementImpl</code> from a given document
     * as a <code>Document</code>.
     *
     * @param document document.
     *
     * @return new <code>Element</code> node as a root of the <code>Document</code>.
     */
    public static Element  newInstance(Document  document) {
        Node  root = document.getDocumentElement();
        return new ElementImpl(root);
    }


    /**
     * Inserts the node <code>newChild</code> before the existing
     * child node <code>refChild</code>. If <code>refChild</code> is
     * <code>null</code>, insert <code>newChild</code> at the end of
     * the list of children.
     *
     * @param newChild the <code>Node</code> to insert.
     * @param refChild the reference <code>Node</code>.
     *
     * @return the node being inserted.
     *
     * @exception IllegalArgumentException if <code>newChild</code> is
     * <code>null</code>.
     */
    public Node  insertBefore(Node  newChild, Node  refChild) {
        super.insertBefore(newChild,refChild);
        return newChild;
    }


    /**
     * Replaces the child node <code>oldChild</code> with
     * <code>newChild</code> in the list of children, and returns the
     * <code>oldChild</code> node.
     *
     * @param newChild the <code>Node</code> to insert.
     * @param oldChild the <code>Node</code> to be replaced.
     *
     * @return the node replaced.
     *
     * @exception IllegalArgumentException if <code>newChild</code> is
     * <code>null</code>.
     */
    public Node  replaceChild(Node  newChild, Node  oldChild) {
        super.replaceChild(newChild,oldChild);
        return oldChild;
    }


    /**
     * Removes the child node indicated by <code>oldChild</code> from
     * the list of children, and returns it.
     *
     * @param oldChild the <code>Node</code> to be removed.
     *
     * @return the node removed.
     *
     * @exception IllegalArgumentException if <code>oldChild</code> is
     * <code>null</code>.
     */
    public Node  removeChild(Node  oldChild) {
        super.removeChild(oldChild);
        return oldChild;
    }


    /**
     * Returns a duplicate of this node.  The duplicate node has no
     * parent (<code>getParentNode</code> returns <code>null</code>).
     * If a shallow clone is being performed (<code>deep</code> is
     * <code>false</code>), the new node will not have any children or
     * siblings.  If a deep clone is being performed, the new node
     * will form the root of a complete cloned subtree.
     *
     * @param deep if <code>true</code>, recursively clone the subtree
     * under the specified node; if <code>false</code>, clone only the
     * node itself.
     *
     * @return the duplicate node.
     */
    public Node  cloneNode(boolean deep) {
        return new ElementImpl(this,deep);
    }





    // Methods from Element


    /**
     * Returns tag name of this node.
     *
     * @return tag name of this node as a <code>String</code>.
     */
    public String  getTagName() {
        return nodeName;
    }


    /**
     * Returns all attribute nodes of this node.
     *
     * @return all attribute nodes of this node as a <code>NamedNodeMap</code>.
     */
    public NamedNodeMap  getAttributes() {
        return new HashMapNamedNodeMap(attributes);
    }


    /**
     * Returns the value of the attribute with given name.
     *
     * @param name name of attribute.
     *
     * @return value of attribute.
     */
    public String  getAttribute(String  name) {
        Attr  attr = getAttributeNode(name);
        if (attr == null) {
            return "";
        }
        return attr.getValue();
    }


    /**
     * Equivalent to <code>getAttribute(localName)</code>.
     *
     * @see #setAttributeNS
     */
    public String  getAttributeNS(String  namespaceURI, String  localName) {
        return getAttribute(localName);
    }


    /**
     * To the <code>name</code> attribute set value to <code>value</code>.
     *
     * @param name attribute value.
     * @param value new attribute value.
     */
    public void setAttribute(String  name, String  value) {
        attributes.put(name, new AttrImpl(this, name, value));
    }


    /**
     * Equivalent to <code>setAttribute(qualifiedName, value)</code>.
     *
     * @see #getAttributeNS
     */
    public void setAttributeNS(String  namespaceURI, String  qualifiedName, String  value) {
        setAttribute(qualifiedName, value);
    }


    /**
     * Removes attribute with the given name.
     *
     * @param name attribute name.
     */
    public void removeAttribute(String  name) {
        if (type != ELEMENT_NODE)
            throw new NodeDOMException(
                DOMException.NOT_SUPPORTED_ERR,
                "Node doesn't have attributes");
        removeAttribute(name, true);
    }


    private void removeAttribute(String  name, boolean checkPresent) {
        if (attributes.remove(name) != null)
            return;
        // If we get here, the attribute doesn't exist
 if (checkPresent) {
            throw new NodeDOMException(
                DOMException.NOT_FOUND_ERR,
                "No such attribute!");
        }
    }


    /**
     * Returns <code>true</code>, if this node has attributes, otherwise
     * <code>false</code>.
     *
     * @return <code>true</code> if node has attributes, otherwise <code>false</code>..
     */
    public boolean hasAttributes() {
        return attributes.size() > 0;
    }


    /**
     * Returns <code>true</code>, if this node has attribute with given name,
     * otherwise <code>false</code>.
     *
     * @return <code>true</code> if node has given attribute, otherwise <code>false</code>..
     */
    public boolean hasAttribute(String  name) {
        return getAttributeNode(name) != null;
    }


    /**
     * Equivalent to <code>removeAttribute(localName)</code>.
     */
    public void removeAttributeNS(String  namespaceURI, String  localName) {
        removeAttribute(localName);
    }


    /**
     * Returns attribute value with given name of this node.
     *
     * @param name name of attribute.
     *
     * @return value of attribute.
     */
    public Attr  getAttributeNode(String  name) {
        return (Attr ) attributes.get(name);
    }


    /**
     * Equivalent to <code>getAttributeNode(localName)</code>.
     *
     * @see #setAttributeNodeNS
     */
    public Attr  getAttributeNodeNS(String  namespaceURI, String  localName) {
        return getAttributeNode(localName);
    }


    /**
     * Add new attribute to this node.
     *
     * @param newAttr new attribute.
     *
     * @return new attribute as <code>AttrImpl</code>.
     */
    public Attr  setAttributeNode(Attr  newAttr) throws DOMException  {
        AttrImpl attr;
        if (newAttr instanceof AttrImpl) {
            attr = (AttrImpl) newAttr;
        } else {
            attr = new AttrImpl(newAttr);
        }
        attributes.put(attr.getName(), attr);
        return attr;
    }


    /**
     * Equivalent to <code>setAttributeNode(newAttr)</code>.
     *
     * @see #getAttributeNodeNS
     */
    public Attr  setAttributeNodeNS(Attr  newAttr) {
        return setAttributeNode(newAttr);
    }


    /**
     * Remove attribute from this node.
     *
     * @param oldAttr attribute that will be removed.
     *
     * @return old attribute as <code>AttrImpl</code>.
     */
    public Attr  removeAttributeNode(Attr  oldAttr) {
        removeAttribute(oldAttr.getName());
        return oldAttr;
    }


    /**
     * Equivalent to <code>hasAttribute(localName)</code>.
     */
    public boolean hasAttributeNS(String  namespaceURI, String  localName) {
        return hasAttribute(localName);
    }


    /**
     * Returns all <code>Element</code> nodes with given name,
     * searching by all sub nodes from this node.
     *
     * @param name tag name.
     *
     * @return all <code>Element</code> vith given name as <code>NodeList</code>.
     */
    public NodeList  getElementsByTagName(String  name) {
        List  list = new ArrayList ();
        getElementsByTagName(name, list);
        return new NodeListImpl(list);
    }


    private void getElementsByTagName(String  name, List  list) {
        if (nodeName.equals(name)) {
            list.add(this);
        }

        Node  child = getFirstChild();
        while (child != null) {
            if (child.getNodeType() == Node.ELEMENT_NODE)
                ((ElementImpl)child).getElementsByTagName(name, list);
            child = child.getNextSibling();
        }
    }


    /**
     * Equivalent to <code>getElementsByTagName(localName)</code>.
     */
    public NodeList  getElementsByTagNameNS(String  namespaceURI, String  localName) {
        return getElementsByTagName(localName);
    }


    /**
     * Returns <code>true</code> if this node has children nodes.
     *
     * @return <code>true</code> if this node has children.
     */
    public boolean hasElementChildNodes() {
        Node  child = getFirstChild();
        while (child != null) {
            if (child.getNodeType() == Node.ELEMENT_NODE)
                return true;
            child = child.getNextSibling();
        }
        return false;
    }


    /**
     * Method beginToString for this class writes the xml
     * begining tag string and all attributes.
     *
     * @param sb string buffer to add resulting string.
     * @param indent used in formating the output.
     */
    protected void beginToString(StringBuffer  sb, Indent indent) {
        sb.append("\n" + indent + "<" + this.nodeName);

        for (Iterator  iter = attributes.values().iterator(); iter.hasNext();) {
            Attr  attr = (Attr ) iter.next();
            sb.append(" " + attr.getNodeName() + "=\"" + attr.getNodeValue() + "\"");
        }

        sb.append(">");
        indent.increment();
    }


    /**
     * Method endToString for this class writes the xml
     * ending tag string.
     *
     * @param sb string buffer to add resulting string.
     * @param indent used in formating the output.
     */
    protected void endToString(StringBuffer  sb, Indent indent) {
        indent.decrement();
        if (hasElementChildNodes())
            sb.append("\n" + indent + "</" + this.nodeName + ">");
        else
            sb.append("</" + this.nodeName + ">");
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Element#getSchemaTypeInfo()
     */
    public TypeInfo  getSchemaTypeInfo() {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Element#setIdAttribute(java.lang.String, boolean)
     */
    public void setIdAttribute(String  arg0, boolean arg1) throws DOMException  {
        // TODO Auto-generated method stub
 
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Element#setIdAttributeNS(java.lang.String, java.lang.String, boolean)
     */
    public void setIdAttributeNS(String  arg0, String  arg1, boolean arg2) throws DOMException  {
        // TODO Auto-generated method stub
 
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Element#setIdAttributeNode(org.w3c.dom.Attr, boolean)
     */
    public void setIdAttributeNode(Attr  arg0, boolean arg1) throws DOMException  {
        // TODO Auto-generated method stub
 
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#getNamespaceURI()
     */
    public String  getNamespaceURI() {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#getBaseURI()
     */
    public String  getBaseURI() {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#compareDocumentPosition(org.w3c.dom.Node)
     */
    public short compareDocumentPosition(Node  arg0) throws DOMException  {
        // TODO Auto-generated method stub
 return 0;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#getTextContent()
     */
    public String  getTextContent() throws DOMException  {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#setTextContent(java.lang.String)
     */
    public void setTextContent(String  arg0) throws DOMException  {
        // TODO Auto-generated method stub
 
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#isSameNode(org.w3c.dom.Node)
     */
    public boolean isSameNode(Node  arg0) {
        // TODO Auto-generated method stub
 return false;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#lookupPrefix(java.lang.String)
     */
    public String  lookupPrefix(String  arg0) {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#isDefaultNamespace(java.lang.String)
     */
    public boolean isDefaultNamespace(String  arg0) {
        // TODO Auto-generated method stub
 return false;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#lookupNamespaceURI(java.lang.String)
     */
    public String  lookupNamespaceURI(String  arg0) {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#isEqualNode(org.w3c.dom.Node)
     */
    public boolean isEqualNode(Node  arg0) {
        // TODO Auto-generated method stub
 return false;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#getFeature(java.lang.String, java.lang.String)
     */
    public Object  getFeature(String  arg0, String  arg1) {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#setUserData(java.lang.String, java.lang.Object, org.w3c.dom.UserDataHandler)
     */
    public Object  setUserData(String  arg0, Object  arg1, UserDataHandler  arg2) {
        // TODO Auto-generated method stub
 return null;
    }


    /* (non-Javadoc)
     * @see org.w3c.dom.Node#getUserData(java.lang.String)
     */
    public Object  getUserData(String  arg0) {
        // TODO Auto-generated method stub
 return null;
    }
}
