

/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: KeywordValueTable.java,v 1.3 2005/03/24 12:23:18 slobodan Exp $
 */





package com.lutris.util;

import java.lang.reflect.Array ;
import java.lang.reflect.Method ;
import java.util.Enumeration ;
import java.util.Hashtable ;
import java.util.Vector ;

/**
 * Class that implements a recursive keyword/value table.  The key is a string
 * that is restricted to be a valid Java identifier.  That is, starting with
 * an letter and containing letters or digits.  The characters '_' and '$'
 * are also allowed and are treated as letters.  The value maybe any object.
 * A keyword and its value are collectively referred to as a <I>field</I>
 *
 * The table is recursive. Values of class KeywordValueTable are referred to as
 * <I>sections</I>.  A field of a section maybe addressed from the parent
 * object using a dot ('.') separated name path.
 *
 * @version $Revision: 1.3 $
 * @author  Mark Diekhans
 * @since   Harmony 1.0
 */
public class KeywordValueTable implements java.io.Serializable  {
    private Hashtable  hashTable;   // Table keyed by keyword component.
 
    /**
     * Constructor.
     */
    public KeywordValueTable() {
        hashTable = new Hashtable ();
    }

    
    /**
     * Recursively locate the section named by a keyword path.  This finds
     * the section containing the last component of the path.
     *
     * @param keywordPath The parsed keyword to locate.
     * @param create If <CODE>true</CODE>, create missing section, if
     *  <CODE>false</CODE> return null on a missing section.
     * @param pathIdx Current index in keywordPath.  Zero for top level call.
     * @return A reference to the section.  For a single component keyword,
     *  <CODE>this</CODE> is returned.  If the second is not found.
     * @exception KeywordValueException If a non-leaf
     *  element of the keyword is not a section.
     */
    private synchronized KeywordValueTable findSection(String [] keywordPath,
                                                       boolean create,
                                                       int pathIdx)
            throws KeywordValueException {
        /*
         * We don't return the leaf key, stop one above.
         */
        if (pathIdx == keywordPath.length - 1) {
            return this;
        }

        /*
         * Recurse down to the leaf's section.
         */
        Object  value = hashTable.get(keywordPath [pathIdx]);
        if (value != null) {
            /*
             * If its a different type, then replace if create otherwise its
             * an error.
             */
            if (!(value instanceof KeywordValueTable)) {
                if (!create) {
                    String  msg = "keyword specifies a non-leaf component " +
                        "that is not a KeywordValueTable: " +
                        KeywordParser.join (keywordPath) +
                        " (component #" + pathIdx + ")";
                    throw new KeywordValueException(msg);
                }
                value = newSection();
                hashTable.put(keywordPath [pathIdx], value);
            }
        } else {
            /*
             * Value does not exist.
             */
            if (!create) {
                return null;
            }
            value = newSection();
            hashTable.put(keywordPath [pathIdx], value);
        }
        return ((KeywordValueTable) value).findSection(keywordPath,
                                                       create,
                                                       pathIdx + 1);
    }
    
    /**
     * Allocate a new section.  The default implementation of this
     * method returns a new KeywordValueTable object.  A class derived
     * from KeywordValueTable overrides this method to create a new
     * object of the derived type.  Sections are only allocated by
     * this method.
     *
     * @return A reference to a new section.
     */
    protected KeywordValueTable newSection() {
        return new KeywordValueTable();
    }

    /**
     * Get the value of a field as an object.
     *
     * @param keyword  The keyword of the field.  This can be a simple keyword
     *  or a recursive, dot-seperated keyword path.
     * @return The object value or null if its not found.
     * @exception KeywordValueException If the keyword is not syntactically 
     *  legal or is a non-leaf element of the keyword is not a section.
     */
    public synchronized Object  get(String  keyword)
        throws KeywordValueException {
        
        String  [] keywordPath = KeywordParser.parse(keyword);
        KeywordValueTable section = findSection(keywordPath,
                                                false,  // create
 0);
        if (section == null) {
            return null;
        }
        return section.hashTable.get(keywordPath [keywordPath.length - 1]);
    }

    /**
     * Get the value of a field as an object, return a default if it
     * does not exist.
     *
     * @param keyword  The keyword of the field.  This can be a simple keyword
     *  or a recursive, dot-seperated keyword path.
     * @param defaultValue The default value to return it the keyword does not
     *  exist.
     * @return The object value or <CODE>defaultValue</CODE> if its not found.
     * @exception KeywordValueException If the keyword is not syntactically 
     *  legal or is a non-leaf element of the keyword is not a section.
     */
    public synchronized Object  get(String  keyword,
                                   Object  defaultValue)
        throws KeywordValueException {

        Object  value;
        String  [] keywordPath = KeywordParser.parse (keyword);
        KeywordValueTable section = findSection(keywordPath,
                                                false,  // create
 0);
        if (section == null) {
            value = defaultValue;
        } else {
            value = section.hashTable.get(keywordPath [keywordPath.length - 1]);
            if (value == null) {
                value = defaultValue;
            }
        }
        return value;
    }

    /**
     * Get the value of a field as a string
     *
     * @param keyword  The keyword of the field.  This can be a simple keyword 
     *  or a recursive, dot-seperated keyword path.
     * @return The result of calling toString on the value object
     *  or null if its not found.
     * @exception KeywordValueException If the keyword is not syntactically 
     *  legal or is a non-leaf element of the keyword is not a section.
     */
    public synchronized String  getString(String  keyword)
        throws KeywordValueException {
        
        Object  value = get(keyword);
        if (value == null) {
            return null;
        }
        return value.toString();
    }

    /**
     * Get the value of a field as a string, return a default if it
     * does not exist.
     *
     * @param keyword  The keyword of the field.  This can be a simple keyword 
     *  or a recursive, dot-seperated keyword path.
     * @return The result of calling toString on the value object
     *  or <CODE>defaultValue</CODE> if its not found.
     * @exception KeywordValueException If the keyword is not syntactically
     *  legal or the value object is not a String.
     */
    public synchronized String  getString(String  keyword,
                                         String  defaultValue)
        throws KeywordValueException {
        
        Object  value = get(keyword);
        if (value == null) {
            return defaultValue;
        }
        return value.toString();
    }

    /**
     * Get the value of a section.  The section is a value that is another
     * KeywordValueTable object.
     *
     * @param keyword  The keyword of the field.  This can be a simple keyword
     *  or a recursive, dot-seperated keyword path.
     * @return A reference to the section object or null if not found.
     * @exception KeywordValueException If the keyword is not syntactically
     *  legal or a non-leaf element of the
     *  keyword is not a section or the value object is not a
     *  KeywordValueTable.
     */
    public synchronized KeywordValueTable getSection(String  keyword)
            throws KeywordValueException {
        
        Object  value = get(keyword);
        if (value == null) {
            return null;
        }
        if (!(value instanceof KeywordValueTable)) {
            String  msg = "Value of field \"" + keyword +
                " is not a KeywordValueTable; it is " +
                value.getClass().getName ();
            throw new KeywordValueException(msg);
        }
        return (KeywordValueTable)value;
    }

    /**
     * Set the value of a field.  If a keyword path is specified and the
     * subsections do not exist, they are created.  If a field other than
     * a KeywordValueTable is one of the intermediate sections in the path, it
     * will be deleted and replaced by a section.
     *
     * @param keyword  The keyword of the field.  This can be a simple keyword
     *  or a recursive, dot-seperated keyword path.
     * @param value The value to associate with the keyword.  The value may
     *  not be null.
     * @exception KeywordValueException If the keyword is not syntactically
     *  legal.
     */
    public synchronized void set(String  keyword,
                                 Object  value)
        throws KeywordValueException {

        String  [] keywordPath = KeywordParser.parse(keyword);
        KeywordValueTable section = findSection(keywordPath,
                                                true,  // create
 0);
        section.hashTable.put (keywordPath[keywordPath.length - 1],
                               value);
    }


    /**
     * Sets a default value for a keyword.  This method only sets a value
     * for the specified keyword if a value is <B>not</B> already set for
     * that keyword.  If a value is not set for the keyword, then
     * if a keyword path is specified and the
     * subsections do not exist, they are created.  If a field other than
     * a KeywordValueTable is one of the intermediate sections in the path, it
     * will be deleted and replaced by a section.  
     *
     * @param keyword  The keyword of the field.  This can be a simple keyword
     *  or a recursive, dot-seperated keyword path.
     * @param defaultValue The default value to associate with the keyword.
     *  The default value may not be null.  The default value is only
     *  set if the specified keyword does
     *  not already have a value associated with it.
     * @exception KeywordValueException If the keyword is not syntactically
     *  legal.
     */
    public synchronized void setDefault(String  keyword,
                                        Object  defaultValue)
        throws KeywordValueException
    {
        if (!containsKey(keyword))
            set(keyword, defaultValue);
    }
    
    /**
     * Determine if the a field with the specified keyword exists.
     *
     * @param keyword  The keyword of the field.  This can be a simple keyword
     *  or a recursive, dot-seperated keyword path.
     * @return <CODE>true</CODE> if the code is in the table;
     *  <CODE>false</CODE> if its not.
     * @exception KeywordValueException If the keyword is not syntactically
     *  legal.
     */
    public synchronized boolean containsKey (String  keyword)
            throws KeywordValueException {

        String  [] keywordPath = KeywordParser.parse (keyword);
        KeywordValueTable section = findSection(keywordPath,
                                                false,  // create
 0);
        if (section == null) {
            return false;
        }
        return section.hashTable.containsKey(keywordPath [keywordPath.length - 1]);
    }

    /**
     * Get the keywords in the table.  This is only the keywords at the top
     * level, its doesn't recurse.
     *
     * @return An string array of the keywords.
     */
    public synchronized String [] keys() {
        Enumeration  keyEnum = hashTable.keys();

        /*
         * Build list as vector then convert it to a array when size is known.
         */
        Vector  keyList = new Vector  ();
        while (keyEnum.hasMoreElements()) {
            keyList.addElement (keyEnum.nextElement());
        }
        
        String  [] keyStrings = new String  [keyList.size()];
        for (int idx = 0; idx < keyList.size (); idx++) {
            keyStrings [idx] = (String ) keyList.elementAt (idx);
        }
        return keyStrings;
    }
  
    /**
     * Recursively get the keywords for the entire table.  This returns the
     * full keyword of all leaf values.
     *
     * @return An string array of the keywords.
     */
    public synchronized String  [] leafKeys() {
        Enumeration  keyEnum = hashTable.keys ();

        /*
         * Build list as vector then convert it to a array when size is known.
         * Recurse down each value if its another table.
         */
        Vector  keyList = new Vector  ();
        while (keyEnum.hasMoreElements()) {
            String  key = (String )keyEnum.nextElement();
            Object  value = hashTable.get(key);

            if (value instanceof KeywordValueTable) {
                String  subKeys [] = ((KeywordValueTable) value).leafKeys ();
                for (int idx = 0; idx < subKeys.length; idx++) {
                    keyList.addElement(KeywordParser.concat(key,
                                                            subKeys[idx]));
                }
            } else {
                keyList.addElement(key);
            }
        }
        
        String  [] keyStrings = new String [keyList.size()];
        for (int idx = 0; idx < keyList.size(); idx++) {
            keyStrings [idx] = (String )keyList.elementAt(idx);
        }
        return keyStrings;
    }
  
    /**
     * Delete a field, if the field does not exist, the operation is ignored.
     *
     * @param keyword  The keyword of the field.  This can be a simple keyword
     *  or a recursive, dot-seperated keyword path.
     * @exception KeywordValueException If the keyword is not syntactically
     *  legal.
     */
    public synchronized void remove(String  keyword)
        throws KeywordValueException {

        String  [] keywordPath = KeywordParser.parse(keyword);
        KeywordValueTable section = findSection(keywordPath,
                                                false,  // create
 0);
        if (section != null) {
            section.hashTable.remove(keywordPath[keywordPath.length - 1]);
        }
    }

    /**
     * Convert to a string.
     *
     * @return Generate a string representation of this object.
     */
    public synchronized String  toString () {
        return hashTable.toString();
    }

    /**
     * Convert to an Html representation.
     *
     * @return the generated Html.
     */
    public synchronized String  toHtml() {

    StringBuffer  html = new StringBuffer ();
        Enumeration  keyEnum = hashTable.keys();

        html.append ("<UL>\n");
        Vector  keyList = new Vector ();
        if (!keyEnum.hasMoreElements())
            return "";  // This makes it possible to detect empty tables.
 while (keyEnum.hasMoreElements()) {
            String  key = (String )keyEnum.nextElement();
            Object  value = hashTable.get(key);

            html.append("<LI> <TT>");
            html.append(key);
            html.append(": </TT>");
            html.append(formatFieldAsHtml(value));
            html.append ("\n");
        }
        html.append ("</UL>\n");

    return html.toString();
    }

    /**
     * Format an array section as a HTML ordered list, appending it
     * to the list.
     *
     * @param arrayObj Field array object to format.
     * @return generated Html
     */
    private String  formatArrayAsHtml(Object  arrayObj) {

    StringBuffer  html = new StringBuffer ();

    html.append("<OL START=0>\n");
    for (int idx = 0; idx < Array.getLength(arrayObj); idx++) {
            html.append("<LI>");
            html.append(formatFieldAsHtml(Array.get(arrayObj, idx)));
            html.append("\n");
    }
    html.append("</OL>\n");
    
    return html.toString();
    }

    /**
     * Format an object as HTML. Known printable objects are converted
     * to a string, while others are simple listed as &lt;Object&gt;.
     *
     * @param obj Field array object to format.
     * @return generated Html
     */
    private String  formatObjectAsHtml(Object  obj) {
    String  html;

    if (obj instanceof String ) {
            html = obj.toString();
    } else if (obj instanceof Integer ) {
        html = "<I><FONT SIZE=-1>(Integer)</FONT></I>" + obj.toString();
    } else if (obj instanceof Boolean ) {
        html = "<I><FONT SIZE=-1>(Boolean)</FONT></I>" + obj.toString();
    } else if (obj instanceof Double ) {
        html = "<I><FONT SIZE=-1>(Double)</FONT></I>" + obj.toString();
    } else if (obj instanceof Long ) {
        html = "<I><FONT SIZE=-1>(Long)</FONT></I>" + obj.toString();
    } else if (obj instanceof Short ) {
        html = "<I><FONT SIZE=-1>(Short)</FONT></I>" + obj.toString();
    } else if (obj instanceof Float ) {
        html = "<I><FONT SIZE=-1>(Float)</FONT></I>" + obj.toString();
    } else if (obj instanceof Character ) {
        html = "<I><FONT SIZE=-1>(Character)</FONT></I>" + obj.toString();
    } else {
        // If object has a toHtml() method then call it,
 // else print nothing.
 try {
            Class  objClass = obj.getClass();
            Method  toHtmlMethod = objClass.getMethod("toHtml", (Class[]) null);
        html = new String  ("<I><FONT SIZE=-1>(Object)" +
                   toHtmlMethod.invoke(obj, (Object[]) null) +
                   "</FONT></I>");
        } catch (Exception  e) {
        html = "<I><FONT SIZE=-1>(Object)</FONT></I>";
        }
    }
    
    return html;
    }
    
    /**
     * Format an field.  Normally just append toString the object.
     * However, arrays and recursive data are formatted.
     *
     * @param fieldObj Field object to format.
     * @return generated Html
     */
    private String  formatFieldAsHtml(Object  fieldObj) {

    String  html;

        if (fieldObj instanceof KeywordValueTable) {
            html = ((KeywordValueTable)fieldObj).toHtml();
        } else if (fieldObj.getClass().isArray()) {
            html = formatArrayAsHtml(fieldObj);
        } else {
            html = formatObjectAsHtml(fieldObj);
        }

    return html;
    }
}
