

/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: Config.java,v 1.6 2005/04/27 18:30:33 slobodan Exp $
 */

package com.lutris.util;


import java.lang.reflect.Array ;
import java.lang.reflect.Constructor ;
import java.util.Hashtable ;

import org.enhydra.util.ConfigFileInterface;

/**
 * Config is essentially a KeywordValueTable used for recursive
 * storage of data derived from a config file.  The contents is
 * initialized but <CODE>ConfigFile</CODE>.
 *
 * @see ConfigFile
 * @see KeywordValueTable
 * @author John Marco
 * @author Shawn McMurdo
 * @version $Revision: 1.6 $
 */
public class Config extends KeywordValueTable {

/**
 * Passed as the "count" argument to getInts, getLongs(), etc. to
 * indicate that all available elements are to be retrieved.
 */
 private static final int   GET_ALL = -1;

/**
 * The ConfigFile object this Config is associated with (if any)
 */
  private ConfigFileInterface configFile = null;

/**
 * Default constructor for an empty Config.
 */
  public Config() {
    super();
  }

/**
 * Constructor that takes a KeywordValueTable as initialization.
 * @param kvt   KeywordValueTable with which to initialize Config
 */
 public Config(KeywordValueTable kvt) {
   super();
   String [] keys = kvt.keys();
   for (int i = 0; i < keys.length; i++) {
     try {
       set(keys[i], kvt.get(keys[i]));
     }
     catch (KeywordValueException e) {
       // This shouldn't happen
 throw new FatalExceptionError(e);
     }
   }
 }

/**
 * Constructor that takes a KeywordValueTable and a ConfigFile as
 * initialization. The ConfigFile is associated with this Config
 * object.
 * @param kvt   KeywordValueTable with which to initialize Config
 * @param configFile ConfigFile to associate this Config object with
 */
 public Config(KeywordValueTable kvt, ConfigFileInterface configFile) {
    this(kvt);
  this.configFile = configFile;
 }

/**
 * Constructor that associates this Config with a given ConfigFile.
 * @param configFile ConfigFile to associate this object with
 */
 public Config(ConfigFileInterface configFile) {
    this();
  this.configFile = configFile;
 }

/**
 * Returnes cloned Config object. The cloned and original objects have exatly
 * same key value pairs, but free of any common object references. Only
 * reference which is same for both objects (the original, and it's clone) is
 * reference to associated ConfigFile object (private argument of Config class).
 * @return A cloned Config object.
 * @exception KeywordValueException
 */
 public Config getClonedConfig() throws KeywordValueException{ // VR 13.12.2002.

   Config returnConfig = this.getClonedConfigParams();

// TJ 08.11.2003. put under comment begin
//   ConfigFileInterface cf = new ConfigFile(returnConfig);
//   cf.setFile(this.getConfigFile().getFile());
//   returnConfig.setConfigFile(cf);
//   return returnConfig;
// TJ 08.11.2003. put under comment end

   ConfigFileInterface cf = null;
   Class  classObj = this.getConfigFile().getClass();
   String  className = classObj.getName();

   Class [] classParam = new Class [1];
   Constructor  constr = null;
   Object [] arguments = new Object [1];
   try {
     classParam[0] = Class.forName("com.lutris.util.Config");
     constr = classObj.getConstructor(classParam);
     arguments[0] = (Object )(this);
     cf = (ConfigFileInterface)constr.newInstance(arguments);
   cf.setFile(this.getConfigFile().getFile());
   returnConfig.setConfigFile(cf);
   }
   catch (Exception  ex){
    returnConfig.setConfigFile(cf);
   }
   return returnConfig;
 }


/**
 * Returnes cloned Config object. The cloned and original objects have exatly
 * same key-value pairs, but free of any common object references. Returned
 * cloned Config object has not defined it's private argument of type ConfigFile
 * (reference to it's configuration file on hard disc). Method is specialy
 * designed to enable recursion calls, and it should be called by public method
 * getClonedConfig().
 * @return A cloned Config object withouth reference to it's configuration file.
 * @exception KeywordValueException
 */
 private Config getClonedConfigParams() throws KeywordValueException{ // VR 13.12.2002.

   Config returnConfig = new Config();
   String [] keys = this.keys();

   for (int i = 0; i < keys.length; i++) {
     Object  tempValue = this.get(keys[i]);

     if(tempValue instanceof String ) {
       returnConfig.set( new String (keys[i]), new String ((String )tempValue) );
     }
     else if (tempValue instanceof KeywordValueTable) { // useing of recursion
 Config kvtTempConfig = (new Config( (KeywordValueTable)tempValue)).getClonedConfigParams();
       returnConfig.set( new String (keys[i]), (KeywordValueTable)kvtTempConfig );
     }
     else if (tempValue instanceof Config) { // useing of recursion
 Config tempConfig = ((Config)tempValue).getClonedConfigParams();
       returnConfig.set( new String (keys[i]), tempConfig );
     }
     else if( tempValue.getClass().isArray() ) {
       int len = Array.getLength(tempValue);
       String [] newTemp = new String [len];
       for (int k=0; k<len; k++) {
         newTemp[k] = new String ( Array.get(tempValue,k).toString() );
       }
       returnConfig.set( new String (keys[i]), newTemp );
     }
   }
   return returnConfig;
 }
// DT 30.12.2003 BEG
/**
* Returns the hashtable containing names and values of all config parameters
* Names are in the form name-level-0_name-level-1_name-level-2...
*
* @param prefix - previous level name
* @return hashtable of all config parameters
*/
public Hashtable  allConfigParams(String  prefix) throws KeywordValueException{ 

   
   String [] keys = this.keys();
    Hashtable  ht=new Hashtable ();   
   for (int i = 0; i < keys.length; i++) {
     Object  tempValue = this.get(keys[i]);
     String  name;
     
     if(prefix != null && prefix.length()>0){
        name=prefix+"__"+keys[i];
     }else{
        name=keys[i];
     }
     
     if(tempValue instanceof String ) {
        ht.put(name, (String )tempValue);
     }
     else if (tempValue instanceof KeywordValueTable) { // useing of recursion
       ht.putAll(((Config)tempValue).allConfigParams(name));
     }
     else if (tempValue instanceof Config) { // useing of recursion
       ht.putAll(((Config)tempValue).allConfigParams(name));
     }
     else if( tempValue.getClass().isArray() ) {
       int len = Array.getLength(tempValue);
       for (int k=0; k<len; k++) {
           ht.put( name + "__" + k, Array.get(tempValue,k).toString() );
       }
     }
   }
   return ht;
 }
// DT 30.12.2003 END
/**
 * Imorts and synchronizes to, all parameters (key-value pairs) according to
 * given Config object parameters. All parameters which do not exist in imported
 * Config object will be removed.
 * @param config Config object which has key-value pairs for importing.
 * @exception KeywordValueException
 */
 public void importConfig(Config config) throws KeywordValueException{ // VR 14.12.2002.

   String [] original =  this.keys();
   String [] in = config.keys();

//adding or changing keys from imported Config object
 for(int i=0; i<in.length; i++)
     this.set( in[i], config.get(in[i]) );
//removing keys which did not exist in imported Config object
 for(int i=0; i<original.length; i++) {
     if(!config.containsKey(original[i]))
       this.remove( in[i] );
   }
 }


/**
 * Allocates a new section.  The overrides the default method to allocate a
 * section that is of class <CODE>Config<CODE>.
 * @return A reference to a new section.
 * @see KeywordValueTable#newSection
 */
  protected KeywordValueTable newSection() {
      return new Config(configFile);
  }

/**
 * Gets the <code>ConfigFile</code> associated with this object.
 * @return the associated <code>ConfigFile</code>,
 * <code>null</code> if there is no config file associated with this
 * object.
 */
  public ConfigFileInterface getConfigFile() {
    return configFile;
  }

/**
 * Sets the <code>ConfigFile</code> associated with this object.
 * For use by <code>ConfigFile</code> only, anyone else please use
 * the appropriate constructor
 * @param configFile ConfigFile object associated with this object
 */
  public void setConfigFile(ConfigFileInterface configFile) {
    this.configFile = configFile;
  }

/**
 * Get the value of a section as a <CODE>Config</CODE> object.
 * @param keyword   The keyword of the field.  This can be a simple keyword
 * or a recursive, dot-seperated keyword path.
 * @return A reference to the section object or null if not found.
 * @exception KeywordValueException If the keyword is not syntactically
 * legal or a non-leaf element of the   keyword is not a section or the value
 * object is not a KeywordValueTable.
 * @see KeywordValueTable#getSection
 */
  public synchronized Config getConfig(String  keyword)
      throws KeywordValueException {
    return (Config) getSection(keyword);
  }

/**
 * Gets the value of a section as a <CODE>KeywordValueTable</CODE> object.
 * This method overrides the KeywordValueTable.getSection in order to
 * insure that Config.getSection() always returns a Config object even
 * if a KeywordValueTable was inserted into the Config as a section.
 * @param keyword   The keyword of the field.  This can be a simple keyword
 * or a recursive, dot-seperated keyword path.
 * @return A reference to the section object or null if not found.
 * @exception KeywordValueException If the keyword is not syntactically legal
 * or a non-leaf element of the keyword is not a section or the value object is
 * not a KeywordValueTable.
 * @see KeywordValueTable#getSection
 */
  public synchronized KeywordValueTable getSection(String  keyword)
          throws KeywordValueException {
    KeywordValueTable kvt = super.getSection(keyword);
    if (kvt == null) {
      return null;
    }
    if (kvt instanceof Config) {
      return kvt;
    }
    else {
      return new Config(kvt, configFile);
    }
  }

/**
 * Returns <code>true</code> if the specified key is found,
 * <code>false</code> otherwise.
 * @param   key The key whose existence is to be tested.
 * @return  <code>true</code> if the key was found, otherwise <code>false</code>.
 */
  public boolean containsKey(String  key) {
    boolean result = false;
    try {
      result = super.containsKey(key);
    }
    catch (KeywordValueException e) {
      result = false;
    }
    return result;
  }

/**
 * Returns the number of data elements for a given key, or <code>-1</code> if
 * the key is not found.
 * @param   key The key to search for.
 * @return      The number of entries for the given key, or <code>-1</code> if
 * the key is not found.
 * @exception ConfigException
 */
  public int containsCount(String  key) throws ConfigException {
    Object  valObj = null;
    try {
      valObj = get(key);
      if (valObj == null) return -1;
      return Array.getLength(valObj);
    }
    catch (KeywordValueException e) {
      throw new ConfigException(e.getMessage());
    }
    catch (IllegalArgumentException  e) {
      // Caused by object not being array.
    }
// Assume if object was not null, it was single-valued entity.
 if (valObj == null) return -1;
    return 1;
  }

/**
 * Is the key is an array, or a single value. If this returns true,
 * you should use <CODE>getStrings()</CODE> (or if you know the type
 * of the data, you can use, for example, <CODE>getInts()</CODE>).
 * If this returns false, you shoud use <CODE>getString()</CODE>
 * (or if you know the type of the data, you can use, for example,
 * <CODE>getInt()</CODE>).
 * @param   key The key to search for.
 * @return True if the key is an array, false if it is a single value.
 * @exception ConfigException If the key is not found.
 */
  public boolean isArray(String  key) throws ConfigException {
    Object  valObj = null;
    try {
      valObj = get(key);
      if (valObj == null)
          throw new ConfigException("Key \"" + key + "\" not found.");
      // Attempt array access. This will fail if not an array.
      Array.getLength(valObj);
      // It must be an array if we made it to here.
 return true;
    }
    catch (KeywordValueException e) {
      throw new ConfigException(e.getMessage());
    }
    catch (IllegalArgumentException  e) {
      // Caused by object not being array.
 return false;
    }
  }

/**
 * Returns the array of longs associated with a given key.  If the
 * <code>count</code> parameter is not <code>GET_ALL</code> and the
 * number of elements in the retrieved array does not match <code>
 * count</code> then a ConfigException is thrown with <code>reason</code>
 * set to <code>COUNT</code>.
 * If any of the retrieved elements cannot be converted to
 * longs due to invalid syntax or overflow, then a ConfigException
 * is thrown with <code>reason</code> set to <code>FORMAT</code>.
 * @param key The key to use to search for the configuration entries.
 * @param count The number of entries expected in the result.  If the number of
 * retrieved entries does not match <code>count</code> and <code>count</code>
 * is not <code> GET_ALL</code> then a <code>ConfigException</code> error   is
 * thrown.
 * @return  An array of longs containing the list of long values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 * @see Config#GET_ALL
 */
  private final long[] getLongsInternal(String  key, int count)
        throws ConfigException {

    Object  obj;
    try {
      obj = get(key);
    }
    catch (KeywordValueException e) {
      throw new ConfigException(e.getMessage());
    }
    if (obj == null) {
      throw new ConfigException(ConfigException.NOT_FOUND,
                                "Key \"" + key
                                + "\" not found in configuration.");
    }
    long[] la = null;
    if (obj.getClass().isArray()) {
      int len = Array.getLength(obj);
      la = new long[len];
      for (int i=0; i<len; i++) {
        try {
          la[i] = (long) Long.parseLong(Array.get(obj,i).toString());
        }
        catch (Throwable  e) {
          throw new ConfigException("Element " + i +
                                    " is not a long.");
        }
      }
    }
    else {
      la = new long[1];
      try {
        la[0] = Long.parseLong(obj.toString());
      }
      catch (Throwable  e) {
        throw new ConfigException("Element 0 is not a long.");
      }
    }
    if ((count != GET_ALL) && (la.length != count)) {
      throw new ConfigException(ConfigException.COUNT,
                                "Key \"" + key
                                + "\" has " + la.length + " elements. (expected "
                                + count + ")");
    }
    return la;
  }

/**
 * Returns a single long integer value associated with a given key.
 * If the key is associated with more than one element, or the
 * retrieved element cannot be converted to a long integer then a
 * <code>ConfigException</code> exception is thrown.
 * @param key The key to use to search for the configuration entry.
 * @return The long integer value associated with the given key.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public long getLong(String  key) throws ConfigException
  {
    return (getLongsInternal(key, 1))[0];
  }

/**
 * Returns a single long integer value associated with a given key.
 * If the key is associated with more than one element then a
 * <code>ConfigException</code> error is thrown with <code>reason</code>
 * set to <code>COUNT</code>.  If the retrieved element cannot be
 * converted to a long integer then a <code>ConfigException</code> error
 * is thrown with <code>reason</code> set to <code>FORMAT</code>.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry
 * does not exist.
 * @return The long integer value associated with the given key.
 * @exception ConfigException Thrown if there was not exactly one   requested
 * element, or if the element is of the wrong   data type or format.
 * @see ConfigException
 */
  public long getLong(String  key, long defaultValue) throws ConfigException {
    try {
      return (getLongsInternal(key, 1))[0];
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }

/**
 * Returns all long integer values associated with a given key.
 * If any of the elements associated with the key cannot be converted
 * to a long integer then a <code>ConfigException</code> error is thrown.
 * @param key The key to use to search for the configuration entry.
 * @return  An array of longs containing the list of long values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public long[] getLongs(String  key) throws ConfigException
  {
    return getLongsInternal(key, GET_ALL);
  }

/**
 * Returns all long integer values associated with a given key.
 * If any of the elements associated with the key cannot be converted
 * to a long integer then a <code>ConfigException</code> error is thrown.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry
 * does not exist.
 * @return  An array of longs containing the list of long values from the
 * configuration input stream.
 * @exception ConfigException Thrown if the requested entries are of the wrong
 * data type or format.
 * @see ConfigException
 */
  public long[] getLongs(String  key, long[] defaultValue)
        throws ConfigException
  {
    try {
      return getLongsInternal(key, GET_ALL);
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }

/**
 * Returns the array of integers associated with a given key.  If the
 * <code>count</code> parameter is not <code>GET_ALL</code> and the
 * number of elements in the retrieved array does not match <code>
 * count</code> then a ConfigException is thrown with <code>reason</code>
 * set to <code>COUNT</code>.
 * If any of the retrieved elements cannot be converted to
 * integers due to invalid syntax or overflow, then a ConfigException
 * is thrown with <code>reason</code> set to <code>FORMAT</code>.
 * @param key The key to use to search for the configuration entries.
 * @param count The number of entries expected in the result.  If   the number of
 * retrieved entries does not match <code>count</code> and <code>count</code> is
 * not <code>   GET_ALL</code> then a <code>ConfigException</code> error is thrown.
 * @return  An array of integers containing the list of integer values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entry does not exist or
 * elements are not in the requested format.
 * @see ConfigException
 * @see Config#GET_ALL
 */
  private final int[] getIntsInternal(String  key, int count)
        throws ConfigException
  {
    Object  obj;
    try {
      obj = get(key);
    }
    catch (KeywordValueException e) {
      throw new ConfigException(e.getMessage());
    }
    if (obj == null) {
      throw new ConfigException(ConfigException.NOT_FOUND,
                                "Key \"" + key
                                + "\" not found in configuration.");
    }
    int[] ia = null;
    if (obj.getClass().isArray()) {
      int len = Array.getLength(obj);
      ia = new int[len];
      for (int i=0; i<len; i++) {
        try {
          ia[i] = (int)Integer.parseInt(Array.get(obj,i).toString());
        }
        catch (Throwable  e) {
          throw new ConfigException("Element " + i +
                                    " is not an integer.");
        }
      }
    }
    else {
      ia = new int[1];
      try {
        ia[0] = Integer.parseInt(obj.toString());
      }
      catch (Throwable  e) {
        throw new ConfigException("Element 0 is not an integer.");
      }
    }
    if ((count != GET_ALL) && (ia.length != count)) {
      throw new ConfigException(ConfigException.COUNT,
                                "Key \"" + key
                                + "\" has " + ia.length + " elements. (expected "
                                + count + ")");
    }
    return ia;
  }

/**
 * Returns a single integer value associated with a given key.
 * If the key is associated with more than one element, or the
 * retrieved element cannot be converted to a integer then a
 * <code>ConfigException</code> exception is thrown.
 * @param key The key to use to search for the configuration entry.
 * @return The integer value associated with the given key.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public int getInt(String  key) throws ConfigException
  {
    return (getIntsInternal(key, 1))[0];
  }

/**
 * Returns a single integer value associated with a given key.
 * If the key is associated with more than one element, or the
 * retrieved element cannot be converted to a integer then a
 * <code>ConfigException</code> exception is thrown.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry
 * does not exist.
 * @return The integer value associated with the given key.
 * @exception ConfigException Thrown if there was not exactly one
 * requested element, or if the element is of the wrong data type or format.
 * @see ConfigException
 */
  public int getInt(String  key, int defaultValue) throws ConfigException
  {
    try {
      return (getIntsInternal(key, 1))[0];
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }

/**
 * Returns all integer values associated with a given key.
 * If any of the elements associated with the key cannot be converted
 * to a integer then a <code>ConfigException</code> error is thrown.
 * @param key The key to use to search for the configuration entry.
 * @return  An array of integers containing the list of integer values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public int[] getInts(String  key) throws ConfigException
  {
    return getIntsInternal(key, GET_ALL);
  }

/**
 * Returns all integer values associated with a given key.
 * If any of the elements associated with the key cannot be converted
 * to a integer then a <code>ConfigException</code> error is thrown.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry
 * does not exist.
 * @return  An array of integers containing the list of integer values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entries are of
 * the wrong data type or format.
 * @see ConfigException
 */
  public int[] getInts(String  key, int[] defaultValue) throws ConfigException
  {
    try {
      return getIntsInternal(key, GET_ALL);
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }

/**
 * Returns the array of strings associated with a given key.  If the
 * <code>count</code> parameter is not <code>GET_ALL</code> and the
 * number of elements in the retrieved array does not match <code>
 * count</code> then a ConfigException is thrown with <code>reason</code>
 * set to <code>COUNT</code>.
 * @param key The key to use to search for the configuration entries.
 * @param count The number of entries expected in the result. If the number of
 * retrieved entries does not match <code>count</code> and <code>count</code> is
 * not <code>   GET_ALL</code> then a <code>ConfigException</code> error is thrown.
 * @return  An array of Strings containing the list of String values from   the
 * configuration input stream.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 * @see Config#GET_ALL
 */
  private final String [] getStringsInternal(String  key, int count)
        throws ConfigException
  {
    Object  obj;
    try {
      obj = get(key);
    }
    catch (KeywordValueException e) {
      throw new ConfigException(e.getMessage());
    }
    if (obj == null) {
      throw new ConfigException(ConfigException.NOT_FOUND,
                                "Key \"" + key
                                + "\" not found in configuration.");
    }
    String [] sa = null;
    if (obj.getClass().isArray()) {
      int len = Array.getLength(obj);
      sa = new String [len];
      for (int i=0; i<len; i++) {
        try {
          sa[i] = Array.get(obj,i).toString();
        }
        catch (Throwable  e) {
          throw new ConfigException("Element " + i +
                                    " is not a String.");
        }
      }
    }
    else {
      sa = new String [1];
      try {
        sa[0] = obj.toString();
      }
      catch (Throwable  e) {
        throw new ConfigException("Element 0 is not a String.");
      }
    }
    if ((count != GET_ALL) && (sa.length != count)) {
      throw new ConfigException(ConfigException.COUNT,
                                "Key \"" + key
                                + "\" has " + sa.length + " elements. (expected "
                                + count + ")");
    }
    return sa;
  }

/**
 * Returns a single String value associated with a given key.
 * If the key is associated with more than one element, or the
 * retrieved element cannot be converted to a String then a
 * <code>ConfigException</code> exception is thrown.
 * @param key The key to use to search for the configuration entry.
 * @return The string value associated with the given key.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public String  getString(String  key)
    throws ConfigException
  {
    return (getStringsInternal(key, 1))[0];
  }

/**
 * Returns a single String value associated with a given key.
 * If the key is associated with more than one element, or the
 * retrieved element cannot be converted to a String then a
 * <code>ConfigException</code> exception is thrown.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry
 * does not exist.
 * @return The string value associated with the given key.
 * @exception ConfigException Thrown if there was not exactly one
 * requested element, or if the element is of the wrong data type or format.
 * @see ConfigException
 */
  public String  getString(String  key, String  defaultValue)
        throws ConfigException
  {
    try {
      return (getStringsInternal(key, 1))[0];
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }

/**
 * Returns all String values associated with a given key.
 * If any of the elements associated with the key cannot be converted
 * to a String then a <code>ConfigException</code> error is thrown.
 * @param key The key to use to search for the configuration entry.
 * @return  An array of strings containing the list of string values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public String [] getStrings(String  key) throws ConfigException
  {
    return getStringsInternal(key, GET_ALL);
  }

/**
 * Returns all String values associated with a given key.
 * If any of the elements associated with the key cannot be converted
 * to a String then a <code>ConfigException</code> error is thrown.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry
 * does not exist.
 * @return  An array of strings containing the list of string values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entries are of
 * the wrong data type or format.
 * @see ConfigException
 */
  public String [] getStrings(String  key, String [] defaultValue)
        throws ConfigException
  {
    try {
      return getStringsInternal(key, GET_ALL);
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }

/**
 * Returns the array of booleans associated with a given key.  If the
 * <code>count</code> parameter is not <code>GET_ALL</code> and the
 * number of elements in the retrieved array does not match <code>
 * count</code> then a ConfigException is thrown with <code>reason</code>
 * set to <code>COUNT</code>.
 * If any of the retrieved elements cannot be converted to
 * booleans due to invalid syntax or overflow, then a ConfigException
 * is thrown with <code>reason</code> set to <code>FORMAT</code>.
 * A boolean value is represented in a case independent manner as
 * either the string <code>true</code> or <code>false</code>.
 * @param key The key to use to search for the configuration entries.
 * @param count The number of entries expected in the result. If the number of
 * retrieved entries does not match <code>count</code> and <code>count</code> is
 * not <code>   GET_ALL</code> then a <code>ConfigException</code> error is thrown.
 * @return  An array of booleans containing the list of boolean values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 * @see Config#GET_ALL
 */
  private final boolean[] getBooleansInternal(String  key, int count)
        throws ConfigException
  {
    Object  obj;
    try {
      obj = get(key);
    }
    catch (KeywordValueException e) {
      throw new ConfigException(e.getMessage());
    }
    if (obj == null) {
      throw new ConfigException(ConfigException.NOT_FOUND,
                                "Key \"" + key
                                + "\" not found in configuration.");
    }
    boolean[] ba = null;
    if (obj.getClass().isArray()) {
      int len = Array.getLength(obj);
      ba = new boolean[len];
      for (int i=0; i<len; i++) {
        try {
          ba[i] = Boolean.valueOf(
          Array.get(obj,i).toString().toLowerCase()).booleanValue();
        } catch (Throwable  e) {
          throw new ConfigException("Element " + i +
                                    " is not a boolean.");
        }
      }
    }
    else {
      ba = new boolean[1];
      try {
        ba[0] =
            Boolean.valueOf(obj.toString().toLowerCase()).booleanValue();
      }
      catch (Throwable  e) {
        throw new ConfigException("Element 0 is not a boolean.");
      }
    }
    if ((count != GET_ALL) && (ba.length != count)) {
      throw new ConfigException(ConfigException.COUNT,
                                "Key \"" + key
                                + "\" has " + ba.length + " elements. (expected "
                                + count + ")");
    }
    return ba;
  }

/**
 * Returns a single boolean value associated with a given key.
 * If the key is associated with more than one element, or the
 * retrieved element cannot be converted to a boolean then a
 * <code>ConfigException</code> exception is thrown.
 * @param key The key to use to search for the configuration entry.
 * @return The boolean value associated with the given key.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public boolean getBoolean(String  key) throws ConfigException
  {
    return (getBooleansInternal(key, 1))[0];
  }

/**
 * Returns a single boolean value associated with a given key.
 * If the key is associated with more than one element, or the
 * retrieved element cannot be converted to a boolean then a
 * <code>ConfigException</code> exception is thrown.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry
 * does not exist.
 * @return The boolean value associated with the given key.
 * @exception ConfigException Thrown if there was not exactly one   requested
 * element, or if the element is of the wrong   data type or format.
 * @see ConfigException
 */
  public boolean getBoolean(String  key, boolean defaultValue)
        throws ConfigException
  {
    try {
      return(getBooleansInternal(key, 1)[0]);
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }

/**
 * Returns all boolean values associated with a given key.
 * If any of the elements associated with the key cannot be converted
 * to a boolean then a <code>ConfigException</code> error is thrown.
 * @param key The key to use to search for the configuration entry.
 * @return  An array of booleans containing the list of boolean values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public boolean[] getBooleans(String  key) throws ConfigException
  {
    return getBooleansInternal(key, GET_ALL);
  }

/**
 * Returns all boolean values associated with a given key.
 * If any of the elements associated with the key cannot be converted
 * to a boolean then a <code>ConfigException</code> error is thrown.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry does not
 * exist.
 * @return  An array of booleans containing the list of boolean values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entries are of
 * the wrong data type or format.
 * @see ConfigException
 */
  public boolean[] getBooleans(String  key, boolean[] defaultValue)
        throws ConfigException
  {
    try {
      return getBooleansInternal(key, GET_ALL);
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }

/**
 * Returns the array of doubles associated with a given key.  If the
 * <code>count</code> parameter is not <code>GET_ALL</code> and the
 * number of elements in the retrieved array does not match <code>
 * count</code> then a ConfigException is thrown with <code>reason</code>
 * set to <code>COUNT</code>. If any of the retrieved elements cannot be
 * converted to doubles due to invalid syntax or overflow, then a
 * ConfigException is thrown with <code>reason</code> set to
 * <code>FORMAT</code>.
 * @param key The key to use to search for the configuration entries.
 * @param count The number of entries expected in the result. If    the number of
 * retrieved entries does not match <code>count</code> and <code>count</code> is
 * not <code>   GET_ALL</code> then a <code>ConfigException</code> error is thrown.
 * @return  An array of doubles containing the list of double values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 * @see Config#GET_ALL
 */
  private final double[] getDoublesInternal(String  key, int count)
        throws ConfigException
  {
    Object  obj;
    try {
      obj = get(key);
    }
    catch (KeywordValueException e) {
      throw new ConfigException(e.getMessage());
    }
    if (obj == null) {
      throw new ConfigException(ConfigException.NOT_FOUND,
                                "Key \"" + key
                                + "\" not found in configuration.");
    }
    double[] da = null;
    if (obj.getClass().isArray()) {
      int len = Array.getLength(obj);
      da = new double[len];
      for (int i=0; i<len; i++) {
        try {
          da[i] = Double.valueOf(
          Array.get(obj,i).toString()).doubleValue();
        }
        catch (Throwable  e) {
          throw new ConfigException("Element " + i +
                                    " is not a double.");
        }
      }
    }
    else {
      da = new double[1];
      try {
        da[0] = Double.valueOf(obj.toString()).doubleValue();
      }
      catch (Throwable  e) {
        throw new ConfigException("Element 0 is not a long.");
      }
    }
    if ((count != GET_ALL) && (da.length != count)) {
      throw new ConfigException(ConfigException.COUNT,
                                "Key \"" + key
                                + "\" has " + da.length + " elements. (expected "
                                + count + ")");
    }
    return da;
  }

/**
 * Returns a single double value associated with a given key. If the key is
 * associated with more than one element, or the retrieved element cannot be
 * converted to a double then a <code>ConfigException</code> exception is thrown.
 * @param key The key to use to search for the configuration entry.
 * @return The double value associated with the given key.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public double getDouble(String  key) throws ConfigException
  {
    return (getDoublesInternal(key, 1))[0];
  }

/**
 * Returns a single double value associated with a given key.
 * If the key is associated with more than one element, or the
 * retrieved element cannot be converted to a double then a
 * <code>ConfigException</code> exception is thrown.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry
 * does not exist.
 * @return The double value associated with the given key.
 * @exception ConfigException Thrown if there was not exactly one   requested
 * element, or if the element is of the wrong   data type or format.
 * @see ConfigException
 */
  public double getDouble(String  key, double defaultValue)
        throws ConfigException
  {
    try {
      return(getDoublesInternal(key, 1)[0]);
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }

/**
 * Returns all double values associated with a given key.
 * If any of the elements associated with the key cannot be converted
 * to a double then a <code>ConfigException</code> error is thrown.
 * @param key The key to use to search for the configuration entry.
 * @return  An array of doubles containing the list of double values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public double[] getDoubles(String  key) throws ConfigException
  {
    return getDoublesInternal(key, GET_ALL);
  }

/**
 * Returns all double values associated with a given key.
 * If any of the elements associated with the key cannot be converted
 * to a double then a <code>ConfigException</code> error is thrown.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry
 * does not exist.
 * @return  An array of doubles containing the list of double values from
 * the configuration input stream.
 * @exception ConfigException Thrown if the requested entries are of
 * the wrong data type or format.
 * @see ConfigException
 */
  public double[] getDoubles(String  key, double[] defaultValue)
        throws ConfigException
  {
    try {
      return getDoublesInternal(key, GET_ALL);
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }
/**
 * Returns a DataSource value associated with a given key.
 * @param key The key to use to search for the configuration entry.
 * @return The DataSource value associated with the given key.
 * @exception ConfigException Thrown if the requested entry does not exist
 * or elements are not in the requested format.
 * @see ConfigException
 */
  public Object  getDataSource(String  key) throws ConfigException {
    Object  obj;
    try {
      obj = get(key);
    }
    catch (KeywordValueException e) {
      throw new ConfigException(e.getMessage());
    }
    if (obj == null) {
      throw new ConfigException(ConfigException.NOT_FOUND,
                                "Key \"" + key
                                + "\" not found in configuration.");
    }
    Object  ds = null;
    if (obj.getClass().isArray()) {
      throw new ConfigException("The value is an array. The method returns a single DataStruct Object.");
    }
    else {

      try {
        ds = obj;
      }
      catch (Throwable  e) {
        throw new ConfigException("Element is not a DataSource.");
      }
    }
    return ds;
  }

/**
 * Returns a single DataSource value associated with a given key.
 * If the key is associated with more than one element, or the
 * retrieved element cannot be converted to a String then a
 * <code>ConfigException</code> exception is thrown.
 * @param key The key to use to search for the configuration entry.
 * @param defaultValue The default value to use if the requested entry
 * does not exist.
 * @return The DataSource value associated with the given key.
 * @exception ConfigException Thrown if there was not exactly one
 * requested element, or if the element is of the wrong data type or format.
 * @see ConfigException
 */
  public Object  getDataSource(String  key, Object  defaultValue)
        throws ConfigException {
    try {
      return getDataSource(key);
    }
    catch (ConfigException e) {
      if (e.reason != e.NOT_FOUND) {
        throw e;
      }
      return defaultValue;
    }
  }


}

