
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: StandardLogger.java,v 1.2 2005/03/24 10:51:25 slobodan Exp $
 */
package com.lutris.logging;

import java.io.File ;
import java.io.FileInputStream ;
import java.io.FileOutputStream ;
import java.io.IOException ;
import java.io.PrintWriter ;
import java.util.Hashtable ;

import com.lutris.util.Config;
import com.lutris.util.ConfigException;
import com.lutris.util.ConfigFile;
import com.lutris.util.KeywordValueException;

/**
 * Standard implementation of the <CODE>Logger</CODE>.  This is 
 * general-purpose logging facility.  A client that needs additional
 * functionality can either extend this class or provide there own
 * implementationm of <CODE>Logger</CODE>. <P>
 * 
 * Currently this is a bare-bones class that writes INFO and above
 * levels to stderr and all others to a log file.
 *
 * @author Mark Diekhans
 * @see com.lutris.logging.Logger
 * @see com.lutris.logging.LogChannel
 * @see com.lutris.logging.StandardLogChannel
 */
public class StandardLogger extends Logger {
    
    private static final String  LOG_SECTION = "Server";
    private static final String  LOG_FILE = "LogFile";
    private static final String  LOG_TO_FILE = "LogToFile";
    private static final String  LOG_TO_STDERR = "LogToStderr";
    
    /**
     * Table of level names to level numbers.  While level configuration
     * is local to a facility, a global table is kept assigning numbers
     * to each level name.
     */
    private Hashtable  levelNumbers = new Hashtable ();

    /**
     * Table translating level number to name and the largest entry in the
     * array that is valid.  Will be expanded if needed.
     */
    protected String [] levelNames = new String [MAX_STD_LEVEL * 2];
    protected int numLevels = 0;

    /**
     * Table of levels that are to be enabled.
     * Accessed directly by the channel.  If null, ignored.
     */
    protected boolean[] enabledLevelFlags = null;

    /**
     * Table of levels that are to be written to the log file.
     * Accessed directly by the channel.  If null, ignored.
     */
    protected boolean[] logFileLevelFlags = null;

    /**
     * Table of levels that are to be written to stderr
     * Accessed directly by the channel.  If null, then
     * the default behavior of writing serious standard
     * levels to stderr is in affect.
     */
    protected boolean[] stderrLevelFlags = null;

    /**
     * Log file name.
     */
    File  activeLogFile;

    /**
     * Log file writter.  Use directly by channels.
     */
    PrintWriter  logFileStream;

    /**
     * Stderr writter.  Use directly by channels.
     */
    PrintWriter  stderrStream;

    /**
     * Table of <CODE>StandardLogChannel<CODE> objects, indexed by facility
     * name.
     */
    private Hashtable  logChannels = new Hashtable ();

    /**
     * Construct a new logger.  Configuration is not done now, to allow
     * the logger to be created very early.
     *
     * @param makeCentral Make this object the central logging object.
     */
    public StandardLogger(boolean makeCentral) {
        int level;

        for (level = 0; level <= MAX_STD_LEVEL; level++) {
            String  name = standardLevelNames[level];

            levelNumbers.put(name, new Integer (level));
            levelNames[level] = name;
        }
        numLevels = level;
        if (makeCentral) {
            centralLogger = this;
        }
    }

    /**
     * Get maximum level number in a set of level names.
     *
     * @param levels String names of levels.
     * @return The maximum level number
     */
    private int getMaxLevel(String [] levels) {
        int levelNum;
        int maxLevelNum = 0;

        for (int idx = 0; idx < levels.length; idx++) {
            levelNum = getLevel(levels[idx]);
            if (levelNum > maxLevelNum) {
                maxLevelNum = levelNum;
            }
        }
        return maxLevelNum;
    }

    /**
     * Generate a boolean array for all of the listed levels, indicating
     * if they are enabled.
     *
     * @param levels String names of levels.
     * @param maxLevelNum Size to make the array.
     */
    private boolean[] getLevelStateArray(String [] levels, int maxLevelNum) {
        int levelNum;
        // Initialize the stated.
 boolean[] levelNums = new boolean[maxLevelNum + 1];

        for (int idx = 0; idx < levels.length; idx++) {
            levelNums[getLevel(levels[idx])] = true;
        }
        return levelNums;
    }

    /**
     * Switch a log file; replacing the old one with a new one.
     *
     *
     * @param logFile The new log file.
     * @return The File object for the previous log file, or null if there
     *  wasn't one.
     * @exception java.io.IOException If an error occurs opening the log file.
     */
    public synchronized File  switchLogFile(File  logFile)
        throws java.io.IOException  {
        PrintWriter  oldLogFileStream = logFileStream;
        File  oldActiveLogFile = activeLogFile;
        // Append output stream without auto-flush (we do it explictly).
 FileOutputStream  out = new FileOutputStream (logFile.getPath(),
                true);

        logFileStream = new PrintWriter (out, false);
        activeLogFile = logFile;
        // Close old, if it exists.  Waiting for any accessors.
 if (oldLogFileStream != null) {
            synchronized (oldLogFileStream) {
                oldLogFileStream.close();
            }
        }
        return oldActiveLogFile;
    }

    /**
     * Configure the logger. All current configuration is discarded.
     * This is a simplistic initial implementation that just allows
     * directing to a single log file or stderr on a level basis.
     * A more complete interface will be provided in the future.
     *
     * @param logFile The log file to write to.  
     * @param fileLevels List of levels that will be written to the file.
     * @param stderrLevels List of levels that will be written to stderr.
     *  The same level may appear in both lists.
     * @exception java.io.IOException If an error occurs opening the log file.
     */
    public synchronized void configure(File  logFile,
            String [] fileLevels,
            String [] stderrLevels)
        throws java.io.IOException  {
        // Ensure that the directory exists.
 if (logFile.getParent() != null) {
            new File (logFile.getParent()).mkdirs();
        }
        // Output streams without auto-flush (we do it explictly).
        switchLogFile(logFile);
        stderrStream = new PrintWriter (System.err, false);

        /*
         * Tables must be created after streams, as they are 
         * the checked before accessing the stream.  Care is taken
         * that all three arrays that are indexed by level are of
         * the same size and never shrink on reconfigure.  This 
         * mean no synchronization is required to access them.  Also
         * enabled table must be done last.
         */
        int maxLevelNum;
        int levelNum;

        if (enabledLevelFlags != null) {
            maxLevelNum = enabledLevelFlags.length - 1;
        } else {
            maxLevelNum = MAX_STD_LEVEL;
        }
        levelNum = getMaxLevel(fileLevels);
        if (levelNum > maxLevelNum) {
            maxLevelNum = levelNum;
        }
        levelNum = getMaxLevel(stderrLevels);
        if (levelNum > maxLevelNum) {
            maxLevelNum = levelNum;
        }
        // Build boolean tables.
        logFileLevelFlags = getLevelStateArray(fileLevels, maxLevelNum);
        stderrLevelFlags = getLevelStateArray(stderrLevels, maxLevelNum);
        enabledLevelFlags = new boolean[maxLevelNum + 1];
        for (int idx = 0; idx < logFileLevelFlags.length; idx++) {
            if (logFileLevelFlags[idx]) {
                enabledLevelFlags[idx] = true;
            }
        }
        for (int idx = 0; idx < stderrLevelFlags.length; idx++) {
            if (stderrLevelFlags[idx]) {
                enabledLevelFlags[idx] = true;
            }
        }
    }

    /**
     * Create a log channel.
     */
    private synchronized StandardLogChannel createChannel(String  facility) {
        StandardLogChannel channel
                = (StandardLogChannel) logChannels.get(facility);

        if (channel == null) {
            channel = new StandardLogChannel(facility, this);
            logChannels.put(facility, channel);
        }
        return channel;
    }

    /**
     * Get the log channel object for a facility.  For a given facility,
     * the same object is always returned.
     * 
     * @param facility Facility the channel is associated with.
     */
    public LogChannel getChannel(String  facility) {
        StandardLogChannel channel
                = (StandardLogChannel) logChannels.get(facility);

        if (channel == null) {
            // Slow path, synchronized
            channel = createChannel(facility);
        }
        return channel;
    }

    /**
     * Create a log level.
     */
    private synchronized Integer  createLevel(String  level) {
        Integer  intLevel = (Integer ) levelNumbers.get(level);

        if (intLevel == null) {
            intLevel = new Integer (numLevels);
            levelNames[numLevels] = level;
            levelNumbers.put(level, intLevel);
            numLevels++;
        }
        return intLevel;
    }

    /**
     * Convert a symbolic level to an integer identifier,
     * creating it if it doesn't exist
     *
     * @param level Symbolic level that is to be checked.
     * @return The numeric level identifier
     */
    public synchronized int getLevel(String  level) {
        Integer  intLevel = (Integer ) levelNumbers.get(level);

        if (intLevel == null) {
            // Slow path, synchronized
            intLevel = createLevel(level);
        }
        return intLevel.intValue();
    }

    /**
     * Convert an int to a symbolic level name.
     *
     * @param level an int level.
     * @return The String symolic level name or null if there is not one.
     */
    public String  getLevelName(int level) {
        if ((level >= 0) && (level < numLevels)) {
            return levelNames[level];
        } else {
            return null;
        }
    }
    
    /**
     * Configure Logger with given config file, interpreting of config file is 
     * logger implementation specific.
     *
     * @param confFilePath Path to configuration file.
     */
    public void configure(String  confFilePath) throws ConfigException {
        try {
            FileInputStream  configFIS = new FileInputStream (confFilePath);         
            ConfigFile cFile = new ConfigFile(configFIS);                 
            Config config = cFile.getConfig();

            configFIS.close();                                                 
            Config logConfig = (Config) config.getSection(LOG_SECTION);

            configure(logConfig);
        } catch (KeywordValueException kve) {
            throw new ConfigException("Error parsing configuration for logger.",
                    kve);
        } catch (IOException  ioe) {
            throw new ConfigException("Error configuring logger.", ioe);
        }
    }

    /**
     * Configure Logger with given config section
     *
     * @param logConfig containing parameters for configuring logger
     */
    public void configure(Config logConfig) throws ConfigException {
        if (logConfig == null) {
            throw new ConfigException("Cannot configure logger. Config is null.");
        } else {
            String  logFile = null;

            logFile = logConfig.getString(LOG_FILE);
            String [] toFile = null;

            toFile = logConfig.getStrings(LOG_TO_FILE);
            String [] toStderr = null;

            toStderr = logConfig.getStrings(LOG_TO_STDERR);
            File  theLogFile = new File (logFile);

            try {
                configure(theLogFile, toFile, toStderr);
            } catch (IOException  ioe) {
                throw new ConfigException("Error configuring logger.",
                        ioe);
            }
        }
    }
}
