

/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: StandardLogChannel.java,v 1.2 2005/03/24 10:51:25 slobodan Exp $
 */

package com.lutris.logging;

import java.io.PrintWriter ;
import java.io.StringWriter ;
import java.text.DateFormat ;
import java.text.FieldPosition ;
import java.text.SimpleDateFormat ;
import java.util.Date ;

/**
 * Standard implementation of a channel associated with a logging
 * facility.  All messages for the facility are written using a channel.
 * Care is take to avoid synchronization when possible for performance
 * reasons.
 *
 * @author Mark Diekhans
 * @see com.lutris.logging.LogChannel
 * @see com.lutris.logging.StandardLogger
 */
public class StandardLogChannel implements LogChannel {
    /**
     * Our symbolic name.
     */
    private String  facility;

    /**
     * Format for the date.
     */
    private static final SimpleDateFormat  dateFormatter =
        new SimpleDateFormat ("yyyy.MM.dd HH:mm:ss");

    /**
     * Logger object with which we are associated.
     */
    private StandardLogger logger;


    /**
     * Construct a new log channel.
     *
     * @param chanFacility The facility that the channel is associate with.
     * @param loggerObj The logging object that this channel will be associated
     *  with.
     */
    protected StandardLogChannel(String  chanFacility,
                                 StandardLogger loggerObj) {
        facility = chanFacility;
        logger = loggerObj;

    }

    /**
     * @see LogChannel.getLevel
     */
    public int getLevel(String  level) {
        return logger.getLevel(level);
    }

    /**
     * @see LogChannel.getLogWriter#String
     */
    public LogWriter getLogWriter(String  level) {
        return new LogWriter(this, level);
    }

    /**
     * @see LogChannel.getLogWriter#int
     */
    public LogWriter getLogWriter(int level) {
        return new LogWriter(this, level);
    }

    /**
     * @see LogChannel.isEnabled
     */
    public boolean isEnabled(int level) {
        // Copy to avoid needing to be synchronized.
 boolean[] enabledLevelFlags = logger.enabledLevelFlags;
        return (enabledLevelFlags != null)
            && (level >= 0 && level < enabledLevelFlags.length)
            && enabledLevelFlags[level];
    }

    /**
     * @see LogChannel.isEnabled
     */
    public boolean isEnabled(String  level) {
        return isEnabled(logger.getLevel(level));
    }

    /**
     * Do the work of writting a log message.  It should already be
     * determined if the channel is enabled; but limited synchronization
     * is required, as we ignore any error while writing.  Handles
     * multiline messages.
     *
     *
     * @param out Print output stream to write to.
     * @param date The date to format into the message
     * @param level Level string for message.
     */
    private void doWrite(PrintWriter  out, Date  date, int level, String  msg) {
        try {
            // Create prefix for each line.
 StringBuffer  msgBuf = new StringBuffer ();
            dateFormatter.format(date, msgBuf,
                                 new FieldPosition (DateFormat.YEAR_FIELD));
            msgBuf.append(": ");
            msgBuf.append(facility);
            msgBuf.append(',');

            msgBuf.append(logger.levelNames[level]);

            // Write as individual lines with the same prefix.
 int nextIdx = 0;
            int newlineIdx;

            synchronized (out) {
                while (nextIdx < msg.length()) {
                    out.print(msgBuf);
                    if (nextIdx == 0) {
                        out.print(": ");
                    } else {
                        out.print("+ ");
                    }
                    newlineIdx = msg.indexOf('\n', nextIdx);
                    if (newlineIdx < 0) {
                        newlineIdx = msg.length();
                    }
                    out.write(msg, nextIdx, (newlineIdx - nextIdx));
                    out.println();
                    nextIdx = newlineIdx + 1;
                }
                out.flush();
            }
        } catch (Throwable  ignore) {
            // FIXME: Delete eventually or maybe make a fatal exception??
            System.err.println("StandardLogChannel.doWrite ignored exception:");
            ignore.printStackTrace();
            // All errors, including runtime ones, are ignored.
        }
    }


    /**
     * @see LogChannel.write
     */
    public void write(int level, String  msg) {
        if (isEnabled(level)) {
            Date  date = new Date ();
            if (logger.logFileLevelFlags[level]) {
                doWrite(logger.logFileStream, date, level, msg);
            }
            if (logger.stderrLevelFlags[level]) {
                doWrite(logger.stderrStream, date, level, msg);
            }
        }
    }

    /**
     * @see LogChannel.write
     */
    public synchronized void write(String  level, String  msg) {
        write(getLevel(level), msg);
    }

    /**
     * @see LogChannel.write
     */
    public synchronized void write(int level, String  msg, Throwable  throwable) {
        if (isEnabled(level)) {
            Date  date = new Date ();
            StringWriter  stackBuf = new StringWriter ();
            throwable.printStackTrace(new PrintWriter (stackBuf));
            stackBuf.flush();

            String  errMsg = msg + ":" + " " + throwable.getMessage() + '\n'
                + stackBuf;

            if (logger.logFileLevelFlags[level]) {
                doWrite(logger.logFileStream, date, level, errMsg);
            }
            if (logger.stderrLevelFlags[level]) {
                doWrite(logger.stderrStream, date, level, errMsg);
            }
        }
    }

    /**
     * @see LogChannel.write
     */
    public synchronized void write(String  level, String  msg, Throwable  throwable) {
        write(getLevel(level), msg, throwable);
    }
}
