
package org.enhydra.xml;

import java.util.ArrayList ;
import java.util.List ;

import org.w3c.dom.DOMException ;
import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.w3c.dom.Node ;
import org.w3c.dom.NodeList ;

/**
 * @author Tweety
 *
 * A class representing a node in a meta-data tree, which implements
 * the <a HREF="../../../../api/org/w3c/dom/Element.html">
 *
 * <p> Namespaces are ignored in this implementation.  The terms "tag
 * name" and "node name" are always considered to be synonymous.
 *
 * @version 1.0
 */
public class SearchElement extends HashMapElement {

    public static String  TAG_SEPARATOR = "/";
    public static String  ATTR_SEPARATOR = "@";
    public static String  EQUAL_SEPARATOR = "=";

    /**
     * Constructs an empty <code>SearchElement</code>.
     */
    public SearchElement() {
    }

    /**
     * Constructs an <code>SearchElement</code> with a given document owner and node name.
     *
     * @param ownerDoc the document owner of the node, as a <code>Document</code>.
     * @param name the name of the node, as a <code>String</code>.
     */
    public SearchElement(Document  ownerDoc, String  name) {
        super(ownerDoc, name);
    }

    /**
     * Constructs an <code>SearchElement</code> with a given <code>Node</code>.
     *
     * @param node <code>Node</code>.
     */
    public SearchElement(Node  node) {
        super(node);
    }

    /**
     * Constructs an <code>SearchElement</code> with a given <code>SearchElement</code>.
     *
     * @param node <code>SearchElement</code>.
     */
    public SearchElement(SearchElement node) {
        super((HashMapElement) node);
    }

    /**
     * Returns new <code>SearchElement</code> instance from a given <code>Node</code>.
     *
     * @param node <code>Node</code>.
     */
    protected Node  newElementInstance(Node  node) {
        return new SearchElement(node);
    }

  /**
     * Creates new instance of <code>SearchElement</code> from a given document
     * as a <code>Document</code>.
     *
     * @param document document.
     *
     * @return new <code>SearchElement</code> node as a root of the <code>Document</code>.
     */
    public static Element  newInstance(Document  document) {
        Node  root = document.getDocumentElement();
        return new SearchElement(root);
    }

    /**
     * Returns a list of elements in the subtree of this node, with the given tag name.
     *
     * @param namePath relative path to the </code>Element</code> (through children).
     *
     * @return list of elements in the subtree of this node, with the given tag name.
     */
    public NodeList  getSubElementsByTagName(String  namePath) {
        List  list = new ArrayList ();
        getSubElementsByTagName(namePath, list);
        return new NodeListImpl(list);
    }

    /*
     * Recursively fullfills the <code>list</code> with all the nodes on the given path.
     */
    private void getSubElementsByTagName(String  name, List  list) {
        String [] keys = this.split(name, this.TAG_SEPARATOR);
        if (keys.length == 1) {
            List  l = (List ) this.children.get(name);
            if (l != null)
                list.addAll(l);
            return;
        }
        NodeList  tagChildren = this.getChildrenByTagName(keys[0]);
        if (tagChildren != null)
            for (int i = 0; i < tagChildren.getLength(); i++)
                ((SearchElement) tagChildren.item(i)).getSubElementsByTagName(
                    keys[1],
                    list);
    }

    /**
     * Returns a list of <code>Element</code>s in the subtree of this node,
     * which contain attribute with the given name and value.
     *
     * @param attrPath relative path to the attribute name.
     * @param attrValue attribute value.
     *
     * @return list of <code>Element</code>s in the subtree of this node,
     * which contain attribute with the given name and value.
     */
    public NodeList  getSubElementsByAttrValue(String  attrPath, String  attrValue) {
        String [] keys = this.split(attrPath, this.ATTR_SEPARATOR);
        if (keys.length != 2)
            throw new DOMException (
                DOMException.INVALID_ACCESS_ERR,
                "Parameter not supported");
        List  list = new ArrayList ();
        getSubElementsByAttrValue(keys[0], keys[1], attrValue, list);
        return new NodeListImpl(list);
    }


    /*
     * Recursively fullfills the <code>list</code> with all the nodes in the given path.
     */
    private void getSubElementsByAttrValue(String  tagName, String  attrName, String  attrValue, List  list) {
        String [] keys = this.split(tagName, this.TAG_SEPARATOR);
        if (keys.length == 1) {
            List  fList = (List ) this.children.get(tagName);
            if (fList != null) {
                for (int i = 0; i < fList.size(); i++) {
                    Element  elm = (Element ) fList.get(i);
                    String  val = (String ) elm.getAttribute(attrName);
                    if (val != null)
                        if (val.equals(attrValue))
                            list.add(elm);
                }
            }
            return;
        }
        NodeList  tagChildren = this.getChildrenByTagName(keys[0]);
        if (tagChildren != null) {
            for (int i = 0; i < tagChildren.getLength(); i++)
                ((SearchElement) tagChildren.item(i)).getSubElementsByAttrValue(
                    keys[1],
                    attrName,
                    attrValue,
                    list);
        }
    }


    /**
     * Returns a list of <code>Element</code>s in the subtree of this node,
     * with the given tag name and value.
     *
     * @param tagPath relative path to the tag name.
     * @param tagValue <code>Element</code> value.
     *
     * @return list of <code>Element</code>s in the subtree of this node,
     * with the given tag name and value.
     */
    public NodeList  getSubElementsByTagText(String  tagPath, String  tagValue) {
        List  list = new ArrayList ();
        getSubElementsByTagText(tagPath, tagValue, list);
        return new NodeListImpl(list);
    }


    /*
     * Recursively fullfills the <code>list</code> with all the nodes in the given path.
     */
    private void getSubElementsByTagText(
        String  tagName,
        String  tagValue,
        List  list) {
        String [] keys = this.split(tagName, this.TAG_SEPARATOR);
        if (keys.length == 1) {
            List  fList = (List ) this.children.get(tagName);
            if (fList != null) {
                for (int i = 0; i < fList.size(); i++) {
                    HashMapElement elm = (HashMapElement) fList.get(i);
                    String  val = (String ) elm.getText();
                    if (val != null) {
                        if (val.equals(tagValue))
                            list.add(elm);
                    }
                    else if ("null".equals(tagValue)) {
                        list.add(elm);
                    }
                }
            }
            return;
        }
        NodeList  tagChildren = this.getChildrenByTagName(keys[0]);
        if (tagChildren != null) {
            for (int i = 0; i < tagChildren.getLength(); i++)
                ((SearchElement) tagChildren.item(i)).getSubElementsByTagText(
                    keys[1],
                    tagValue,
                    list);
        }
    }


    /**
     * Returns a list of <code>Element</code>s in the subtree of this node,
     * that satisfy the given condition.
     *
     * @param condition condition.
     *
     * @return list of <code>Element</code>s in the subtree of this node,
     * that satisfy the given condition.
     */
    public NodeList  getSubElementsByCondition(String  condition) {
        String [] keys = this.split(condition, this.EQUAL_SEPARATOR);
        String  namePath = keys[0];
        if (namePath.indexOf(ATTR_SEPARATOR) != -1)
            return getSubElementsByAttrValue(namePath, keys[1]);
        else
            return getSubElementsByTagText(namePath, keys[1]);
    }


    /**
     * Returns the first <code>Element</code> in the subtree of this node,
     * that satisfy the given condition.
     *
     * @param condition condition.
     *
     * @return the first <code>Element</code> in the subtree of this node,
     * that satisfy the given condition.
     */
    public Element  getFirstSubElementsByCondition(String  condition) {
        NodeList  nodes = getSubElementsByCondition(condition);
        if (nodes != null && nodes.getLength() > 0)
            return (Element ) nodes.item(0);
        return null;
    }


    /**
     * Returns the first <code>Element</code> in the subtree of this node,
     * with the given tag name.
     *
     * @param namePath relative path to the <code>Element</code>.
     *
     * @return the first <code>Element</code> in the subtree of this node,
     * with the given tag name.
     */
    public Element  getFirstSubElementByTagName(String  namePath) {
        NodeList  nodes = getSubElementsByTagName(namePath);
        if (nodes != null && nodes.getLength() > 0)
            return (Element ) nodes.item(0);
        return null;
    }


    /**
     * Return the text of the <code>Element</code> found on the given path.
     *
     * @param namePath relative path to the <code>Element</code> node.
     *
     * @return text of the <code>Element</code> found on the given path.
     */
    public String  getText(String  namePath) {
        NodeList  nodes = this.getSubElementsByTagName(namePath);
        if (nodes != null && nodes.getLength() > 0)
            return ((SearchElement) nodes.item(0)).getText();
        return null;
    }


    /**
     * Sets the given text to the <code>Element</code> found on the given path.
     *
     * @param namePath relative path to the <code>Element</code> node.
     * @param text new text.
     */
    public void setText(String  namePath, String  text) {
        NodeList  nodes = this.getSubElementsByTagName(namePath);
        if (nodes != null && nodes.getLength() > 0)
             ((SearchElement) nodes.item(0)).setText(text);
    }


    /**
     * Sets the value of an attribute found on the given path.
     *
     * @param namePath relative path to the attribute.
     * @param text new value.
     */
    public void setAttr(String  namePath, String  value) {
        String [] keys = this.split(namePath, this.ATTR_SEPARATOR);

        NodeList  nodes = this.getSubElementsByTagName(keys[0]);
        if (nodes != null && nodes.getLength() > 0)
             ((SearchElement) nodes.item(0)).setAttribute(keys[1], value);
    }

    private String[] split(String text, String separator) {
        String[] splitText = new String[2];
        int separatorIndex = text.indexOf(separator);
        if (separatorIndex == -1) {
            splitText = new String[]{text};
        } else {
            splitText[0] = text.substring(0, separatorIndex);
            splitText[1] = text.substring(separatorIndex + 1, text.length());
        }

        return splitText;
    }

}
