
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: MultiClassLoader.java,v 1.3 2005/03/24 10:51:25 slobodan Exp $
 *
 * formatted with JxBeauty (c) johann.langhofer@nextra.at
 */


package com.lutris.classloader;

import java.io.ByteArrayOutputStream ;
import java.io.File ;
import java.io.FileNotFoundException ;
import java.io.IOException ;
import java.io.InputStream ;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException ;
import java.net.URL ;
import java.util.Enumeration ;
import java.util.Hashtable ;
import java.util.StringTokenizer ;
import java.util.Vector ;

import com.lutris.logging.LogChannel;
import com.lutris.util.FatalExceptionError;

/**
 * <P><B>Summary:</B>
 * <P>A class loader that can load classes from
 * class files and zip files residing in a specified class path.
 *
 * <P>This class loader can also load resources that reside on the
 * class path and return them as is, as input streams, or as byte arrays.
 *
 * <P><B>Features:</B>
 * <UL>
 * <LI>Loads classes from class files and zip files.
 * <LI>Finds resources and can return them as is, as input streams and as byte arrays.
 * <LI>Loads classes and resources using the specified class path. Class
 *     path entries can be URLs, directories, zip files or jar files.
 * <P> A secondary classloader can be specified which is used to locate
 *     the class if its not found by this classloader.
 * <LI> A filter object can be supplied to determine if the current class
 *      loader should load the class or it should be passed to the parent or
 *      secondary class loader.
 * </UL>
 *
 * <H3>Class Path</H3>
 *
 * <P>If classes are to be loaded from a class path, it must be set by the
 * <CODE>setClassPath</CODE> or <CODE>addClassPath</CODE> methods or by
 * the constructor prior to calling <CODE>loadClass</CODE>. If the class
 * path is not set, the class will be loaded with the system class loader.
 *
 * <P>The class path can consist of directories, files, and/or URLs.
 * <P>Example valid class path entries are:
 * <PRE>
 *     <EM>Files and directories on the local file system</EM>
 *     ../../java/classes
 *     /users/kristen/java/classes
 *     /users/kristen/java/classes/
 *     /users/kristen/java/zipfiles/MyClasses.zip
 *     /users/kristen/java/jarfiles/MyClasses.jar
 *     file:///users/kristen/java/classes
 *     file://localhost/users/kristen/java/classes
 *     <BR>
 *     <EM>Files and directories on a remote file system
 *         (must be in URL format)</EM>
 *     ftp://www.foo.com/pub/java/classes
 *     file://www.foo.com/pub/java/classes/
 *     http://www.foo.com/web/java/classes/
 *     file://www.foo.com:8080/pub/java/zipfiles/MyClasses.zip
 *     http://www.foo.com:8080/web/java/jarfiles/MyClasses.jar
 * </PRE>
 *
 * <P>Note that the location of the entry includes the protocol, the host name,
 * and the port while the file name is everything else.  For example,
 *
 * <PRE>
 *     http://www.foo.com:8080/web/java/jarfiles/MyClasses.jar
 * </PRE>
 * has the form [location][name] or
 * <PRE>
 *     [http://www.foo.com:8080/][/web/java/jarfiles/MyClasses.jar]
 * </PRE>
 * so the location is "http://www.foo.com:8080/" and the name is
 * "/web/java/jarfiles/MyClasses.jar".
 *
 * <P>Note that the two references
 * <PRE>
 *     /users/kristen/java/classes/
 *     file:///users/kristen/java/classes/
 * </PRE>
 * represent the same directory on a Unix machine, and
 * <PRE>
 *     C|/windows/java/classes/
 *     file:///C|/windows/java/classes/
 * </PRE>
 * are equivalent directories on a Windows box.
 *
 * <P>But the two references
 * <PRE>
 *     /users/kristen/java/classes/
 *     file://monet.lutris.com/users/kristen/java/classes/
 * </PRE>
 * are not equivalent even if the directory
 * <EM>/users/kristen/java/classes/</EM> lives
 * on the machine named <EM>monet.lutris.com</EM> and all development
 * is on this machine.  Why?  Because the web (browser?) protocol is different
 * for URLs with host information and those without.  If no host is
 * specified, the file is assumed to be on the local machine and the
 * path is determined from the <EM>ROOT</EM> of the machine.  If the
 * host is specified, then the <EM>ftp protocol</EM> is used and the path
 * is determined from the <EM>ftp ROOT</EM> (e.g. /users/ftp/) rather
 * than the machine's ROOT.  Thus, on a machine that support's anonymous
 * ftp, the following two directories are the same:
 * <PRE>
 *     /users/ftp/pub/classes/
 *     file://picasso.lutris.com/pub/classes/
 * </PRE>
 * assuming the development is being done on <EM>picasso.lutris.com</EM>.
 *
 * <H3>System Class Path</H3>
 *
 * <P>The system class path is the system-dependent path of directories
 * and files (e.g. CLASSPATH on Unix and Windows) used by the system
 * class loader to load classes.  This class path is usually configured
 * prior to executing a Java program but can be dynamically configured
 * during runtime if desired.  If you want to use the system class path
 * for this class loader, the convenience method <CODE>getSystemClassPath</CODE>
 * has been provided.
 *
 * <P>Valid system class path entries are
 * directories and zip files, specified by absolute path or relative path
 * on the system.  Any valid system class path entry is also valid
 * for this class loader.
 *
 * <H3>Example</H3>
 *
 * <P>Here is an example of how to use this class loader:
 * <PRE>
 *     MultiClassLoader loader = new MultiClassLoader();
 *     loader.setClassPath("/web/java/lutris.jar");
 *     loader.addClassPath("/users/kristen/java/");
 *     loader.addClassPath("/usr/local/lib/graphics.zip");
 *     try {
 *         Class c = loader.loadClass("com.lutris.util.MyClass");
 *         System.out.println("My loader is " + c.getClassLoader());
 *         Object o = (Object) c.newInstance();
 *         System.out.println("My class is " + o.getClass());
 *     } catch (ClassNotFoundException e) {
 *         Throwable t = new Throwable();
 *         t.printStackTrace();
 *     }
 * </PRE>
 *
 * <P>
 * <B>Warning:</B>  This class loader is not yet fully compliant with Java 1.2.  It
 * maybe used on 1.2, but not all features are available.  The <em>parent</em> loader,
 * <em>secondary</em> loader, and <em>filter</em> may change in a future release without
 * maintaining compatibility.
 *
 * <!--
 * FIXME:
 * This class needs revisited for JDK 1.2, including adding security
 * managers.  Should implement the find* methods, but probably
 * must have its own loadClass.  Need to have resource loading
 * follow same paradigm as class loading.
 *
 * -->
 *
 * @author Kristen Pol, Lutris Technologies
 * @version $Revision : 1.0 $
 * @see java.lang.ClassLoader
 * @see com.lutris.classloader.Resource
 * @see java.net.URL
 */
public class MultiClassLoader extends ClassLoader  {

  /**
   * A filter interface, used in deciding if a class should be loaded
   * by this class loader.
   */
  public interface ClassFilter {
    /**
     * Value returned to indicate that we don't care, other filters
     * maybe checked.
     */
    public static final int NORMAL_LOAD = 1;
    /**
     * Value returned to indicate that the class should not be loaded
     * by this class loader.  Normal delation/secondary class loader
     * checks will be done.
     */
    public static final int DONT_LOAD = 2;
    /**
     * Value returned to indicate that the class can be loaded
     * by this class loader.  If it is not loaded, it will be
     * passed to the secondary class loader.  The delegate
     * is not check, since it is check before this class loader.
     */
    public static final int CAN_LOAD = 3;
    /**
     * Value returned to indicate that the class must loaded
     * by this class loader.  If it is not loaded, not other
     * class loader will be checked.
     */
    public static final int MUST_LOAD = 4;

    /**
     * Check if a class should be loaded by this class loader.
     * @param className the class being loaded.
     * @return One of NORMAL_LOAD, DONT_LOAD, CAN_LOAD, or MUST_LOAD.
     */
    public int loadCheck(String  className);
  }

    static {
        try {
            Method getSystemClassLoaderMethod = ClassLoader.class.getMethod( "getSystemClassLoader");
            sysClassLoader = (ClassLoader)getSystemClassLoaderMethod.invoke(ClassLoader.class);
        } catch (NoSuchMethodException var1) {
            sysClassLoader = new MultiClassLoader.SysClassLoader();
        } catch ( InvocationTargetException var2) {
            throw new FatalExceptionError( var2);
        } catch (IllegalAccessException var3) {
            throw new FatalExceptionError(var3);
        }

    }

    static class SysClassLoader extends ClassLoader {
        SysClassLoader() {
        }

        public Class loadClass(String className, boolean resolve) throws ClassNotFoundException {
            Class foundClass = this.findSystemClass(className);
            if (resolve) {
                this.resolveClass(foundClass);
            }

            return foundClass;
        }

        public URL getResource(String var1) {
            return getSystemResource(var1);
        }

        public InputStream getResourceAsStream(String var1) {
            return getSystemResourceAsStream(var1);
        }
    }

  /**
   * Information kept about a loaded class.
   */
  public class ClassResource {

    /**
     * Class object
     */
    private Class  classObj;

    /**
     * Resource were the class was obtained.
     */
    private Resource resource;

    /**
     * Constructor.
     */
    public ClassResource(Class  classObj, Resource resource) {
      this.classObj = classObj;
      this.resource = resource;
    }

    /**
     * Get the class object.
     */
    public Class  getClassObj() {
      return classObj;
    }

    /**
     * Get the resource.  Maybe null if the class was not loaded by
     * a MultiClassLoader.
     */
    public Resource getResource() {
      return resource;
    }
  }

  /**
   * The ClassResource objects for classes loaded by this classloader
   * (but not those handled by the parent).
   */
  private Hashtable  loadedClasses = new Hashtable ();

  /**
   * The class path to check when loading classes.
   */
  private ClassPath classPath;

  /**
   * Log level symbolic name
   */
  public static final String  LOG_LEVEL = "CLASSLOAD";

  /**
   * Is logging enabled?
   */
  private boolean loggingEnabled = false;

  /**
   * Log channel to write messages to; maybe null.
   */
  private LogChannel logChannel;

  /**
   * Numeric log level number for LOG_LEVEL string
   */
  private int logLevel;

  /**
   * If any class filters are supplied, this is a non-null list of
   * filters to check.
   */
  private Vector  filters = null;

  /**
   * Parent class loader, or null if there isn't one.
   */
  private ClassLoader  parentClassLoader;

  /**
   * Secondary class loader, or null if there isn't one.
   */
  private ClassLoader  secondaryClassLoader;

  private static ClassLoader sysClassLoader;

  public static ClassLoader getSystemClassLoader() {
      return sysClassLoader;
  }

  /**
   * Constructs class loader with no initial class path and a
   * specified parent class loader.
   * @param parent The parent class loader for delegation,
   * or null if no parent is defined.
   * @param secondary The secondary class loader.
   * Use <code>getSystemClassLoader</code> to get the system class loader
   * to specify as the secondary.
   * @param loadLogChannel The log channel, maybe null.
   * @see getSystemClassLoader
   */
  public MultiClassLoader(ClassLoader  parent, ClassLoader  secondary, LogChannel loadLogChannel) {
    classPath = new ClassPath(loadLogChannel);
// v. puskas, 10.03.2003    native compile changes;
//    parentClassLoader = parent;
    setParent( parent );
    secondaryClassLoader = secondary;
    logChannel = loadLogChannel;
    if (logChannel != null) {
      logLevel = logChannel.getLevel(LOG_LEVEL);
      loggingEnabled = logChannel.isEnabled(logLevel);
    }

  }

  /**
   * Constructs class loader with no initial class path and the
   * system class loader as the secondary.
   * @param loadLogChannel The log channel, maybe null.
   */
  public MultiClassLoader(LogChannel loadLogChannel) {
    this(null, getSystemClassLoader(), loadLogChannel);
  }

  /**
   * Constructs class loader with specified class path.  The parameter is
   * assumed to be either a directory, URL, or zip file.
   * @param path The class path represented by a String.
   * @param loadLogChannel The log channel, maybe null.
   */
  public MultiClassLoader(String  path, LogChannel loadLogChannel) {
    this(new String [] {path}, loadLogChannel);
  }

  /**
   * Constructs class loader with specified class path.  The parameter is
   * assumed to be an array of directories, URLs, and/or zip files.
   * @param path The class path represented by a String array.
   * @param loadLogChannel The log channel, maybe null.
   */
  public MultiClassLoader(String [] path, LogChannel loadLogChannel) {
    this(loadLogChannel);
    setClassPath(path);
  }

  /**
   * Constructs class loader with specified class path.  The parameter is
   * assumed to be either a zip file or directory.
   * @param path The class path represented by a File.
   * @param loadLogChannel The log channel, maybe null.
   */
  public MultiClassLoader(File  path, LogChannel loadLogChannel) {
    this(new File [] {path}
         , loadLogChannel);
  }

  /**
   * Constructs class loader with specified class path.  The parameter is
   * assumed to be an array of zip files and/or directories.
   * @param path The class path represented by a File array.
   * @param loadLogChannel The log channel, maybe null.
   */
  public MultiClassLoader(File [] path, LogChannel loadLogChannel) {
    this(loadLogChannel);
    setClassPath(path);
  }

  /**
   * Constructs class loader with specified class path.  The parameter is
   * represent a directory or zip file on the local machine or a
   * remote machine.
   * @param path The class path represented by a URL.
   * @param loadLogChannel The log channel, maybe null.
   */
  public MultiClassLoader(URL  path, LogChannel loadLogChannel) {
    this(new URL [] {path}
         , loadLogChannel);
  }

  /**
   * Constructs class loader with specified class path.  The parameter is
   * represent directories and/or zip files on the local machine and/or
   * on remote machines.
   * @param path The class path represented by a URL array.
   * @param loadLogChannel The log channel, maybe null.
   */
  public MultiClassLoader(URL [] path, LogChannel loadLogChannel) {
    this(loadLogChannel);
    setClassPath(path);
  }

  /**
   * Sets class loader with specified class path.  The parameter is
   * assumed to be either a directory, URL, or zip file.
   * @param path The class path to be used when loading classes.
   */
  public void setClassPath(String  path) {
    setClassPath(new String [] {path});
  }

  /**
   * Sets class loader with specified class path.  The parameter is
   * assumed to be an array of directories, URLs, and/or zip files.
   * @param path The class path to be used when loading classes.
   */
  public synchronized void setClassPath(String [] path) {
    classPath.set(path);
  }

  /**
   * Sets class loader with specified class path.  The parameter is
   * assumed to be either a zip file or directory.
   * @param path The class path to be used when loading classes.
   */
  public void setClassPath(File  path) {
    setClassPath(new File [] {path});
  }

  /**
   * Sets class loader with specified class path.  The parameter is
   * assumed to be an array of zip files and/or directories.
   * @param path The class path to be used when loading classes.
   */
  public synchronized void setClassPath(File [] path) {
    classPath.set(path);
  }

  /**
   * Sets class loader with specified class path.  The parameter is
   * represent a directory or zip file on the local machine or a
   * remote machine.
   * @param path The class path to be used when loading classes.
   */
  public void setClassPath(URL  path) {
    setClassPath(new URL [] {path});
  }

  /**
   * Sets class loader with specified class path.  The parameter is
   * represent directories and/or zip files on the local machine and/or
   * on remote machines.
   * @param path The class path to be used when loading classes.
   */
  public synchronized void setClassPath(URL [] path) {
    classPath.set(path);
  }

  /**
   * Adds specified class path to beginning of existing path.
   * The parameter is
   * assumed to be either a directory, URL, or zip file.
   * @param path The class path to be added to current class path.
   */
  public void addClassPath(String  path) {
    addClassPath(new String [] {path});
  }

  /**
   * Adds specified class path to beginning of existing path.
   * The parameter is
   * assumed to be an array of directories, URLs, and/or zip files.
   * @param path The class path to be added to current class path.
   */
  public synchronized void addClassPath(String [] path) {
    classPath.add(path);
  }

  /**
   * Adds specified class path to beginning of existing path.
   * The parameter is assumed to be either a zip file or directory.
   * @param path The class path to be added to current class path.
   */
  public void addClassPath(File  path) {
    addClassPath(new File [] {path});
  }

  /**
   * Adds specified class path to beginning of existing path.
   * The parameter is
   * assumed to be an array of zip files and/or directories.
   * @param path The class path to be added to current class path.
   */
  public synchronized void addClassPath(File [] path) {
    classPath.add(path);
  }

  /**
   * Adds specified class path to beginning of existing path.
   * The parameter is
   * represent a directory or zip file on the local machine or a
   * remote machine.
   * @param path The class path to be added to current class path.
   */
  public void addClassPath(URL  path) {
    addClassPath(new URL [] {path});
  }

  /**
   * Adds specified class path to beginning of existing path.
   * The parameter is
   * represent directories and/or zip files on the local machine and/or
   * on remote machines.
   * @param path The class path to be added to current class path.
   */
  public synchronized void addClassPath(URL [] path) {
    classPath.add(path);
  }

  /**
   * Clears class path entries.
   * @see #setClassPath
   */
  public synchronized void clearClassPath() {
    classPath.clear();
  }

  /**
   * Gets class path for class loader defined previously by constructor and
   * <CODE>setClassPath</CODE>/<CODE>addClassPath</CODE> methods.
   * @return the class path represented by an <CODE>Enumeration</CODE>
   * of URL objects.
   * @see #setClassPath
   * @see #addClassPath
   * <!-- FIXME: should really return a format that can be passed
   *      to another class loader, like an array -->
   */
  public URL [] getClassPath() {
    int len = classPath.getLength();
    URL [] urlPath = new URL [len];
    Enumeration  cpeEnum = classPath.getPath();
    for (int i = 0; i < len; i++) {
      ClassPathEntry cpe = (ClassPathEntry) cpeEnum.nextElement();
      urlPath[i] = cpe.getURL();
    }
    return urlPath;
  }

  /**
   * Parse a class-path string using the system path separator.
   */
  public static String [] parseClassPath(String  path) {
    String  systemSeparator = System.getProperty("path.separator");
    if (systemSeparator == null) {
      throw new NullPointerException ("path.separator property not defined");
    }
    StringTokenizer  tokenizer = new StringTokenizer (path, systemSeparator);
    String [] parsed = new String [tokenizer.countTokens()];
    for (int i = 0; tokenizer.hasMoreTokens(); i++) {
      parsed[i] = tokenizer.nextToken();
    }
    return parsed;
  }

  /**
   * Gets class path from system.
   *
   * @return the system class path represented by an
   *         array of URL objects.
   */
  public static URL [] getSystemClassPath() {
    // Parse system class path into its components
 String  systemClassPath = System.getProperty("java.class.path");
    if (systemClassPath == null) {
      systemClassPath = "";
    }
    String [] parsedPath = parseClassPath(systemClassPath);
    // Convert to URLs, dropping invalid entries
 Vector  urlVector = new Vector (parsedPath.length);
    for (int i = 0; i < parsedPath.length; i++) {
      try {
        urlVector.addElement(new URL ("file", "", parsedPath[i]));
      }
      catch (MalformedURLException  mue) {
        // Do not add this entry
      }
    }
    URL [] urlArray = new URL [urlVector.size()];
    urlVector.copyInto(urlArray);
    return urlArray;
  }

  /**
   * Set the parent class loader for delegation.
   */
  public void setParent(ClassLoader  parent) {
      this.parentClassLoader = null != parent ? parent : getSystemClassLoader();
  }

  /**
   * Get the secondary class loader.
   */
  public ClassLoader  getSecondary() {
    return secondaryClassLoader;
  }

  /**
   * Set the secondary class loader.
   */
  public void setSecondary(ClassLoader  secondary) {
    secondaryClassLoader = secondary;
  }

  /**
   * Get the log channel associated this class loader.
   */
  public LogChannel getLogChannel() {
    return logChannel;
  }

  /**
   * Add a filter to the list of filters that check if a class maybe
   * loaded by this class loader.
   */
  public synchronized void addClassFilter(ClassFilter filter) {
    if (filters == null) {
      filters = new Vector ();
    }
    filters.addElement(filter);
  }

  /**
   * Check class filters to determine if a class should be loaded by
   * this class loader.  This also handles check an implicit filter
   * for java.lang.* classes, which must go to the main class loader.
   * @return One of NORMAL_LOAD, DONT_LOAD, CAN_LOAD, or MUST_LOAD.
   */
  private int checkFilters(String  className) {
    int restrict = ClassFilter.NORMAL_LOAD;
    if (className.startsWith("java.lang.")) {
      restrict = ClassFilter.DONT_LOAD;
    }
    else if (filters != null) {
      int numFilters = filters.size();
      for (int idx = 0; idx < numFilters; idx++) {
        restrict = ( (ClassFilter) filters.elementAt(idx)).loadCheck(className);
        if (restrict != ClassFilter.NORMAL_LOAD) {
          break;
        }
      }
    }
    // Log restriction, if any.
 if ( (restrict != ClassFilter.NORMAL_LOAD) && loggingEnabled) {
      String  msg = "";
      switch (restrict) {
        case ClassFilter.DONT_LOAD:
          msg = "Filter disallows loading by this classloader: ";
          break;
        case ClassFilter.CAN_LOAD:
          msg = "Filter allows loading by this classloader: ";
          break;
        case ClassFilter.MUST_LOAD:
          msg = "Filter requires loading by this classloader: ";
          break;
      }
      if (loggingEnabled) {
        logChannel.write(logLevel, msg + className);
      }
    }
    return restrict;
  }

  /**
   * Check the table of loaded classes to determine if a class is already
   * loaded.  Purposely not synchronized, allowing a quick check for a class
   * being loaded.  A second check, which higher level synchronization is
   * required if the class is not found.  This function does logs a message
   * when a class is found.
   * @param className The class to look up.
   * @return The ClassResource object for the class, or null if not found.
   */
  private ClassResource checkForLoadedClass(String  className) {
    ClassResource cr = (ClassResource) loadedClasses.get(className);
    if ( (cr != null) && loggingEnabled) {
      logChannel.write(logLevel, "loadClass already loaded: " + className);
    }
    return cr;
  }

  //FIXME: method with resolve parameter should be protected (see base class).
 /**
   * Loads and, optionally, resolves the specified class. If the class
   * needs to be instantiated, the class must be resolved.  If only the
   * existence of the class needs verification, class resolution is
   * unnecessary.
   *
   * Calling <CODE>loadClass(String className)</CODE> is equivalent to calling
   * this method with <CODE>resolve</CODE> set to <CODE>true</CODE>.
   *
   * Purposely not synchronized. If class is not found in loadedClasses table, then
   * doLoadClass will do the operation in a synchronized manner.
   *
   * @param className The name of the class to be loaded, e.g.
   *        "com.lutris.util.Table".
   * @param resolve Set to <CODE>true</CODE> for class resolution,
   *        <CODE>false</CODE> for no resolution.
   * @return the loaded Class.
   * @exception ClassNotFoundException if the class could not be load.
   * @see #setClassPath
   * @see #addClassPath
   */
  public Class  loadClass(String  className, boolean resolve) throws ClassNotFoundException  {
    return loadClassResource(className, resolve).getClassObj();
  }

  /**
   * Log a class load failer.
   */
  private void logClassLoadFailure(String  className, Throwable  except) {
    if (loggingEnabled) {
      logChannel.write(logLevel, "load of class failed: " + className, except);
    }
  }

  /**
   * Log a resource read failer.
   */
  private void logReadFailure(String  name, Throwable  except) {
    if (loggingEnabled) {
      logChannel.write(logLevel, "read of resource failed: " + name, except);
    }
  }

  /**
   * Loads and, optionally, resolves the specified class, returning the
   * <CODE>ClassResource</CODE> object.
   *
   * @param className The name of the class to load.
   * @param resolve Set to <CODE>true</CODE> for class resolution,
   *        <CODE>false</CODE> for no resolution.
   * @return Object containing class and resource.
   * @exception ClassNotFoundException if this loader and the system loader
   *            can not find or successfully load the class.
   * @see #setClassPath
   * @see #addClassPath
   * @see #loadClass
   */
  private ClassResource loadClassResource(String  className, boolean resolve) throws ClassNotFoundException  {
    if (loggingEnabled) {
      logChannel.write(logLevel, "loadClass: " + className);
    }
    try {
      ClassResource cr = checkForLoadedClass(className);
      if (cr == null) {
        cr = doLoadClass(className);
        if (resolve) {
          resolveClass(cr.getClassObj());
        }
      }
      return cr;
    }
    catch (ClassNotFoundException  except) {
      logClassLoadFailure(className, except);
      throw except;
    }
    catch (IOException  except) {
      logClassLoadFailure(className, except);
      throw new ClassNotFoundException (except.getClass().getName() + ": " +
                                       except.getMessage());
    }
    catch (RuntimeException  except) {
      logClassLoadFailure(className, except);
      throw except;
    }
    catch (Error  except) {
      logClassLoadFailure(className, except);
      throw except;
    }
  }

  /**
   * Get the resource for a class loaded by this class loader.
   */
  public Resource getClassResource(String  className) throws ClassNotFoundException  {
    ClassResource cr = checkForLoadedClass(className);
    if (cr == null) {
      throw new ClassNotFoundException ("Class \"" + className + "\" is not loaded by this class loader");
    }
    return cr.getResource();
  }

  /**
   * Load a class with the parent or secondary class loader.
   * Must already be synchronized
   */
  private ClassResource loadClassOther(String  className, ClassLoader  loader,
                                       String  loaderName) {
    ClassResource cr = null;

    if (loggingEnabled) {
      logChannel.write(logLevel, "checking " + loaderName + " class loader: " + className);
    }
    try {
      if (loader instanceof MultiClassLoader) {
        cr = ( (MultiClassLoader) loader).loadClassResource(className, false);
      }
      else {
        Class  c = loader.loadClass(className);
        cr = new ClassResource(c, null);
      }
      if (loggingEnabled) {
        logChannel.write(logLevel, "class loaded by: " + loaderName);
      }
    }
    catch (ClassNotFoundException  except) {
      // Just log and return null
 if (loggingEnabled) {
        logChannel.write(logLevel, "class not loaded by " + loaderName
                         + ": " + except.getClass().getName() + ": " + except.getMessage());
      }
    }
    return cr;
  }

  /**
   * Load a class with this class loader.  Must already be
   * synchronized
   */
  private ClassResource loadClassHere(String  className) throws IOException  {
    ClassResource cr = null;

    if (loggingEnabled) {
      logChannel.write(logLevel, "checking MultiClassLoader: " + className);
    }
    String  fileName = className.replace('.', '/').concat(".class");
    Resource resource = getResourceObject(fileName);

    if (resource != null) {

      Class  c;
      try {
        // standard use of MultiClassLoader
          c = Class.forName(className);
      }
      catch (ClassNotFoundException  e) {
        try {
          // second try to search the class among Dll-mapped classes
 // this requires the call Class.forName with system class
 // loader
          c = Class.forName(className, true, ClassLoader.getSystemClassLoader());
        }
        catch (ClassNotFoundException  e1) {
          // if there is no precompiled class, try to compile it with JET JIT
 // NOTE! this will success only if -Djet.jit property is defined
 byte[] classBytes = resource.getBytes();
          c = defineClass(className, classBytes, 0, classBytes.length);
        }
      }

      cr = new ClassResource(c, resource);
      loadedClasses.put(className, cr);

      if (loggingEnabled) {
        logChannel.write(logLevel, "class loaded by MultiClassLoader: " + className);
      }
    }
    if ( (cr == null) && loggingEnabled) {
      logChannel.write(logLevel, "class not loaded by MultiClassLoader: " + className);
    }
    return cr;
  }

  /**
   * Loads specified class.  If possible, the class will loaded from the
   * class path defined previously with <CODE>setClassPath</CODE>,
   * <CODE>addClassPath</CODE>, and/or the constructor.  Otherwise, the
   * class will be passed off to the parent class loader.  Classes from
   * the java.* packages cannot be loaded by this class loader so will
   * be passed off to the parent class loader.
   * @param className The name of the class to be loaded, e.g.
   *        "com.lutris.util.Table".
   * @return Object containing class and resource.
   * @exception ClassNotFoundException if this loader and the system loader
   *            can not find or successfully load the class.
   * @see #setClassPath
   * @see #addClassPath
   */
  private synchronized ClassResource doLoadClass(String  className) throws ClassNotFoundException ,
      IOException  {
    // Must check loaded class table again, as we are now synchronized.
    ClassResource cr = checkForLoadedClass(className);
    if (cr != null) {
      return cr;
    }
    // Get filter restriction to use (includes check for java.lang.*).
 int filterRestrict = checkFilters(className);

    // Delegate if parent exists and filter allows.
 if ( (parentClassLoader != null) && (filterRestrict != ClassFilter.CAN_LOAD)
        && (filterRestrict != ClassFilter.MUST_LOAD)) {
      cr = loadClassOther(className, parentClassLoader, "parent");
      if (cr != null) {
        return cr;
      }
    }

    // Try loading with this class loader if filter allows.
 if ( (filterRestrict == ClassFilter.NORMAL_LOAD) || (filterRestrict == ClassFilter.CAN_LOAD)
        || (filterRestrict == ClassFilter.MUST_LOAD)) {
      cr = loadClassHere(className);
      if (cr != null) {
        return cr;
      }
    }

    // Try secondary class loader.
 if ( (secondaryClassLoader != null) && (filterRestrict != ClassFilter.MUST_LOAD)) {
      cr = loadClassOther(className, secondaryClassLoader, "secondary");
      if (cr != null) {
        return cr;
      }
    }
    throw new ClassNotFoundException (className);
  }

  /**
   * Gets specified resource as URL.  Doing a getContent() on the URL may
   * return an Image, an AudioClip, or an InputStream.
   * @param name The name of the resource.
   * @return the resource represented by a URL, or null if not found.
   */
  public URL  getResource(String  name) {
    Resource resource = getResourceObject(name);
    if (resource != null) {
      ClassPathEntry location = resource.getLocation();
      try {
        return new URL (location.getURL() + resource.getName());
      }
      catch (MalformedURLException  mue) {
        if (loggingEnabled) {
          logChannel.write(logLevel, "getResource not returned due to exception: " + name, mue);
        }
        return null;
      }
    }
    return null;
  }

  /**
   * Gets specified resource object.
   * @param name The name of the resource.
   * @return the resource if found, null if not.
   * @see Resource
   */
  public Resource getResourceObject(String  name) {
    Resource resource;
    if (loggingEnabled) {
      logChannel.write(logLevel, "getResource loading: " + name);
    }
    try {
      resource = classPath.getResource(name);
      if (loggingEnabled) {
        if (resource == null) {
          logChannel.write(logLevel, "getResource not found: " + name);
        }
        else {
          logChannel.write(logLevel, "getResource finished: " + name);
        }
      }
    }
    catch (RuntimeException  except) {
      logReadFailure(name, except);
      throw except;
    }
    catch (Error  except) {
      logReadFailure(name, except);
      throw except;
    }
    return resource;
  }

  /**
   * Gets specified resource object.
   * @see deprecated Use getResourceObject()
   * @see getResourceObject
   */
  public Resource getResourceAsIs(String  name) {
    return getResourceObject(name);
  }

  /**
   * Gets specified resource as input stream.
   * @param name The name of the resource.
   * @return an input stream representing the specified resource or
   *         null if the resource is not found.
   */
  public InputStream  getResourceAsStream(String  name) {
    Resource resource = getResourceObject(name);
    if (resource != null) {
      try {
        return resource.getInputStream();
      }
      catch (IOException  except) {
        logReadFailure(name, except);
        return null; // This is what getResourceAsStream does..
      }
    }
    return null;
  }

  /**
   * Gets specified resource as array of bytes.
   * @param name The name of the resource.
   * @return an array of bytes representing the specified resource or
   *         null if the resource is not found.
   */
  public byte[] getResourceAsByteArray(String  name) {
    Resource resource = getResourceObject(name);
    if (resource != null) {
      try {
        return resource.getBytes();
      }
      catch (IOException  except) {
        logReadFailure(name, except);
        return null; // This is what getResourceAsStream does..
      }
    }
    return null;
  }

  /**
   * Determine if the classes loaded by this class loader have been modified.
   * If any file associated with loaded in the class loader's class path has
   * changed, the classes should be reloaded.  If the classes need to be
   * reloaded, a new instance of this class loader must be created
   * because a particular class loader instance can only load classes
   * once.
   * @return <CODE>true</CODE> if the classes should be reloaded,
   *     <CODE>false</CODE> if not.
   */
  public boolean shouldReload() {
    Enumeration  classes = loadedClasses.elements();
    if (loggingEnabled) {
      logChannel.write(logLevel, "Checking for out-of-date classes");
    }
    while (classes.hasMoreElements()) {
      boolean isModified;
      try {
        isModified = ( (ClassResource) classes.nextElement()).getResource().hasBeenModified();
      }
      catch (FileNotFoundException  except) {
        if (loggingEnabled) {
          logChannel.write(logLevel, "File for loaded class can no longer be accessed", except);
        }
        isModified = true;
      }
      if (isModified) {
        if (loggingEnabled) {
          logChannel.write(logLevel, "Loaded classes have been modified");
        }
        return true;
      }
    }
    if (loggingEnabled) {
      logChannel.write(logLevel, "Loaded classes have not been modified");
    }
    return false;
  }
}
