

/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: ConfigFile.java,v 1.12 2005/03/25 16:41:03 slobodan Exp $
 */

package com.lutris.util;

import java.io.File ;
import java.io.FileInputStream;
import java.io.FileNotFoundException ;
import java.io.FileOutputStream ;
import java.io.IOException ;
import java.io.InputStream;
import java.io.OutputStream ;
import java.io.PrintWriter;
import java.util.Hashtable ;
import java.util.Vector ;

import org.enhydra.util.ConfigFileInterface;

/**
 * ConfigFile is used to manipulate Multiserver configuration (*.conf) files.
 *
 * Presents all configuration elements in the form of a keyed
 * table.  Configuration elements are grouped by string "keys" according
 * to the function they configure.  The syntax is described more formally
 * below.
 * <PRE>
 *  stream ::= entry | stream entry
 *  entry ::= key "=" value_list | comment | blank
 *  value_list ::= value | value_list "," value
 *  value ::= fragment | value "+" fragment
 *  fragment ::= key | quoted_string
 *  quoted_string ::= (C/C++ style quoted string)
 *  key ::= (A string matching [A-Za-z_\./][A-Za-z0-9_-\./]*)
 *  comment ::= "#" (any text up to a newline)
 *  blank ::= (A line containing only white space)
 * </PRE>
 * In addition to the above syntax, some additional semantic rules apply.
 * The operator "+" concatenates the fragment immediately to the left
 * and to the right of it.  Thus <code>ab + cd</code> results in
 * <code>abcd</code>.  The operator "," terminates one element and begins
 * the next element.  Thus, the line <code>val = hello, world</code>
 * creates a configuration entry with the key "val" and the two elements
 * "hello", and "world".  If the characters "+", ",", or "\" occur at
 * the end of a line, then the entry is continued on the next line.
 * Trailing and leading whitespaces are ignored, and multiple whitespace
 * characters are converted by default into a single space.  The "+"
 * operator leaves no whitespace between operands.
 * Finally, within quoted strings, C-style backslash escapes are
 * recognized.  These include "\n" for newline, "\t" for tab, etc.
 * An example configuration input file is given below.
 * <PRE>
 *  #==============================================================
 *  # Sample.config
 *  #==============================================================
 *  LDAP_SERVERS[] =  "server1.ldap.gov:338",
 *            "server2.ldap.gov:1000"
 *  USER_TIMEOUT =  3600  # seconds.  Comments can follow on same line.
 *  STRANGE_ENTRY = "This is a long long long long long " +
 *          "long long line.  Notice how the \"+\" " +
 *          "operator is used to break this line up " +
 *          "into more than one line in the config file." ,
 *          "And this is the second element."
 *  # etc.
 * </PRE>
 *
 * @see Config
 * @author Shawn McMurdo
 * @version $Revision: 1.12 $
 */
public class ConfigFile implements ConfigFileInterface {

    /**
     * The configuration object to which this configuration file belongs to.
     */
    private Config config;

    /**
     * Order in which are configuration parameters added.
     */
    private Vector  order;

    /**
     * Comments of configuration parameters.
     */
    private Hashtable  comments;

    /**
     * The associated file (if any).
     */
    private File  file = null;

    public static final String TRAILING_COMMENT = "ConfigFileTrailingComment";

    /**
     * Default constructor for an empty configuration file.
     */
    public ConfigFile () {
        config = new Config();
        order = new Vector ();
        comments = new Hashtable ();
    }

    /**
     * Constructor from an InputStream.
     * @param inputStream   The input stream from which to parse the config file.
     * @exception ConfigException
     */

  public ConfigFile ( InputStream inputStream) throws ConfigException {
      this.config = new Config();
      this.order = new Vector();
      this.comments = new Hashtable();
      ConfigParser configParser = new ConfigParser(inputStream);

      try {
          configParser.process(this);
      } catch (ParseException var4) {
          throw new ConfigException(ConfigException.SYNTAX, var4.getMessage());
      }
  }


    /**
     * Constructor from a File. Allows to later write back the configuration to the
     * same file.
     * @param file The local file to parse.
     * @exception IOException
     * @exception ConfigException
     */
    public ConfigFile (File  file) throws ConfigException, IOException  {
        this( new FileInputStream( file ) );
        this.file = file;
        config.setConfigFile(this);
    }

    /**
     * Constructor from a KeywordValueTable.
     * @param kvt   A KeywordValueTable from which to populate the configuration file.
     * @exception ConfigException
     */
    public ConfigFile(KeywordValueTable kvt) throws ConfigException {
        config = new Config(kvt);
        order = new Vector ();
        comments = new Hashtable ();
    }

    /**
     * Returns the Config object representing the config data in the file.
     * @return The Config object for this ConfigFile.
     */
    public Config getConfig() {
        return config;
    }

    /**
     * Returns the comment associated with a given key, or <code>null</code> if
     * there is no comment. Pass in <code>ConfigFileInterface.TRAILING_COMMENT</code> to get
     * the trailing comment.
     * @param key the key to look up.
     * @return the associated comment or <code>null</code>
     */
    public String  getComment(String  key) {
        return (String )comments.get(key);
    }

    /**
     * Adds an entry to the configuration.
     * @param key   The config element name.
     * @param values A string array of values.
     * @param comment   A string containing a properly commented configuration file
     * comment.
     * @exception KeywordValueException
     */
    public void addEntry(String  key, String [] values, String  comment)
        throws KeywordValueException {

        // Don't add an actual config entry for the trailing comment
        if (!key.equals(TRAILING_COMMENT)) {
            config.set(key, values);
            if (!order.contains(key)) {
                order.addElement(key);
            }
        }
        comments.put(key, comment);
    }

    /**
     * Adds an entry to the configuration.
     * @param key   The config element name.
     * @param value A String value.
     * @param comment   A string containing a properly commented configuration file
     * comment.
     * @exception KeywordValueException
     */
    public void addEntry(String  kkey, String  value, String  comment)
        throws KeywordValueException {
        // Don't add an actual config entry for the trailing comment
        if (!kkey.equals(TRAILING_COMMENT)) {
            config.set(kkey, value);
            if (!order.contains(kkey)) {
                order.addElement(kkey);
            }
        }
        comments.put(kkey, comment);
    }

    /**
     * Removes an entry from the configuration.
     * @param key   The config element name.
     * @exception KeywordValueException
     */
    public void removeEntry(String  key) throws KeywordValueException {
        // There is no config entry for the trailing comment
        if (!key.equals(TRAILING_COMMENT)) {
            config.remove(key);
            order.removeElement(key);
        }
        comments.remove(key);
    }

    /**
     * Gets the associated file. If no file is associated with this config, <code>null</code> is
     * returned.
     * @return associated file
     */
    public File  getFile() {
        return file;
    }

    /**
     * Sets the configuration file. This method can be useful in case when in
     * construction of ConfigFile object is not defined associated file. After the
     * file is set, Configuration parameters are read by using JNDI Context.
     * @param file given reference to configuration file represented as File object.
     */
    public void  setFile(File  file) {
        this.file = file;
    }

    /**
     * Writes this config to the associated configuration file. If no file is
     * associated with this config, throws a <code>FileNotFoundException</code>.
     * @exception IOException
     * @exception FileNotFoundException
     */
    public void write() throws IOException , FileNotFoundException  {
        if (file == null) {
            throw new FileNotFoundException ("No file associated with this object");
        }
        FileOutputStream  out = new FileOutputStream (file);
        write(out);
        out.close();
    }

    /**
     * Writes out a config file to the OutputStream specified. Note that Objects
     * other than String or String[] will be converted into a String.
     * @param outputStream The output stream on which to write the config file.
     */
    public void write(OutputStream  outputStream) {
        PrintWriter out = new PrintWriter ( outputStream, true);
        boolean isArray = false;
        String  key;
        String  comment;
        String [] values;
        Hashtable  remaining = new Hashtable ();
        String [] remainingkeys = config.leafKeys();

        // Set up the remaining keys list
        // The value doesn't matter, just the key
        for (int i = 0; i < remainingkeys.length; i++)
            remaining.put(remainingkeys[i], "X");

        // Do all the entries for which we have comments
        for (int i = 0; i < order.size(); i++) {
            key = (String ) order.elementAt(i);
            comment = (String ) comments.get(key);
            isArray = false;
            try {
                Object  o = config.get(key);
                if (o == null) {
                    continue;
                }
                isArray = o.getClass().isArray();
                if (isArray) {
                    Object [] oa = (Object [])o;
                    if ((oa.length > 0) && (oa[0] instanceof java.lang.String ))
                        values = (String []) o;
                    else {
                        values = new String [oa.length];
                        for (int k = 0; k < oa.length; k++)
                            values[k] = oa[k].toString();
                    }
                }
                else {
                    values = new String [1];
                    if (o instanceof java.lang.String )
                        values[0] = (String ) o;
                    else
                        values[0] = o.toString();
                }
            }
            catch (KeywordValueException e) {
                values = null;
            }
            // write out entry
            if ((values == null) || (values.length == 0)) {
                if ((comment != null) && !(comment.equals(""))) {
                    if (comment.endsWith("\n"))
                        out.print(comment);
                    else
                        out.println(comment);
                }
                out.print(key);
                if (isArray) out.print("[]");
                out.println(" =");
            }
            else {
                if ((comment != null) && !(comment.equals(""))) {
                    if (comment.endsWith("\n"))
                        out.print(comment);
                    else
                        out.println(comment);
                }
                if (isArray)
                    out.print(key + "[] = " + quoteStr(values[0]));
                else {
                    out.print(key + " = " + quoteStr(values[0]));
                }
                for (int j = 1; j < values.length; j++)
                    out.print(", " + quoteStr(values[j]));
                out.println();
            }
            remaining.remove(key);
        }

        // Do the trailing comment
        comment = (String ) comments.get(TRAILING_COMMENT);
        if ((comment != null) && !(comment.equals(""))) {
            if (comment.endsWith("\n")) {
                out.print(comment);
            }
            else {
                out.println(comment);
            }
        }
        remaining.remove(TRAILING_COMMENT);


        // The new keys are in a hash table with no order
        // sort them here so that they will be grouped correctly
        // in the conf file. It makes it easier to read.
        // XXX This is dependent on JDK 1.2 so comment it out
        // XXX until we don't support JDK 1.1 anymore.
/* XXX
      java.util.Arrays.sort(remainingkeys,new Comparator() {
         public int compare(Object o1, Object o2) {
            String s1 = (String) o1;
            return s1.compareTo((String) o2);
         }
      });
XXX */


        int i=0;
        String  lastWord = "";
        while (i != remainingkeys.length) {
            key = remainingkeys[i];
            i++;
            // Do the remaining config entries
            if (remaining.get(key) != null) {
                isArray = false;
                if (!key.startsWith(lastWord))
                    out.println("");
                int dot = key.indexOf('.');
                if (dot == -1)
                    dot = key.length();
                lastWord = key.substring(0,dot);

                try {
                    Object  o = config.get(key);
                    if (o == null)
                        continue;
                    isArray = o.getClass().isArray();
                    if (isArray) {
                        Object [] oa = (Object [])o;
                        if (oa[0] instanceof java.lang.String )
                            values = (String []) o;
                        else {
                            values = new String [oa.length];
                            for (int k = 0; k < oa.length; k++)
                                values[k] = oa[k].toString();
                        }
                    }
                    else {
                        values = new String [1];
                        if (o instanceof java.lang.String )
                            values[0] = (String ) o;
                        else
                            values[0] = o.toString();
                    }
                }
                catch (KeywordValueException e) {
                    values = null;
                }
                // write out entry
                if ((values == null) || (values.length == 0)) {
                    out.println(key + " =");
                }
                else {
                    if (isArray)
                        out.print(key + "[] = " + quoteStr(values[0]));
                    else
                        out.print(key + " = " + quoteStr(values[0]));

                    for (int j = 1; j < values.length; j++)
                        out.print(", " + quoteStr(values[j])); // VR 14.12.2002
                    //            out.print(key + ", " + quoteStr(values[j])); VR 14.12.2002
                    out.println();
                }
            }
        }
    }

    private boolean containsWhiteSpace(String  str) {
        if (str.indexOf(" ") != -1) {
            return true;
        }
        else if (str.indexOf("\t") != -1) {
            return true;
        }
        return false;
    }


    private static final String  quoteStr(String  s) {
        if ((s == null) || (s.length() < 1)) return "";
        char[] chars = s.toCharArray();
        StringBuffer  sb = new StringBuffer ();
        boolean needQuotes = false;
        for (int i=0; i<chars.length; i++) {
            switch (chars[i]) {
                // Chars that get special backquotes
                case '\n':
                    needQuotes = true;
                    sb.append("\\n");
                    break;
                case '\b':
                    needQuotes = true;
                    sb.append("\\b");
                    break;
                case '\r':
                    needQuotes = true;
                    sb.append("\\r");
                    break;
                case '\f':
                    needQuotes = true;
                    sb.append("\\f");
                    break;
                case '"':
                    needQuotes = true;
                    sb.append("\\\"");
                    break;
                case '\\':
                    needQuotes = true;
                    sb.append("\\\\");
                    break;

                // Chars that cause the string to be enclosed in
                // double quotes.
                case '\t': case ' ': case '!': case '#': case '$':
                case '%': case '&': case '\'': case '(': case ')':
                case '*': case '+': case ',': case '/': case ':':
                case ';': case '<': case '=': case '>': case '?':
                case '[': case ']': case '^': case '`': case '{':
                case '|': case '}': case '~':
                    needQuotes = true;
                    sb.append(chars[i]);
                    break;

                // All other characters.
                default:
                    if ((chars[i] < ' ') || (chars[i] == 0x7f)) {
                        needQuotes = true;
                        int ival = (int) chars[i];
                        sb.append('\\');
                        sb.append(digits[(ival & 0xc0) >> 6]);
                        sb.append(digits[(ival & 0x38) >> 3]);
                        sb.append(digits[(ival & 0x07)]);
                    }
                    else if (chars[i] > 0x7f) {
                        needQuotes = true;
                        int ival = (int) chars[i];
                        sb.append("\\u");
                        sb.append(digits[(ival & 0xf000) >> 12]);
                        sb.append(digits[(ival & 0x0f00) >> 8]);
                        sb.append(digits[(ival & 0x00f0) >> 4]);
                        sb.append(digits[(ival & 0x000f)]);
                    }
                    else
                        sb.append(chars[i]);
            }
        }
        if (needQuotes)
            return( "\"" + sb.toString() + "\"");
        return sb.toString();
    }

    private static final char[] digits = {
        '0', '1', '2', '3', '4', '5', '6', '7',
        '8', '9', 'a', 'b', 'c', 'd', 'e', 'f'
    };

}

