

/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: KeywordParser.java,v 1.3 2005/03/24 12:22:51 slobodan Exp $
 */





package com.lutris.util;
import java.text.MessageFormat ;
import java.util.StringTokenizer ;

/**
 * Class used to parse KeywordsValueTable keywords.
 *
 * @version $Revision: 1.3 $
 * @author  Mark Diekhans
 * @since   Jolt1.0
 */
public class KeywordParser {
    static private String  separator = ".";

    /**
     * Parse and validate the next component of a keyword.
     *
     * @param tokens Parsed token object.
     * @param compIdx Index of the component we are about to parse.  Used for
     *  error messages.
     * @param keyword Entire keyword for use in error messages.
     * @return The parsed keyword component.
     */
    static private String  parseComponent (StringTokenizer  tokens,
                                          int compIdx,
                                          String  keyword)
            throws KeywordValueException {
        String  comp = tokens.nextToken ();

        /*
         * Special check for keywords starting out with a separator, generates
         * a more helpful error message.
         */
        if (comp.equals (separator) && (compIdx == 0)) {
            String  pattern = "keyword should not start with a {0} separator: \"{1} \"";
            String  msg = MessageFormat.format(pattern, separator, keyword);

            //String msg = "keyword should not start with a '" +
 //    separator + "' separator: \"" + keyword + "\"";
 throw new KeywordValueException (msg);
        }

        /*
         * Validate the legality of the name.
         */
        boolean isOk = (comp.length () > 0);
        if (isOk) {
            if (!Character.isJavaIdentifierStart (comp.charAt (0))) {
                isOk = false;
            }
            for (int j = 1; j < comp.length (); j++) {
                if (!Character.isJavaIdentifierPart (comp.charAt (j))) {
                    isOk = false;
                    break;
                }
            }
        }
        if (!isOk) {
            String  msg = "keyword component must be a legal Java identifier " +
                "component \"" + comp + "\": \"" + keyword + "\"";
            throw new KeywordValueException (msg);
        }

        /*
         * Check the separator if its not the last component.
         */
        if (tokens.hasMoreTokens ()) {
            String  sep = tokens.nextToken ();
            if (!sep.equals (separator)) {
                String  msg = "keyword component separator must be a " +
                    "single '" + separator + "', got \"" + sep + "\": " +
                    keyword + "\"";
                throw new KeywordValueException (msg);
            }
        }
        return comp;
    }

    /**
     * Parse a keyword into its components, validating that the components
     * are legal names.
     *
     * @return An array of the keyword components.
     * @exception KeywordValueException If the keyword is not syntactically legal.
     */
    static public String  [] parse (String  keyword)
            throws KeywordValueException {

        StringTokenizer  tokens = new StringTokenizer  (keyword,
                                                      separator,
                                                      true);
        /*
         * Set up the array for the keywords, making sure that an odd
         * number of tokens are present, either single word or words
         * separated by `.'.
         */
        int numTokens = tokens.countTokens ();
        if ((numTokens % 2) != 1) {
            String  msg = "keyword component must be single word or words " +
                "separated by '" + separator + "': \"" + keyword + "\"";
            throw new KeywordValueException (msg);
        }
        int numComps = (numTokens / 2) + 1;
        String  [] keyParts = new String  [numComps];

        for (int compIdx = 0; compIdx < numComps; compIdx++) {
            keyParts [compIdx] =  parseComponent (tokens,
                                                  compIdx,
                                                  keyword);
        }

        return keyParts;
    }

    /**
     * Create a keyword from its components.
     *
     * @param keywordPath Array of path components.
     * @return The keyword path string.
     */
    static public String  join (String  [] keywordPath) {
        StringBuffer  keyword = new StringBuffer  ();

        for (int idx = 0; idx < keywordPath.length; idx++) {
            if (idx > 0) {
                keyword.append (separator);
            }
            keyword.append (keywordPath [idx]);
        }
        return keyword.toString ();
    }

    /**
     * Concatenate two keyword paths.
     *
     * @param keyword1 First keyword.
     * @param keyword2 Second keyword.
     * @return The keyword path string.
     */
    static public String  concat (String  keyword1,
                                 String  keyword2) {
        return keyword1 + separator + keyword2;
    }
}
