
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: LogChannel.java,v 1.1 2005/07/13 11:09:06 slobodan Exp $
 */
package com.lutris.logging;

/**
 * Interface of a channel associated with a logging facility.  All messages
 * for the facility are written using a channel. 
 *
 * @author Mark Diekhans
 * @see com.lutris.logging.Logger
 * @see com.lutris.logging.LogWriter
 */
public interface LogChannel {

    /**
     * Determine if logging is enabled for the specified level.  This
     * is useful to prevent a series of unnecessary logging calls,
     * as often encountered with debug logging, or a call where generating
     * the message is expensive.
     *
     * @param level Numeric level that is to be checked.
     * @return <CODE>true</CODE> if enabled, <CODE>false</CODE> if not
     *         enabled.
     */
    boolean isEnabled(int level);

    /**
     * Determine if logging is enabled for the specified level.  This
     * is useful to prevent a series of unnecessary logging calls,
     * as often encountered with debug logging, or a call where generating
     * the message is expensive.
     *
     * @param level Symbolic level that is to be checked.
     * @return <CODE>true</CODE> if enabled, <CODE>false</CODE> if not
     *         enabled.
     */
    boolean isEnabled(String  level);

    /**
     * Convert a symbolic level to an integer identifier.
     *
     * @param level Symbolic level to convert
     * @return The numeric level identifier
     */
    int getLevel(String  level);

    /**
     * Create a LogWrite associated with a particular level.  This
     * is often an easier object to use than a LogChannel if limited
     * levels are needed.
     *
     * @param level Symbolic level that is to be checked.
     * @return A log writer object.
     */
    LogWriter getLogWriter(String  level);

    /**
     * Create a LogWrite associated with a particular level.  This
     * is often an easier object to use than a LogChannel if limited
     * levels are needed.
     *
     * @param level Numeric level that is to be checked.
     * @return A log writer object.
     */
    LogWriter getLogWriter(int level);

    /**
     * Write a string to the log file.
     *
     * @param level Numeric level the message is associated with.
     * @param msg The message to log.
     */
    void write(int level, String  msg);
    
    /**
     * Write a string to the log file.
     *
     * @param level Symbolic level the message is associated with.
     * @param msg The message to log.
     */
    void write(String  level, String  msg);
    
    /**
     * Write a string and exception to the log file.
     *
     * @param level Numeric level the message is associated with.
     * @param msg The message to log.
     * @param throwable Exception or error to log.
     */
    void write(int level, String  msg, Throwable  throwable);
    
    /**
     * Write a string and exception to the log file.
     *
     * @param level Symbolic level the message is associated with.
     * @param msg The message to log.
     * @param throwable Exception or error to log.
     */
    void write(String  level, String  msg, Throwable  throwable);
}
