
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See 
 * the License for the specific terms governing rights and limitations
 * under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 * 
 * Contributor(s):
 * 
 * $Id: Log4jLogger.java,v 1.5 2005/03/24 10:51:20 slobodan Exp $
 */
package com.lutris.logging;

import java.io.File ;
import java.io.IOException ;
import java.io.PrintWriter ;
import java.util.Hashtable ;

import org.apache.log4j.BasicConfigurator;
import org.apache.log4j.ConsoleAppender;
import org.apache.log4j.Level;
import org.apache.log4j.Logger;
import org.apache.log4j.PatternLayout;
import org.apache.log4j.RollingFileAppender;
import org.apache.log4j.spi.LoggerRepository;
import org.apache.log4j.xml.DOMConfigurator;

import com.lutris.util.Config;
import com.lutris.util.ConfigException;

/**
 * Log4j implementation of the <CODE>Logger</CODE>.  This is 
 * general-purpose logging facility.  A client that needs additional
 * functionality can either extend this class or provide there own
 * implementationm of <CODE>Logger</CODE>. <P>
 * 
 * Currently this is a bare-bones class that writes INFO and above
 * levels to stderr and all others to a log file.
 *
 * @author Vladimir Puskas
 * @author Predrag Djoic
 * @author Sinisa Milosevic
 * @see com.lutris.logging.Logger
 * @see com.lutris.logging.LogChannel
 */
public class Log4jLogger extends com.lutris.logging.Logger {

    /**
     * Table of level names to level numbers.  While level configuration
     * is local to a facility, a global table is kept assigning numbers
     * to each level name.
     */
    private Hashtable  levelNumbers = new Hashtable ();

    /**
     * Table translating level number to name and the largest entry in the
     * array that is valid.  Will be expanded if needed.
     */
    protected String [] levelNames = new String [MAX_STD_LEVEL * 2];
    protected int numLevels = 0;

    /**
     * Table of levels that are to be enabled.
     * Accessed directly by the channel.  If null, ignored.
     */
    protected boolean[] enabledLevelFlags = null;

    /**
     * Table of levels that are to be written to the log file.
     * Accessed directly by the channel.  If null, ignored.
     */
    protected boolean[] logFileLevelFlags = null;

    /**
     * Table of levels that are to be written to stderr
     * Accessed directly by the channel.  If null, then
     * the default behavior of writing serious standard
     * levels to stderr is in affect.
     */
    protected boolean[] stderrLevelFlags = null;
        
    /**
     * Log file name.
     */
    File  activeLogFile;

    /**
     * Log file writter.  Use directly by channels.
     */
    PrintWriter  logFileStream;

    /**
     * Stderr writter.  Use directly by channels.
     */
    PrintWriter  stderrStream;

    /**
     * Table of <CODE>StandardLogChannel<CODE> objects, indexed by facility
     * name.
     */
    private Hashtable  logChannels = new Hashtable ();

    static String separatorLine = "";
    
    /**
     * Construct a new logger.  Configuration is not done now, to allow
     * the logger to be created very early.
     *
     * @param makeCentral Make this object the central logging object.
     */
    public Log4jLogger(boolean makeCentral) {
        int level;

        for (level = 0; level <= MAX_STD_LEVEL; level++) {
            String  name = standardLevelNames[level];

            levelNumbers.put(name, new Integer (level));
            levelNames[level] = name;
        }
        numLevels = level;
        if (makeCentral) {
            centralLogger = this;
        }
    }

    /**
     * Get maximum level number in a set of level names.
     *
     * @param levels String names of levels.
     * @return The maximum level number
     */
    private int getMaxLevel(String [] levels) {
        int levelNum;
        int maxLevelNum = 0;

        for (int idx = 0; idx < levels.length; idx++) {
            levelNum = getLevel(levels[idx]);
            if (levelNum > maxLevelNum) {
                maxLevelNum = levelNum;
            }
        }
        return maxLevelNum;
    }

    /**
     * Generate a boolean array for all of the listed levels, indicating
     * if they are enabled.
     *
     * @param levels String names of levels.
     * @param maxLevelNum Size to make the array.
     */
    private boolean[] getLevelStateArray(String [] levels, int maxLevelNum) {
        int levelNum;
        // Initialize the stated.
 boolean[] levelNums = new boolean[maxLevelNum + 1];

        for (int idx = 0; idx < levels.length; idx++) {
            levelNums[getLevel(levels[idx])] = true;
        }
        return levelNums;
    }

    /**
     * Configure the logger. All current configuration is discarded.
     * This is a simplistic initial implementation that just allows
     * directing to a single log file or stderr on a level basis.
     * A more complete interface will be provided in the future.
     *
     * @param logFile The log file to write to.  
     * @param fileLevels List of levels that will be written to the file.
     * @param stderrLevels List of levels that will be written to stderr.
     *  The same level may appear in both lists.
     * @exception java.io.IOException If an error occurs opening the log file.
     */
    public synchronized void configure(String  log4jConfFile)
        throws ConfigException {
        try {
            DOMConfigurator.configure(log4jConfFile);
        } catch (javax.xml.parsers.FactoryConfigurationError  fce) {
            throw new ConfigException("Cannot configure Log4jLogger:",
                    fce);
        }

  }

    /**
     * This method configures Logger if configure data was not found in config
     * file (at command line)
     */
    static private void configureLogger() {
        // do the basic configuration:
        BasicConfigurator.configure();
        Logger root = Logger.getRootLogger();

        // remove predefined appenders from basic configuration:
        root.removeAllAppenders();
        LoggerRepository repository = root.getLoggerRepository();

        // set level for log info:
        repository.setThreshold(Level.INFO);
        RollingFileAppender multiserverAppender = null;
        RollingFileAppender accessAppender = null;
        // strale@uns.ns.ac.yu, 29 nov 2002
 ConsoleAppender conAppender = null;
        // set path to log files:
 String  enhydraLogs = System.getProperty("enhydra.home")
                + File.separator + "logs" + File.separator;
        String  multiserverLogFile = enhydraLogs + "multiserver.log";
        String  accessLogFile = enhydraLogs + "access.log";

        try {

            /* create default appenders:
             * %d(ISO8601)  - date in format quick to write
             * %t           - time
             * %C(1)        - full class name (one level) i.e.: com.lutris.Enhydra -> Enhydra
             * %p           - priority of the input
             * %c           - category of the input
             * %m           - massage
             * %n           - new line
             */
            multiserverAppender = new RollingFileAppender(new PatternLayout("%d{ISO8601}: [%t], %C(1) %p, %c: %m%n"),
                    multiserverLogFile);
            accessAppender = new RollingFileAppender(new PatternLayout("%m%n"),
                    accessLogFile);
            conAppender = new ConsoleAppender(new PatternLayout("%d{ISO8601}: [%t], %C(1) %p, %c: %m%n"));
        } catch (IOException  e) {
            e.printStackTrace();
        }
        root.addAppender(multiserverAppender);
        // create new logger for with "REQUEST" category that log to access.log:
 Logger requestLogger = Logger.getLogger("REQUEST");

        // does not apply root appender to REQUEST logger:
        requestLogger.setAdditivity(false);
        requestLogger.addAppender(accessAppender);
        Logger sysOutLogger = Logger.getLogger("SysOut");

        sysOutLogger.setAdditivity(false);
        sysOutLogger.addAppender(conAppender);
    }

    /**
     * Create a log channel.
     */
    private synchronized Log4jLogChannel createChannel(String  facility) {
        Log4jLogChannel channel
                = (Log4jLogChannel) logChannels.get(facility);

        if (channel == null) {
            channel = new Log4jLogChannel(facility,
                    Logger.getLogger(facility));
            logChannels.put(facility, channel);
        }
       
        return channel;
    }

    /**
     * Get the log channel object for a facility.  For a given facility,
     * the same object is always returned.
     * 
     * @param facility Facility the channel is associated with.
     */
    public LogChannel getChannel(String  facility) {
    
    
        Log4jLogChannel channel
                = (Log4jLogChannel) logChannels.get(facility);

        if (channel == null) {
            // Slow path, synchronized
            channel = createChannel(facility);
        }
        return channel;
    }

    /**
     * Create a log level.
     */
    private synchronized Integer  createLevel(String  level) {
        Integer  intLevel = (Integer ) levelNumbers.get(level);

        if (intLevel == null) {
            intLevel = new Integer (numLevels);
            levelNames[numLevels] = level;
            levelNumbers.put(level, intLevel);
            numLevels++;
        }
        return intLevel;
    }

    /**
     * Convert a symbolic level to an integer identifier,
     * creating it if it doesn't exist
     *
     * @param level Symbolic level that is to be checked.
     * @return The numeric level identifier
     */
    public synchronized int getLevel(String  level) {
        Integer  intLevel = (Integer ) levelNumbers.get(level);

        if (intLevel == null) {
            // Slow path, synchronized
            intLevel = createLevel(level);
        }
        return intLevel.intValue();
    }

    /**
     * Convert an int to a symbolic level name.
     *
     * @param level an int level.
     * @return The String symolic level name or null if there is not one.
     */
    public String  getLevelName(int level) {
        if ((level >= 0) && (level < numLevels)) {
            return levelNames[level];
        } else {
            return null;
        }
    }

    /**
     * Configure Logger with given config section
     *
     * @param logConfig containing parameters for configuring logger
     */
    public void configure(Config logConfig) throws ConfigException {
          
        String  fileName = null;
        if (logConfig.containsKey("Log4j")) {
            fileName = logConfig.getString("Log4j");
        } else if (null != logConfig.getConfigFile()) {
            fileName = logConfig.getConfigFile().getFile().getParent();
            if (null != fileName) {
                fileName += File.separator + "log4j.xml";
            }
        } else {
            throw new ConfigException("Cannot configure logger. Config file is null.");
        }
        if (logConfig.containsKey("SeparatorLine")) {
        
            separatorLine = logConfig.getString("SeparatorLine");
          }
      configure(fileName);
    }
}
