
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: StandardDBTransaction.java,v 1.4 2005/03/03 17:33:29 tanja Exp $
 *
 */
package  com.lutris.appserver.server.sql.standard;

import com.lutris.appserver.server.sql.CachedDBTransaction;
import com.lutris.appserver.server.sql.CoreDO;
import com.lutris.appserver.server.sql.DBConnection;
import com.lutris.appserver.server.sql.DBQuery;
import com.lutris.appserver.server.sql.DBRowUpdateException;
import com.lutris.appserver.server.sql.DBTransaction;
import com.lutris.appserver.server.sql.DatabaseManager;
import com.lutris.appserver.server.sql.StandardDatabaseManager;
import com.lutris.appserver.server.sql.Transaction;
import com.lutris.dods.builder.generator.dataobject.GenericDO;
import com.lutris.dods.builder.generator.query.DataObjectException;
import com.lutris.logging.Logger;
import org.enhydra.dods.DODS;
import org.enhydra.dods.cache.DOCache;
import org.enhydra.dods.cache.TransactionCacheImpl;
import org.enhydra.dods.cache.Wrapper;
import org.enhydra.dods.exceptions.CacheObjectException;

import java.sql.PreparedStatement;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.LinkedHashSet;
import java.util.Set;
import java.util.Vector;

/**
 * Standard implementation of SQL database transaction.
 *
 * @see         DBTransaction
 * @author      Kyle Clark
 * @since       LBS1.8
 * @version     $Revision: 1.4 $
 */
public class StandardDBTransaction  implements CachedDBTransaction {
    // Identifier for this transaction object.
 protected int id;
    // Next available identifier.
 protected static int nextId;
    // Connection used by this transaction object.
 protected DBConnection conn;
    // Is this object still active, or has it been released.
 protected boolean released = false;
    protected boolean _preventCacheQueries = false;
    protected Set<CoreDO> vecAlreadyHidden = new LinkedHashSet<>();
    protected Vector vecExecutedTransaction = new Vector();
    // compliance with WEBDOCWF begin
 protected Vector vecSortedTransaction = new Vector();
    protected HashMap hmpObjectTransaction = new HashMap();
    // original line
 //
 // compliance with WEBDOCWF end
 // Objects that make up this transaction
 protected Transaction[] trans = {};
    // Action (INSERT, UPDATE, DELETE) to apply to each object
 // in the objects array.
 protected int[] transAction = {};
    // Index of next action to apply
 protected int transIdx = 0;
    // Counts how many times commit will be retried on deadlock exception
 protected int dbLockCounter = 0;
    // Amount by which to increment the array size if it is full.
 protected final int SIZE_DELTA = 10;
    // Transaction Cache size: number of DO objects (-1 unlimited)
 protected final int CACHE_SIZE = -1;

    // Transaction Complex Query Cache size
 protected final int COMPLEX_QUERY_CACHE_SIZE = 0;

    // Transaction Simple Query Cache size
 protected final int SIMPLE_QUERY_CACHE_SIZE = 0;

    // Transaction cache
 protected DOCache cache = null;

    // Transaction deleted DO's - circular references
 protected Set<String> deletedRefs = null;

    /*
     * The log channel.
     */
    //    private LogChannel channel;
 protected static final int INSERT = 1;
    protected static final int UPDATE = 2;
    protected static final int DELETE = 3;

    protected boolean isTransactionCaches;
    protected boolean isAutoWrite = false;
    protected boolean sqlBatch;
    protected boolean firstWrite = true;

    /**
     * Construct a transaction object for use on the supplied dB connection.
     *
     * @param conn
     *   The database connection to use.
     * @exception java.sql.SQLException
     *   If a database access error occurs.
     */
    protected StandardDBTransaction(DBConnection conn) throws SQLException  {
        try {
            firstWrite=true;
            id = nextId++;
            logDebug("new instance");
            this.conn = conn;
            this.conn.setAutoCommit(false);
            setDatabaseName(conn.getDatabaseName());
            readConfigValues();
            if (isTransactionCaches) {
                cache = new TransactionCacheImpl(SIMPLE_QUERY_CACHE_SIZE,
                                                 COMPLEX_QUERY_CACHE_SIZE);
            }
        } catch (SQLException  sqlExcept) {
            this.conn.handleException(sqlExcept);
            this.conn.release();
            throw  sqlExcept;
        } catch (CacheObjectException cacheExcept) {
            DODS.getLogChannel().write(Logger.ERROR,
                                       "Error during transacion cache initialization");
        }
    }

    /**
     * This constructor is only for use in inherited class.
     * Don't direct call.
     */
    protected StandardDBTransaction(){}


    /* Method find a DO in the transaction
     *
     * @param transaction
     *   Object that implements transaction interface.
     * @return DO if the oid was in the transaction, null if it was not
     *
     * WebDocWf extension
     */
    public synchronized Transaction getDO(Transaction transaction) {
        return  getDO(transaction, NONE);
    }

    /**
     * Method find a DO in the transaction
     *
     * @param transaction
     *   Object that implements transaction interface.
     * @param action
     *   if not NONE=0, the DO is found only woth the matching action
     * @return DO if the oid was in the transaction, null if it was not
     *
     * WebDocWf extension
     */
    public synchronized Transaction getDO(Transaction transaction, int action) {
        CoreDO foundDO = null;

        if (transaction instanceof CoreDO) {
            String  strOID = ((CoreDO) transaction).get_OId().toString();

            //            if (this.hmpObjectTransaction.containsValue(strOID)) {
 if (this.hmpObjectTransaction.containsKey(strOID)) {
                if (action == NONE
                    || ((DOAction) this.hmpObjectTransaction.get(strOID)).getAction()
                    == action) {
                    foundDO = (CoreDO) ((DOAction) this.hmpObjectTransaction.get(strOID)).getDO();
                }
            }
        }
        return  foundDO;
    }
    // Parameter for no special action
 protected static final int NONE = 0;
    // original line
 //
 // compliance with WEBDOCWF end
 /**
     * Method to add an object to the transaction list.
     *
     * @param transaction
     *   Object that implements transaction interface.
     */
    // compliance with WEBDOCWF begin
 private synchronized void add(Transaction transaction, int action) {
        if (transaction instanceof CoreDO) {
            String  strOID = ((CoreDO) transaction).get_OId().toString();
            int iDOAction = NONE;

            if (aggregateModifications && this.vecSortedTransaction.size() > 0) {
                DOAction doa = (DOAction) this.vecSortedTransaction.lastElement();

                if (((CoreDO) doa.getDO()).get_OId().toString().equals(strOID)) {
                    iDOAction = ((DOAction) this.vecSortedTransaction.lastElement()).getAction();
                    if (iDOAction == UPDATE
                        && (action == INSERT || action == UPDATE)) {
                        return;
                    }
                    if (iDOAction == INSERT
                        && (action == INSERT || action == UPDATE)) {
                        return;
                    }
                    if (iDOAction == DELETE && action == DELETE) {
                        return;
                    }
                }
            }
            if (isAutoWrite && this.vecSortedTransaction.size() > 0) {
                try {
                    write();
                } catch (SQLException  sqle) {
                    sqle.printStackTrace();
                    // FIXME:
                }
            }
            DOAction objDOAction = new DOAction(transaction, action);

            this.hmpObjectTransaction.put(strOID, objDOAction);
            this.vecSortedTransaction.add(objDOAction);
        }
    }

    /**
     * Method to update an object in the database.
     *
     * @param transaction
     *   Object that implements transaction interface.
     */
    public void update(Transaction transaction) {
        add(transaction, UPDATE);
        aggregateModifications = true;
    }

    /**
     * Method to delete an object in the database.
     *
     * @param transaction
     *   Object that implements transaction interface.
     */
    public void delete(Transaction transaction) {
        // If the transaction is an instance of CoreDO and we're
 // doing a DELETE, then check if the oid of this CoreDO
 // is the same as the oid of an existing CoreDO in our
 // transaction. If this is true, do not add this
 // transaction because we are already deleting the DO.
 // This will happen occassionally during complex CASCADE DELETE
 // scenarios when the programmer passes their own
 // DBTransaction to the ***DO.delete() methods.
        add(transaction, DELETE);
        aggregateModifications = true;
    }

    /**
     * Method to insert an object in the database.
     *
     * @param transaction
     *   Object that implements transaction interface.
     */
    public void insert(Transaction transaction) {
        add(transaction, INSERT);
        aggregateModifications = true;
    }

    /**
     * Method to commit upates.
     *
     *
     * contains WebDocWf bug fix, transaction needs to be cleared
     *
     * @exception java.sql.SQLException If a database access error occurs.
     * @exception DBRowUpdateException If a version error occurs.
     */
    public void commit()
        throws SQLException, DBRowUpdateException
    {
        int retryCount = lockRetryCount;
        long wrapperHandle = 0;
        Vector vecDOclass = new Vector();

        logDebug( "commit" );
        validate();
        if ( !wasReadOnly() )
        {
            // if this transaction has accumulated any writes
            // (either previously done, or pending), we do the drill
            try
            {
                conn.incrRequestCount();  // Record request
                write();
                // lock the main cache wrapper
                while ( 0 == (wrapperHandle = Wrapper.getInstance().lock()) )
                {
                    try
                    {
                        Thread.sleep( lockWaitTime );
                    }
                    catch ( InterruptedException ie )
                    {
                    }
                    if ( retryCount-- == 0 )
                    {
                        throw new SQLException( "Can't wait anymore." );
                    }
                }
                // hide DOs before commit, and gather Class objects for them
                makeHiddenDoEntities( vecDOclass );
                Wrapper.getInstance().makeSimpleComplexQCachesInvisible( vecDOclass );
                Wrapper.getInstance().makeMultiJoinQCachesInvisible();  // tj 01.03.2005
                //               Wrapper.getInstance().removeAllMultiJoinQueries(); // tj 29.08.2004
                Wrapper.getInstance().unlock( wrapperHandle );
                wrapperHandle = 0;
                conn.commit();
                // Notify all objects that the transaction succeeded.
                transactionNotify( true );
                Wrapper.getInstance().removeAllMultiJoinQueries(); // tj 29.08.2004, moved 01.03.2005
            }
            catch ( SQLException sqlExcept )
            {
                handleException( sqlExcept );
                rollback();
                throw sqlExcept;
            }
            finally
            {
                if ( retryCount >= 0 )
                {
                    while ( 0 == (wrapperHandle = Wrapper.getInstance().lock()) )
                    {
                        try
                        {
                            Thread.sleep( lockWaitTime );
                        }
                        catch ( InterruptedException ie )
                        {
                        }
                        if ( --retryCount == 0 )
                        {
                            logDebug( "spent " + lockRetryCount + " times "
                                          + lockWaitTime
                                          + "miliseconds, but musn't give up." );
                            retryCount = lockRetryCount;
                        }
                    }
                    makeVisibleDoEntities();
                    Wrapper.getInstance().makeSimpleComplexQCachesVisible( vecDOclass );
                    Wrapper.getInstance().makeMultiJoinQCachesVisible(); // tj 01.03.2005.
                    Wrapper.getInstance().unlock( wrapperHandle );
                }
                // compliance with WEBDOCWF begin
                this.hmpObjectTransaction = new HashMap();
                this.vecSortedTransaction = new Vector();
                this.vecExecutedTransaction = new Vector();
                this.vecAlreadyHidden = new LinkedHashSet<>();
                _preventCacheQueries = false;
            }
        }
        else
        {
            // this transaciton didn't changed anything in database,
            // so this commit may be safely ignored
            logDebug( "Nothing to write." );
        }
    }

    private void makeHiddenDoEntities( Vector vecDOclass )
    {
        for ( Enumeration e = vecExecutedTransaction.elements(); e.hasMoreElements(); )
        {
            CoreDO cdo = (CoreDO) ((DOAction) e.nextElement()).getDO();
            Class doClass = cdo.getClass();

            if ( !vecDOclass.contains( doClass ) )
            {
                vecDOclass.add( doClass );
            }
            if ( vecAlreadyHidden.add( cdo ) )
            {
                cdo.makeInvisible();
            }
        }
    }

    private void makeVisibleDoEntities()
    {
        for ( CoreDO coreDO : vecAlreadyHidden )
        {
            coreDO.makeVisible();
        }
    }

    /**
     * Method to rollback changes.
     *
     *
     * contains WebDocWf bug fix, transaction needs to be cleared
     *
     * @exception java.sql.SQLException
     *   If a database access error occurs.
     */
    public void rollback() throws SQLException  {
        try {
            logDebug("rollback");
            validate();
            conn.rollback();
            // Updates cache after transaction's rollback
 for (int iCounter = 0; iCounter < this.vecSortedTransaction.size(); iCounter++) {
                try {
                    DOAction objDOAction = (DOAction) this.vecSortedTransaction.get(iCounter);

                    ((CoreDO) objDOAction.getDO()).refresh();
                } catch (DataObjectException dataExept) {
                    // removes DO from cache if it doesn't exist in the database
 if (dataExept.getMessage().indexOf("DO not found for id")
                        != -1) {  // if DO doesn't exist
                        DOAction objDOAction = (DOAction) this.vecSortedTransaction.get(iCounter);

                        ((CoreDO) objDOAction.getDO()).evict();
                    } else {
                        SQLException  g = new SQLException ("INTERNAL ERROR: unexpected DataObjectException");
                        throw g;
                    }
                }
            }
        } catch (SQLException  sqlExcept) {
            handleException(sqlExcept);
            throw  sqlExcept;
        }
        finally {
            // Notify all objects that the transaction failed.
            transactionNotify(false);
            // compliance with WEBDOCWF begin
 this.hmpObjectTransaction = new HashMap();
            this.vecSortedTransaction = new Vector();
            // original line
 //
 // compliance with WEBDOCWF end
        }
    }

    /**
     * Notify all objects in this transaction if the
     * transaction was successfully commited to the database.
     * An object will only be notified once of success or failure
     * even if this method is called multiple times.
     *
     * @param succeeded true if the transaction commited successfully.
     */
    private synchronized void transactionNotify(boolean succeeded) {
        for (int iCounter = 0; iCounter < this.vecExecutedTransaction.size(); iCounter++) {
            DOAction objDOAction = (DOAction) this.vecExecutedTransaction.get(iCounter);

            switch (objDOAction.getAction()) {
                case INSERT:
                    objDOAction.getDO().finalizeInsert(succeeded);
                    break;

                case UPDATE:
                    objDOAction.getDO().finalizeUpdate(succeeded);
                    break;

                case DELETE:
                    objDOAction.getDO().finalizeDelete(succeeded);
                    break;
            }
        }
    }

    /**
     * Frees all resources consumed by this transaction
     * Connections are returned to the connection pool.
     * Subsequent transactions via this object,
     * will allocate a new set of resources (i.e. connection).
     *
     * contains WebDocWf bug fix, transaction needs to be cleared
     *
     */
    public synchronized void release() {
        try {
            logDebug("release");
            if (!released) {
                if (vecExecutedTransaction.size() > 0) {
                    rollback();
                }
                conn.reset();
            }
        } catch (SQLException  sqlExcept) {
            handleException(sqlExcept);
        }
        finally {
            if(conn!=null)
                conn.release();
            conn = null;
            released = true;
            cache = null;
            this.hmpObjectTransaction = new HashMap(); // TODO should be null?
 this.vecSortedTransaction = new Vector();
            this.vecExecutedTransaction = new Vector();
        }
    }

    /**
     * Exception handeler.  This object is should not be
     * used for subsequent queries if this method returns
     * false.
     *
     * @return boolean True if the exception can be handeled
     *   and the object is still valid, false otherwise.
     */
    public synchronized boolean handleException(SQLException  e) {
        logDebug("handle exception");
        return  conn.handleException(e);
    }

    /**
     * Method to ensure this object is still valid.
     * Once this object has been released it cannot be
     * used any more.
     *
     * @exception java.sql.SQLException
     *   If a database access error occurs.
     */
    protected void validate() throws SQLException  {
        if (released) {
            throw new SQLException ("Cannot access DBTransaction object "
                                        + "once it has been released.");
        }
    }


    /**
     * Method to ensure this object is still valid.
     * Once this object has been released it cannot be
     * used any more.
     *
     * @return boolean True if the transaction is released, otherwise false
     */
    public boolean isReleased() {
        return released;
    }

    /**
     * If this object has not been <A HREF=#release>released</A>,
     * this method ensures that garbage collection does so.
     */
    protected void finalize() {
        if (!released) {
            release();
        }
    }

    /**
     * Logging. For debuging only, since it effects all Query objects.
     *
     * @param str
     *   The data to log.
     */
    protected void logDebug(String  str) {
        if (DatabaseManager.debug) {
            DODS.getLogChannel().write(Logger.DEBUG, "DBTransaction[" + id + "]: " + str);
        }
    }
    protected class DOAction {
        private Transaction transDO;
        private int iAction;
        public DOAction(Transaction transDOIn, int iActionIn) {
            this.transDO = transDOIn;
            this.iAction = iActionIn;
        }

        public Transaction getDO() {
            return this.transDO;
        }

        public int getAction() {
            return this.iAction;
        }

        public void setAction(int action) {
            this.iAction = action;
        }
        public boolean equals(Object  o) {
            return o instanceof DOAction
                && iAction == ((DOAction)o).iAction
                && transDO.equals(((DOAction)o).transDO);
        }
    }

    /**
     * Name of used database
     */
    protected String  databaseName;

    /**
     * Method return name of used database
     *
     * @return name of used database
     */
    public String  getDatabaseName() {
        return databaseName;
    }

    /**
     * Method set name of used database
     *
     * @param dbName name of used database
     */
    public void setDatabaseName(String  dbName) {
        databaseName = dbName;
    }

    /**
     * Gets an array of DOs.
     *
     * @return array of DOs from this transaction
     */
    public CoreDO[] getDOs() {
        return (CoreDO[]) vecSortedTransaction.toArray();
    }

    /**
     *
     */
    public void saveDirtyDOs() {
        throw new RuntimeException ("NOT implemented, yet!");
    }

    /**
     *
     * @exception java.sql.SQLException If a database access error occurs.
     * @exception DBRowUpdateException If a version error occurs.
     */
    public void write() throws SQLException , DBRowUpdateException {
        logDebug("write");
        validate();
        // compliance with WEBDOCWF begin
 if (firstWrite){
            firstWrite=false;
            if (conn.getConnection().isClosed()){
                logDebug("Write failed, connection are closed by driver");
                throw new SQLException ("Write failed, connection are closed by driver");
            }
        }
        try {
            PreparedStatement  stmt = null;
            ArrayList batch = new ArrayList();
            for (int iCounter = 0; iCounter < this.vecSortedTransaction.size();/* iCounter++*/) {
                DOAction objDOAction = (DOAction) this.vecSortedTransaction.get(iCounter);
                int action = objDOAction.getAction();
                CoreDO aDO = (CoreDO) objDOAction.getDO();

                if (this.vecSortedTransaction.size() > 1) {
                    DOAction objNextDOAction = (DOAction) this.vecSortedTransaction
                        .get(1 + iCounter);

                    if (DELETE == objNextDOAction.getAction()
                        && aDO.get_OId().equals(((CoreDO) objNextDOAction.getDO()).get_OId())) {
                        if (!aDO.isPersistent()) {
                            this.vecSortedTransaction.remove(objNextDOAction);
                        }
                        this.vecSortedTransaction.remove(objDOAction);
                        continue;
                    }
                }
                switch (action) {
                    case INSERT:
                        if (isSQLbatch()) {
                            if (!(aDO instanceof GenericDO)
                                || ((GenericDO)aDO).isDirty()) {
                                stmt = (aDO.isPersistent())
                                    ? aDO.getUpdateStatement(conn)
                                    : aDO.getInsertStatement(conn);
                                stmt.addBatch();
                                aDO.setPersistent(true);
                                if (!batch.contains(stmt)) {
                                    batch.add(stmt);
                                }
                            }
                        } else
                            aDO.executeInsert(conn);
                        break;

                    case UPDATE:
                        if (isSQLbatch()) {
                            if (!(aDO instanceof GenericDO)
                                || ((GenericDO)aDO).isDirty()) {
                                stmt = aDO.getUpdateStatement(conn);
                                stmt.addBatch();
                                if (!batch.contains(stmt)) {
                                    batch.add(stmt);
                                }
                            }
                        } else
                            aDO.executeUpdate(conn);
                        break;

                    case DELETE:
                        if (isSQLbatch()) {
                            stmt = aDO.getDeleteStatement(conn);
                            stmt.addBatch();
                            if (!batch.contains(stmt))
                                batch.add(stmt);
                        } else
                            aDO.executeDelete(conn);
                        break;
                }
                _preventCacheQueries = true;

                this.vecExecutedTransaction.add(objDOAction);
                this.vecSortedTransaction.remove(iCounter);
            }
            if (isSQLbatch()) {
                //new Throwable("BATCH_SIZE:"+batch.size()).printStackTrace();
 for (Iterator iter = batch.iterator();
                     iter.hasNext();) {
                    PreparedStatement  element = (PreparedStatement )iter.next();
                    int[] affected = element.executeBatch();
                    element.clearBatch();
                    for (int _i = 0; _i < affected.length; ++_i) {
                        if (affected[_i] <= 0)
                            throw new DBRowUpdateException("batch failed for "
                                                               + element
                                                               +" returned "
                                                               +affected[_i]);
                    }
                }
            }
        } catch (SQLException  e) {
            logDebug("Write failed, there are " + vecSortedTransaction.size()
                         + " DOs unwritten.");
            throw e;
        }
    }

    /**
     * Method return transaction Cache
     *
     * @return implementation of DOCache
     */
    public DOCache getTransactionCache() {
        return cache;
    }

    /**
     * Method set Transaction Cache
     *
     * @param implementation of DOCache
     */
    private void setTransactionCache(DOCache transCache) {
        cache = transCache;
    }
    private long lockWaitTime;
    private int lockRetryCount;

    /**
     *
     */
    public Set<String> getDeletedDOs() {
        return deletedRefs;
    }

    /**
     *
     */
    public void addDeletedDO(CoreDO DO) {

        if (deletedRefs == null) {
            deletedRefs = new HashSet<>();
        }
        String  handle = null;

        if (DO.get_OId() != null) {
            handle = getDatabaseName() + "." + DO.get_OId();
            deletedRefs.add(handle);
        }
    }

    /**
     *
     */
    public void resetDeletedDOs() {
        if (deletedRefs != null) {
            deletedRefs = new HashSet<>();
        }
    }

    public void lockDO(Transaction cdo) throws SQLException  {
        ((CoreDO) cdo).executeLockingStatement(conn);
    }

    /**
     * @return true if this transaction has executed a statement
     *         against database, so cached queries are obsolete
     */
    public boolean preventCacheQueries() {
        return _preventCacheQueries;
    }

    /**
     * Return a query for use with this TRANSACTION please!!!
     *
     * @return The query object.
     * @exception SQLException
     *   if a SQL error occurs.
     */
    public DBQuery createQuery() throws SQLException  {
        StandardDBQuery sdbq = new StandardDBQuery(conn);

        sdbq.setReleaseConnection(false);
        return sdbq;
    }
    protected boolean aggregateModifications = true;

    /**
     *
     */
    public void dontAggregateDOModifications() {
        aggregateModifications = false;
    }

    /**
     * Return a query for use with this TRANSACTION please!!!
     *
     * @return The query object.
     * @exception SQLException
     *   if a SQL error occurs.
     */
    protected void readConfigValues() {
        try {
            isTransactionCaches = ((StandardLogicalDatabase) (DODS.getDatabaseManager().findLogicalDatabase(getDatabaseName()))).getDatabaseConfiguration().getTransactionCaches();
        } catch (Exception  ex) {
            isTransactionCaches = ((StandardDatabaseManager) (DODS.getDatabaseManager())).getDatabaseManagerConfiguration().getTransactionCaches();
        }
        try {
            lockWaitTime = ((StandardLogicalDatabase) (DODS.getDatabaseManager().findLogicalDatabase(getDatabaseName()))).getDatabaseConfiguration().getDeadlockWaitTime();
        } catch (Exception  ex) {
            lockWaitTime = ((StandardDatabaseManager) (DODS.getDatabaseManager())).getDatabaseManagerConfiguration().getDeadlockWaitTime();
        }
        try {
            lockRetryCount = ((StandardLogicalDatabase) (DODS.getDatabaseManager().findLogicalDatabase(getDatabaseName()))).getDatabaseConfiguration().getDeadlockRetryCount();
        } catch (Exception  ex) {
            lockRetryCount = ((StandardDatabaseManager) (DODS.getDatabaseManager())).getDatabaseManagerConfiguration().getDeadlockRetryCount();
        }
        try {
            isAutoWrite = ((StandardLogicalDatabase) (DODS.getDatabaseManager().findLogicalDatabase(getDatabaseName()))).getDatabaseConfiguration().getAutoWrite();
        } catch (Exception  ex) {
            isAutoWrite = ((StandardDatabaseManager) (DODS.getDatabaseManager())).getDatabaseManagerConfiguration().getAutoWrite();
        }
        try {
            sqlBatch = ((StandardLogicalDatabase) (DODS.getDatabaseManager().findLogicalDatabase(getDatabaseName()))).getDatabaseConfiguration().isSqlBatch();
        } catch (Exception  ex) {
            sqlBatch = ((StandardDatabaseManager) (DODS.getDatabaseManager())).getDatabaseManagerConfiguration().isSqlBatch();
        }
    }

    /**
     * Method returns value of a data member.
     *
     * @return value of config parameter AutoWrite
     *         true means all DOs are writen into database, as they come in
     *         false - no writes are done implicitly
     */
    public boolean getAutoWrite() {
        return isAutoWrite;
    }
    public boolean isSQLbatch() {
        return sqlBatch;
    }

    /**
     *
     */
    public boolean isFirstWrite(){
        return firstWrite;
    }


    /**
     *
     */
    public void setFirstWrite(boolean newfw){
        firstWrite=newfw;
    }

   public boolean wasReadOnly() {
      return 0 == vecExecutedTransaction.size() + vecSortedTransaction.size();
   }
}
