
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: SimpleConnectionAllocator.java,v 1.1 2005/05/26 08:08:10 predrag Exp $
 */
package com.lutris.appserver.server.sql.standard;

import java.lang.reflect.Constructor ;
import java.sql.SQLException ;
import java.util.Date ;
import java.util.Hashtable ;
import java.util.LinkedList ;
import java.util.NoSuchElementException ;

import org.enhydra.dods.CommonConstants;
import org.enhydra.dods.DODS;

import com.lutris.appserver.server.sql.AbstractDBConnectionFactory;
import com.lutris.appserver.server.sql.ConnectionAllocator;
import com.lutris.appserver.server.sql.DBConnection;
import com.lutris.appserver.server.sql.ExtendedConnectionAllocator;
import com.lutris.appserver.server.sql.ExtendedDBConnection;
import com.lutris.appserver.server.sql.LogicalDatabase;
import com.lutris.appserver.server.sql.SimpleDBConnectionFactory;
import com.lutris.logging.Logger;
import com.lutris.util.Config;
import com.lutris.util.ConfigException;
import com.lutris.util.KeywordValueException;

/**
 * Manages a pool (set) of connections to a database.  The pool is named
 * as a logical database.  By naming a pool, this allows connection  resource
 * control to be on a finer grain that a database and allows for easier
 * migration to multiple databases.  One or more pools can map to the same
 * actual database.  A connection considered part of the pool, even if its
 * allocated to a thread.  These objects are all publicly accessed via the
 * Database Manager, not directly.
 * <P>
 * If an error occurs in a connection, it is dropped from the pool.  The
 * process using the connection has already received an error which aborts
 * the work in progress.  By dropping the connection, waiting threads are
 * restarted.  If something is wrong with the database (e.g. server is down),
 * they will recieve errors and also be aborted.  A generation number is used
 * to close down all connections that were open when the error occured,
 * allowing new connections to be allocated.
 * <P>
 * The configuration data is specified in the section:
 * <B><CODE>DatabaseManager.DB.<I>dbName</I>.Connection</CODE></B>
 * <P>
 * <I>Configuration sub fields are:</I>
 * <UL>
 * <LI> <B><CODE>Url</CODE></B> -
 *      The JDBC URLof the database. Manditary.
 *      E.g. "jdbc:sequelink://dbHost:4000/[Informix];Database=dummy"
 * <LI> <B><CODE>User</CODE></B> -
 *      The database users used to access the database. Manditary.
 * <LI> <B><CODE>Password</CODE></B> -
 *      The database user's password. Manditary.
 * <LI> <B><CODE>MaxPoolSize</CODE></B> -
 *      The maximum number of open connections to the database. Optional,
 *      if not specified, then it default to 0. A value of 0 means that
 *      connections are allocated indefinitely or until the database (JDBC)
 *      refuses any new connections.
 * <LI> <B><CODE>Logging</CODE></B> -
 *      Specify true to enable SQL logging, false to disable it. Optional,
 *      false if not specified.
 * <LI> <B><CODE>AllocationTimeout</CODE></B> -
 *      The Maximum amount of time that a thread will wait for
 *      a connection from the connection allocator before an exception is
 *      thrown. This will prevent possible dead locks. The time out is in
 *      milliseconds. If the time out is <= 0, the allocation of connections
 *      will wait indefinitely. Optional, if not specified, then it
 *      defaults to 1000 (ms).
 * <LI> <B><CODE>QueryTimeout</CODE></B> - The amount of time (in seconds) that
 *  a query will block before throwing an exception. If <= 0 then the
 *  query will not block. Optional, if not specified, then the value
 *  defaults to 0. This is not implemented by all logical databases.
 * <LI> <B><CODE>TransactionTimeout</CODE></B> - The amount of time (in seconds)
 *  that a transaction will block before throwing an exception. If
 *  <= 0 then the transaction will not block. Optional, if not specified,
 *  then the value defaults to 0. This is not implemented by all
 *  logical databases.
 * <LI> <B><CODE>MaxPreparedStatements</CODE></B> - If specified, overrides
 *      the JDBC <CODE>Connection.getMetaData().getMaxStatements()</CODE>
 *      value.  If less than zero, use the meta data value.  Optional,
 *      default is to use the meta data.
 * </UL>
 * It would be nice to add a config parameter that would disable caching
 * of PreparedStatements.
 *
 * @author  Mark Diekhans
 * @author  Kyle Clark
 * @author  Paul Morgan
 * @since   LBS1.8
 * @version $Revision: 1.1 $
 */
// Excelsior, LLC. Memory leak patch begin
/**
 * Brief description of the patch:
 * Since each long-time used connection holds too much memory (about 15Mb after
 * several hours of work) it was decided to close and recreate connections that
 * were allocated more than the MaxConnectionUsages value set in the
 * proper config file.
 * We also changed the stack-like structure of the connection pool to
 * queue-like. This guarantees that no connections become stagnant.
 * 4.07.2003
 */
// Excelsior, LLC. Memory leak patch end
public class SimpleConnectionAllocator implements ExtendedConnectionAllocator {

    /**
     * Reference to the logical database for easy access to the
     * connection pool.
     */
    protected LogicalDatabase logicalDatabase = null;

    /**
     * JDBC URL of database.
     */
    protected String  url;

    /**
     * SQL user name.
     */
    protected String  user;

    /**
     * SQL password..
     */
    protected String  password;

    
    /**
     * database "ShutDown" string 
     */
    protected String  shutDownStr;
    
    /**
     * Maximum number of connections in the pool.
     * If this value is <= zero, then create as many
     * connections as possible.
     */
    private int maxPoolSize;

    /**
     * Current size of the pool; includes allocated connections.
     */
    private int currentPoolSize;
    
    /**
     * Maximum time that connection will wait in connection pool before they will be closed;
     */
    private long connectionIdileTimeout = -1;

    /**
     * Maximum size the pool ever got to, regardless of generation.
     */
    private int biggestPoolSize;

    /**
     * Date at which the biggest pool size occured.
     */
    private Date  biggestPoolDate;

    /**
     * Number of queries or transactions on this logical database.
     */
    protected long numRequests;

    /**
     * The actual pool of DBConnection objects.
     */
    // Excelsior, LLC. Memory leak patch begin
 /* original code
     private Stack pool;
     */
    private LinkedList  pool;

    /**
     * This hashtable holds the number of times a particular DBConnection
     * was allocated. The key is the DBConnection object and the value is
     * an Integer object.
     * If the connectionUsageCounter is null, then the number of connection
     * usages is unlimited.
     * Maybe, it would be better to have the respective instance field in
     * the DBConnection but it worsens locality of changes made...
     */
    private Hashtable  connectionUsageCounter;

    /** JP Patch
     * Maximum number of DBConnections that will be waiting in the pool
     */
     protected int maxWaitingConnections;


    /**
     * Maximum number of times a particular connection is allowed to
     * be allocated before it is closed and replaced with a new one.
     * If this value is <= zero, then the number of allocations is unlimited.
     */
    private int maxConnectionUsages = -1;

    // Excelsior, LLC. Memory leak patch end
 /**
     * Indicates if logging is enabled.
     */
    protected boolean sqlLogging;

    /**
     * Maximum amount of time in milliseconds to wait for a connection.
     */
    private int timeOut;

    /**
     * Maximum amount of time in seconds to block on a query. The
     * DBQuery object will retrieve this value from the connection.
     */
    protected int queryTimeOut;

    /**
     * Maximum amount of time in seconds to block on a transaction. The
     * DBTransaction object will retrieve this value from the connection.
     */
    protected int transactionTimeOut;

    /**
     * Maximum number of prepared statements to use; if less-than zero,
     * then JDBC is queried for this value.
     */
    protected int maxPreparedStatements;

    /**
     * Generation number.  When an SQL error occurs, all objects of the
     * same generation or earlier are dropped.
     */
    protected int generation = 1;


    /**
     * Create a new connection in the pool.
     *
     * @exception java.sql.SQLException If a SQL error occures.
     */
    protected DBConnection createConnection()
        throws java.sql.SQLException  {
            DBConnection dbConnection = dbConnectionFactory.createConnection((ConnectionAllocator)this, url, user,
                    password, maxPreparedStatements, sqlLogging, generation);
            return dbConnection;
    }

    private AbstractDBConnectionFactory dbConnectionFactory = null;
    private String  dbConnectionFactoryName = null;
    private AbstractDBConnectionFactory createDBConnectionFactory(String  factoryName){
        Class  connectionFactoryClass = null;
        Constructor  connectionFactoryConstructor = null;
        Class [] methodTypes={};
        Object [] methodArgs={};
        AbstractDBConnectionFactory factory = null;
        if (factoryName!=null){         
            try{
                connectionFactoryClass = Class.forName(factoryName);
                factory = (AbstractDBConnectionFactory)connectionFactoryClass.newInstance();
            }catch(Exception  e){
                DODS.getLogChannel().write(Logger.INFO,"Faild to make Connection Factory :"+factoryName+" creating StandardDBConnectionFactory insted");
                factory = null;
            }
        }
        if (factoryName==null || factory == null){
            factory = new SimpleDBConnectionFactory();
        }
        return factory;

    }
    
    /**
     * Initialize the connection allocator object.  Connections are
     * opened on demand and stored in a pool.
     *
     * @param logicalDatabase LogicalDatabase.
     * @param conConfig LogicalDatabase Config object
     * @exception ConfigException if bad configuration information is
     *  given in the config file.
     */
    public SimpleConnectionAllocator(LogicalDatabase logicalDatabase,
            Config conConfig)
        throws ConfigException {
        this.logicalDatabase = logicalDatabase;
        try {
            url = conConfig.getString("Url");
            user = conConfig.getString("User");
            password = conConfig.getString("Password");
            timeOut = conConfig.getInt("AllocationTimeout", 1000);
            maxPoolSize = conConfig.getInt("MaxPoolSize", 0);
            sqlLogging = conConfig.getBoolean("Logging", false);
            queryTimeOut = conConfig.getInt("QueryTimeout", 0);
            transactionTimeOut = conConfig.getInt("TransactionTimeout", 0);
            connectionIdileTimeout = conConfig.getLong("ConnectionIdleTimeout", -1);
      
            shutDownStr =  conConfig.getString("ShutDownString",null);
    
            maxPreparedStatements = conConfig.getInt("MaxPreparedStatements", -1);
            // Excelsior, LLC. Memory leak patch begin
            maxConnectionUsages = conConfig.getInt("MaxConnectionUsages", -1);
            // Excelsior, LLC. Memory leak patch end
            maxWaitingConnections = conConfig.getInt("MaxWaitingConnections", Integer.MAX_VALUE);
            
            dbConnectionFactoryName = conConfig.getString(CommonConstants.CONNECTION_FACTORY,null);
            
            dbConnectionFactory = createDBConnectionFactory(dbConnectionFactoryName);
            
            
        } catch (KeywordValueException except) {
            throw new ConfigException("Bad DatabaseManager.DB."
                    + logicalDatabase.getName()
                    + ".Connection section defined in config file.");
        }
        currentPoolSize = 0;
        // Excelsior, LLC. Memory leak patch begin
 /* original code
         pool = new Stack ();
         */
        pool = new LinkedList ();
        if (maxConnectionUsages > 0) {
            connectionUsageCounter = new Hashtable ();
        }
        // Excelsior, LLC. Memory leak patch end
        biggestPoolSize = 0;
        biggestPoolDate = new Date ();
        numRequests = 0;
    }

    /**
     * Allocate a connection to a thread.  If none are available, grow the
     * pool.  If the pool is alredy its maximum size, then the thread waits.
     *
     * @return The allocated connection object.
     * @exception SQLException
     *   If a SQL error occures.
     */
    public synchronized DBConnection allocate()
        throws SQLException  {
        //
 // It isn't always possible to determine the maximum
 // number of connections allowed to the database because
 // of JDBC driver differences. We assume connections are
 // available until we fail to allocate one or we reach
 // the maximum configured.
 //
 boolean createNewConn = true;
        // Excelsior, LLC. Memory leak patch begin
 /* original code
         while (pool.empty()) {
         */
        DBConnection conn = null;

        while (conn == null) {
            while (pool.isEmpty()) {
                // Excelsior, LLC. Memory leak patch end
 if (createNewConn
                        && ((currentPoolSize < maxPoolSize)
                                || (maxPoolSize <= 0))) {
                    try {
                        // Excelsior, LLC. Memory leak patch begin
 /* original code
                         pool.push(createConnection());
                         */
                        DBConnection newConnection = createConnection();

                        if (connectionUsageCounter != null) {
                            connectionUsageCounter.put(newConnection,
                                    new Integer (maxConnectionUsages));
                        }
                        pool.addLast(newConnection);
                        // Excelsior, LLC. Memory leak patch end
                        currentPoolSize++;
                        if (currentPoolSize > biggestPoolSize) {
                            biggestPoolSize = currentPoolSize;
                            biggestPoolDate = new Date ();
                        }
                    } catch (SQLException  e) {
                        if (currentPoolSize > 0) {
                            DODS.getLogChannel().write(Logger.INFO,
                                    "ConnectionAllocator: "
                                    + "failed to allocate a new connection due to"
                                    + e.toString() + "Error code: "
                                    + e.getErrorCode() + "\n" + "SQLState: "
                                    + e.getSQLState() + "\n"
                                    + "\nCurrent pool size is: "
                                    + currentPoolSize
                                    + "\nMaximum configured pool size is now "
                                    + maxPoolSize + "\nContinuing...\n");
                            createNewConn = false;
                        } else {
                            DODS.getLogChannel().write(Logger.EMERGENCY,
                                    "ConnectionAllocator: "
                                    + "failed to allocate a new connection"
                                    + "\nThe connection pool is empty!\n");
                            throw e;
                        }
                    }
                } else {
                    try {
                        if (timeOut > 0) {
                            wait(timeOut);
                            // Excelsior, LLC. Memory leak patch begin
 /* original code
                             if (pool.empty()) {
                             */
                            if (pool.isEmpty()) {
                                // Excelsior, LLC. Memory leak patch begin
                                DODS.getLogChannel().write(Logger.EMERGENCY,
                                        "ConnectionAllocator: "
                                        + "allocation of a new connection timed out."
                                        + "Possible dead lock avoided.");
                                String  msg = "Connections are currently unavailable.\n"
                                        + "Possible dead lock avoided.";

                                throw new SQLException (msg);
                            }
                        } else {
                            wait();
                        }
                    } catch (InterruptedException  intEx) {}
                }
            }
            //
 // A connection is available.
 //
 // Excelsior, LLC. Memory leak patch begin
 /* original code
             conn = (DBConnection)pool.pop();
             */
            // check
 try {
                conn = (DBConnection) pool.removeFirst();
            } catch (NoSuchElementException  e) {}
            if (connectionUsageCounter != null) {
                Integer  connUsages = (Integer ) connectionUsageCounter.get(conn);

                if (connUsages != null && connUsages.intValue() > 0) {
                    // decrease connection usage counter
                    connectionUsageCounter.put(conn,
                            new Integer (connUsages.intValue() - 1));
                } else {                     // the usages limit is over, close the connection
                    conn.close();
                    currentPoolSize--;
                    connectionUsageCounter.remove(conn);
                    DODS.getLogChannel().write(Logger.DEBUG,
                            "ConnectionAllocator: connection closed due to usage counter. currentPoolSize="
                            + currentPoolSize + "\n");
                    conn = null;
                }
            }
            if ((conn!=null) && (conn.getConnection().isClosed())){
                conn.close();
                currentPoolSize--;
                if (connectionUsageCounter!=null && connectionUsageCounter.contains(conn)){
                    connectionUsageCounter.remove(conn);
                }
                DODS.getLogChannel().write(Logger.DEBUG,
                        "ConnectionAllocator: Inactiv connection closed due allocate() operation. Geting new one. currentPoolSize="
                        + currentPoolSize + "\n");
                conn = null;
            }
            if((conn != null)&&(connectionIdileTimeout>0)){
                if((System.currentTimeMillis()
                        -((ExtendedDBConnection)conn).getConnectionEnterPoolTime())
                                >connectionIdileTimeout){
                        conn.close();
                        currentPoolSize--;
                        DODS.getLogChannel().write(Logger.DEBUG,
                                "ConnectionAllocator: Connection closed due allocate() operation - long connection idile time. Geting new one. currentPoolSize="
                                + currentPoolSize + "\n");
                        conn = null;
                }
            } 
        }
        // Excelsior, LLC. Memory leak patch end
 // DBConnection conn = (DBConnection)pool.pop();
        conn.allocate();
        return conn;
    }

    /**
     * Return a connection to the pool.  If it is of an old generation,
     * close and drop.
     *
     * @param dbConnection The connection object to return.
     */
    public synchronized void release(DBConnection dbConnection) {
        try {
            if ( (dbConnection.getGeneration() < generation)
                || (dbConnection.isMarkedForDrop())
                || (dbConnection.getConnection().isClosed())
                || (pool.size() >=maxWaitingConnections) ) {
                    dbConnection.close();
                    dbConnection = null;
                    currentPoolSize--;
            }
            else {
                // Excelsior, LLC. Memory leak patch begin
 /* original code
                 pool.push (dbConnection);
                 */
                 if (connectionIdileTimeout>0){
                    ((ExtendedDBConnection)dbConnection).setConnectionEnterPoolTime(System.currentTimeMillis());
                 }
                pool.addLast(dbConnection);
                // Excelsior, LLC. Memory leak patch end
            }
        }
        catch (SQLException  ex) {
            DODS.getLogChannel().write(Logger.DEBUG,"Eror relasing connection");
        }
        notify();
    }

    /**
     * Called when a connection in this pool has an SQL error.
     * All unallocated connections in the pool are dropped if
     * they have a generation number less
     * than or equal to the error connection.
     * If the current generation number
     * is the same as this generation number, increment it.
     * This way so that all outstanding connections
     * (including the one passed in) of the same generation
     * are dropped when returned.
     *
     * @param dbConnection The connection object to drop.
     *   The connection should be
     *   returned to the pool after this function returns.
     */
    public synchronized void drop(DBConnection dbConnection) {
        if (generation <= dbConnection.getGeneration()) {
            generation++; // new generation
        }
        // Delete all entries of the last generation.
 // Excelsior, LLC. Memory leak patch begin
 /* original code
         Stack holdPool = new Stack ();
         while (!pool.empty ()) {
         DBConnection connect = (DBConnection) pool.pop ();
         if (connect.getGeneration () < generation) {
         connect.close ();
         currentPoolSize--;
         } else {
         holdPool.push (connect);
         }
         }
         // Put all current generation entries back in the pool.
         while (!holdPool.empty ()) {
         pool.push (holdPool.pop ());
         }
         */
        LinkedList  newPool = new LinkedList ();

        try {
            while (!pool.isEmpty()) {
                DBConnection connect = (DBConnection) pool.removeFirst();

                if (connect.getGeneration() < generation) {
                    connect.close();
                    currentPoolSize--;
                } else {
                    newPool.addLast(connect);
                }
            }
        } catch (NoSuchElementException  e) {}
        // replace the original pool with the one that contains
 // only current generation entries
        pool = newPool;
        // Excelsior, LLC. Memory leak patch end
        notify();
    }

    /**
     * Called when the database manager is shutting down:
     * Close all connections immediately.
     */
    public synchronized void dropAllNow() {
        // Excelsior, LLC. Memory leak patch begin
 /* original code
         while (!pool.empty ()) {
         DBConnection connect = (DBConnection) pool.pop ();
         connect.close ();
         currentPoolSize--;
         }
         */
        
        if (shutDownStr!=null) {
            try {
                DBConnection tmpConn = allocate();
                tmpConn.execute(shutDownStr);
                tmpConn.release();
            } catch (SQLException  e1) {}
        }
        try {
            while (!pool.isEmpty()) {
                DBConnection connect = (DBConnection) pool.removeFirst();

                connect.close();
                currentPoolSize--;
            }
        } catch (NoSuchElementException  e) {}
        // Excelsior, LLC. Memory leak patch begin
    }

    /**
     * Return the number of currently active connections.
     *
     * @return The number of connections.
     */
    public int getActiveCount() {
        return currentPoolSize;
    }

    /**
     * Return the maximum number of connections active at one time.
     *
     * @return The number of connections.
     */
    public int getMaxCount() {
        return biggestPoolSize;
    }

    /**
     * Return the time when the maximum connection count occured.
     *
     * @return The <CODE>Date</CODE> when the maximum connection
     *  count occured.
     */
    public Date  getMaxCountDate() {
        return biggestPoolDate;
    }

    /**
     * Reset the maximum connection count and date.
     */
    public void resetMaxCount() {
        biggestPoolSize = currentPoolSize;
        biggestPoolDate = new Date ();
    }

    /**
     * Return the number of database requests.
     *
     * @return The number of database requests (queries or transactions).
     */
    public long getRequestCount() {
        return numRequests;
    }

    /**
     * Finalizer.
     * If any connections allocated by this object have not been closed,
     * this method ensures that garbage collection does so.
     */
    protected void finalize() {
        dropAllNow();
    }

    /**
     * @return database name of current connection
     *
     */
    public String  getDatabaseName() {
        return logicalDatabase.getName();
    }
    
    public void IncrementRequesteCount(){
        numRequests++;
    }
}
