
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *  Mark Diekhans  
 *  Kyle Clark     
 *  Paul Morgan    
 *  Predrag Djojic 
 *
 * $Id: StandardConnectionAllocator.java,v 1.8 2005/05/26 08:08:10 predrag Exp $
 */
package com.lutris.appserver.server.sql.standard;

import java.lang.reflect.Constructor ;
import java.sql.SQLException ;
import java.util.Date ;
import java.util.LinkedList ;
import java.util.NoSuchElementException ;

import org.enhydra.dods.CommonConstants; 
import org.enhydra.dods.DODS;

import com.lutris.appserver.server.sql.AbstractDBConnectionFactory;
import com.lutris.appserver.server.sql.ConnectionAllocator;
import com.lutris.appserver.server.sql.DBConnection;
import com.lutris.appserver.server.sql.ExtendedConnectionAllocator;
import com.lutris.appserver.server.sql.ExtendedDBConnection;
import com.lutris.appserver.server.sql.LogicalDatabase;
import com.lutris.appserver.server.sql.StandardDBConnectionFactory;
import com.lutris.logging.Logger;
import com.lutris.util.Config;
import com.lutris.util.ConfigException;
import com.lutris.util.KeywordValueException;

/**
 * Manages a pool (set) of connections to a database. The pool is owned by 
 * logical database. A connection considered part of the pool, even if its 
 * allocated to a thread. These objects are all publicly accessed via the 
 * Database Manager, not directly.
 * <P>
 * If an error occurs in a connection, it is dropped from the pool. 
 * A generation number is used to close down all connections that were open 
 * when the error occured, allowing new connections to be allocated.
 * <P>
 * The configuration data is specified in the section: <B><CODE>
 * DatabaseManager.DB. <I>dbName </I>.Connection</CODE> </B>
 * <P>
 * <I>Configuration sub fields are: </I>
 * <UL>
 * <LI><B><CODE>Url</CODE> </B>- The JDBC URLof the database. Manditary.
 * E.g. "jdbc:sequelink://dbHost:4000/[Informix];Database=dummy"
 * <LI><B><CODE>User</CODE> </B>- The database users used to access the
 * database. Manditary.
 * <LI><B><CODE>Password</CODE> </B>- The database user's password.
 * Manditary.
 * <LI><B><CODE>MaxPoolSize</CODE> </B>- The maximum number of open
 * connections to the database. Optional, if not specified, then it default to
 * 0. A value of 0 means that connections are allocated indefinitely or until
 * the database (JDBC) refuses any new connections.
 * <LI><B><CODE>Logging</CODE> </B>- Specify true to enable SQL logging,
 * false to disable it. Optional, false if not specified.
 * <LI><B><CODE>AllocationTimeout</CODE> </B>- The Maximum amount of time
 * that a thread will wait for a connection from the connection allocator before
 * an exception is thrown. This will prevent possible dead locks. The time out
 * is in milliseconds. If the time out is <= 0, the allocation of connections
 * will wait indefinitely. Optional, if not specified, then it defaults to 1000
 * (ms).
 * <LI><B><CODE>QueryTimeout</CODE> </B>- The amount of time (in seconds)
 * that a query will block before throwing an exception. If <= 0 then the query
 * will not block. Optional, if not specified, then the value defaults to 0.
 * This is not implemented by all logical databases.
 * <LI><B><CODE>TransactionTimeout</CODE> </B>- The amount of time (in
 * seconds) that a transaction will block before throwing an exception. If <= 0
 * then the transaction will not block. Optional, if not specified, then the
 * value defaults to 0. This is not implemented by all logical databases.
 * <LI><B><CODE>MaxPreparedStatements</CODE> </B>- If specified, overrides
 * the JDBC <CODE>Connection.getMetaData().getMaxStatements()</CODE> value. If
 * less than zero, use the meta data value. Optional, default is to use the meta
 * data.
 * </UL>
 * 
 * @since   LBS1.8
 * @version $Revision: 1.8 $
 * 
 */


public class StandardConnectionAllocator implements ExtendedConnectionAllocator {

    /**
     * Reference to the logical database for easy access to the connection pool.
     */
    protected LogicalDatabase logicalDatabase = null;

    /**
     * JDBC URL of database.
     */
    protected String  url;

    /**
     * SQL user name.
     */
    protected String  user;

    /**
     * SQL password..
     */
    protected String  password;

    /**
     * database "ShutDown" string 
     */
    protected String  shutDownStr;
    
    /**
     * Maximum number of connections in the pool. If this value is <= zero, then
     * create as many connections as possible.
     */
    private int maxPoolSize;

    /**
     * Current size of the pool; includes allocated connections.
     */
    private int currentPoolSize;

    /**
     * Maximum size the pool ever got to, regardless of generation.
     */
    private int biggestPoolSize;

    /**
     * Date at which the biggest pool size occured.
     */
    private Date  biggestPoolDate;

    /**
     * Number of queries or transactions on this logical database.
     */
    protected long numRequests;

    /**
     * The actual pool of DBConnection objects ready for allocate.
     */
    private ConnectionPool allocatePool;

    /**
     * Maximum time that connection will wait in connection pool before they
     * will be closed;
     */
    private long connectionIdileTimeout = -1;

    /**
     * The dealloate pool.
     */
    private ConnectionPool releasePool;

    private int requestCounter = 0;

    private int waitCounter = 0;

    private CountMonitor waitConnectionMonitor;

    private Object  allocateMonitor;

    private Object  releaseMonitor;

    /**
     * JP Patch Maximum number of DBConnections that will be waiting in the pool
     */
    protected int maxWaitingConnections;

    protected int initWaitingConnections;

    /**
     * Maximum number of times a particular connection is allowed to be
     * allocated before it is closed and replaced with a new one. If this value
     * is <= zero, then the number of allocations is unlimited.
     */
    private int maxConnectionUsages = -1;

 
    /**
     * Indicates if logging is enabled.
     */
    protected boolean sqlLogging;

    /**
     * Maximum amount of time in milliseconds to wait for a connection.
     */
    private int timeOut;

    /**
     * Maximum amount of time in seconds to block on a query. The DBQuery object
     * will retrieve this value from the connection.
     */
    protected int queryTimeOut;

    /**
     * Maximum amount of time in seconds to block on a transaction. The
     * DBTransaction object will retrieve this value from the connection.
     */
    protected int transactionTimeOut;

    /**
     * Maximum number of prepared statements to use; if less-than zero, then
     * JDBC is queried for this value.
     */
    protected int maxPreparedStatements;

    /**
     * Generation number. When an SQL error occurs, all objects of the same
     * generation or earlier are dropped.
     */
    protected int generation = 1;

    
    /**
     * Internal ConnectionPool class, encapsulate LinkedList 
     * with addition of drop counter.
     */
    class ConnectionPool {

        int dropCount = 0;

        LinkedList  pool;

        /**
         *  
         */
        public ConnectionPool() {
            pool = new LinkedList ();

        }

    }
    
    
    /**
     * Internal CountMonitor class, implements 
     * Monitor synchronization mechanism.
     */
    class CountMonitor {

        int waitCounter = 0;

        int notifyCounter = 0;

        int maxPoolSize = 0;

        public CountMonitor(int pSize) {
            maxPoolSize = pSize;
        }

        public synchronized void countNotify(int maxNotify) {
            if ((maxPoolSize == 0) && (waitCounter <= 0)) {
                if (notifyCounter < maxNotify) {
                    notifyCounter++;
                } else {
                    notifyCounter = maxNotify;
                }
            } else if ((notifyCounter < maxPoolSize) && (waitCounter <= 0)) {
                if (notifyCounter < maxNotify) {
                    notifyCounter++;
                } else {
                    notifyCounter = maxNotify;
                }
            }
            this.notify();
        }

        public synchronized void countWait() throws InterruptedException  {
            if (notifyCounter > 0) {
                notifyCounter--;
                if ((waitCounter > 0) && (notifyCounter > 0)) {
                    int countX = waitCounter;
                    if (waitCounter > notifyCounter) {
                        countX = notifyCounter;
                    }
                    for (int i = 0; i < countX; i++) {
                        this.notify();
                    }
                }
            } else {
                waitCounter++;
                try {
                    this.wait();
                } finally {
                    waitCounter--;
                    if ((waitCounter > 0) && (notifyCounter > 0)) {
                        int countX = waitCounter;
                        if (waitCounter > notifyCounter) {
                            countX = notifyCounter;
                        }
                        for (int i = 0; i < countX; i++) {
                            this.notify();
                            notifyCounter--;
                        }
                    }

                }
            }
        }

        public synchronized long countWait(long timeout) throws InterruptedException  {
            long ret = timeout;
            if (notifyCounter > 0) {
                notifyCounter--;
                if ((waitCounter > 0) && (notifyCounter > 0)) {
                    int countX = waitCounter;
                    if (waitCounter > notifyCounter) {
                        countX = notifyCounter;
                    }
                    for (int i = 0; i < countX; i++) {
                        this.notify();
                        notifyCounter--;
                    }
                }
                return ret;
            } else {
                ret = System.currentTimeMillis();
                waitCounter++;
                try {
                    this.wait(timeout);
                }catch (Exception  e) {
                    e.printStackTrace();
                } finally {
                    waitCounter--;
                    ret = timeout - (System.currentTimeMillis() - ret);
                    if (ret < 0) {
                        ret = 0;
                    }
                    if ((notifyCounter > 0) && (ret > 0)) {
                        notifyCounter--;
                    }
                    if ((waitCounter > 0) && (notifyCounter > 0)) {
                        int countX = waitCounter;
                        if (waitCounter > notifyCounter) {
                            countX = notifyCounter;
                        }
                        for (int i = 0; i < countX; i++) {
                            this.notify();
                            notifyCounter--;
                        }
                    }
                    //-dp: return ret;
                }
                return ret; //+dp:
            }
        }
    }

    /**
     * Internal ConnectionFactory object. 
     */
    private AbstractDBConnectionFactory dbConnectionFactory = null;

    
    /**
     * Internal ConnectionFactory class name.
     */
    private String  dbConnectionFactoryName = null;

    
    
    /**
     * Creates Internal ConnectionFactory object as instance of factoryName class
     * @param factoryName ConnectionFactory class name.
     * @return ConnectionFactory object.
     */
    private AbstractDBConnectionFactory createDBConnectionFactory(String  factoryName) {
        Class  connectionFactoryClass = null;
        Constructor  connectionFactoryConstructor = null;
        Class [] methodTypes = {};
        Object [] methodArgs = {};
        AbstractDBConnectionFactory factory = null;
        if (factoryName != null) {
            try {
                connectionFactoryClass = Class.forName(factoryName);
                factory = (AbstractDBConnectionFactory) connectionFactoryClass.newInstance();
            } catch (Exception  e) {
                DODS.getLogChannel().write(
                                Logger.INFO,
                                "Faild to make Connection Factory :"
                                        + factoryName
                                        + " creating StandardDBConnectionFactory insted");
                factory = null;
            }
        }
        if (factoryName == null || factory == null) {
            factory = new StandardDBConnectionFactory();
        }
        return factory;
    }

    /**
     * Create a new connection in the pool.
     * 
     * @exception java.sql.SQLException
     *                If a SQL error occures.
     */
    protected DBConnection createConnection() throws java.sql.SQLException  {
        DBConnection dbConnection = dbConnectionFactory.createConnection(
                (ConnectionAllocator) this, url, user, password,
                 maxPreparedStatements, sqlLogging, generation);
        return dbConnection;
    }


    
    /**
     * Creates instance of StandardConnectionAllocator object.
     * @param logicalDatabase LogicalDatabase that is owner of this ConnectionAllocator object.
     * @param conConfig Connection allocator configuration object.
     * @throws ConfigException
     */
    public StandardConnectionAllocator(LogicalDatabase logicalDatabase,
            Config conConfig) throws ConfigException {
        this.logicalDatabase = logicalDatabase;
        try {
            url  = conConfig.getString("Url");
            user = conConfig.getString("User");
            password = conConfig.getString("Password");
            timeOut  = conConfig.getInt("AllocationTimeout", 1000);
            maxPoolSize  = conConfig.getInt("MaxPoolSize", 0);
            sqlLogging   = conConfig.getBoolean("Logging", false);
            queryTimeOut = conConfig.getInt("QueryTimeout", 0);
            shutDownStr  = conConfig.getString("ShutDownString",null);
            transactionTimeOut     = conConfig.getInt ("TransactionTimeout", 0);
            maxPreparedStatements  = conConfig.getInt ("MaxPreparedStatements", -1);
            maxConnectionUsages    = conConfig.getInt ("MaxConnectionUsages", -1);
            maxWaitingConnections  = conConfig.getInt ("MaxWaitingConnections", Integer.MAX_VALUE);
            initWaitingConnections = conConfig.getInt ("InitConnectionsPoolSize", -1);
            connectionIdileTimeout = conConfig.getLong("ConnectionIdleTimeout", -1);

            
            dbConnectionFactoryName = conConfig.getString(CommonConstants.CONNECTION_FACTORY,null);         
            dbConnectionFactory = createDBConnectionFactory(dbConnectionFactoryName);

        } catch (KeywordValueException except) {
            throw new ConfigException("Bad DatabaseManager.DB."
                    + logicalDatabase.getName()
                    + ".Connection section defined in config file.");
        }
        currentPoolSize = 0;

        allocatePool = new ConnectionPool();
        releasePool  = new ConnectionPool();

        waitConnectionMonitor = new CountMonitor(maxPoolSize);
        allocateMonitor = new Object ();
        releaseMonitor  = new Object ();

        biggestPoolSize = 0;
        biggestPoolDate = new Date ();
        numRequests = 0;
        
        try {
            initConnectionPool(initWaitingConnections);
        } catch (SQLException  e) {
        }
    }

    /**
     * Initially fills connection allocator pool with initWaitingConnections number of connections.
     * @param initWaitingConnections Number of connections that will be initially added to pool. 
     */
    private void initConnectionPool(int initWaitingConnections)
            throws SQLException  {
        if (initWaitingConnections > maxPoolSize) {
            initWaitingConnections = maxPoolSize;
        }
        if (initWaitingConnections > 0) {
            DBConnection newConnection;
            for (int i = 1; i <= initWaitingConnections; i++) {
                newConnection = createConnection();
                if (!newConnection.getConnection().isClosed()) {
                    if (maxConnectionUsages > 0) {
                        ((ExtendedDBConnection) newConnection)
                                .setConnectionUsageCounter(maxConnectionUsages);
                    } else {
                        ((ExtendedDBConnection) newConnection)
                                .setConnectionUsageCounter(Integer.MAX_VALUE);
                    }
                    allocatePool.pool.addLast(newConnection);
                    currentPoolSize++;
                    waitConnectionMonitor.countNotify(initWaitingConnections);
                    if (currentPoolSize > biggestPoolSize) {
                        biggestPoolSize = currentPoolSize;
                        biggestPoolDate = new Date ();
                    }
                }
            }
        }
    }

    
    /** 
     * Allocate new connection from pool. 
     * @see com.lutris.appserver.server.sql.ConnectionAllocator#allocate()
     */
    public DBConnection allocate() throws SQLException  {
        boolean createNewConn = true;
        boolean connected = false;
        DBConnection conn = null;
        long enterWaitTime = 0;
        long timeToWait = -1;
        boolean newConn = false;
        if (timeOut > 0) {
            timeToWait = timeOut;
        }
        while (conn == null) {
            connected = false;
            newConn = false;
            synchronized (allocateMonitor) {
                if (allocatePool.pool.isEmpty()) {
                    switchPools();
                }
                if (allocatePool.pool.isEmpty()) {
                    if (createNewConn && ((currentPoolSize < maxPoolSize) || (maxPoolSize <= 0))) {
                        try {
                            conn = createConnection();
                            if (maxConnectionUsages > 0) {
                                ((ExtendedDBConnection) conn)
                                        .setConnectionUsageCounter(maxConnectionUsages);
                            } else {
                                ((ExtendedDBConnection) conn)
                                        .setConnectionUsageCounter(Integer.MAX_VALUE);
                            }
                            currentPoolSize++;
                            newConn = true;
                            if (currentPoolSize > biggestPoolSize) {
                                biggestPoolSize = currentPoolSize;
                                biggestPoolDate = new Date ();
                            }
                        } catch (SQLException  e) {
                            conn = null;
                            if (currentPoolSize > 0) {
                                DODS.getLogChannel().write(
                                                Logger.INFO,
                                                "ConnectionAllocator: "
                                                        + "failed to allocate a new connection due to"
                                                        + e.toString()
                                                        + "Error code: "
                                                        + e.getErrorCode()
                                                        + "\n"
                                                        + "SQLState: "
                                                        + e.getSQLState()
                                                        + "\n"
                                                        + "\nCurrent pool size is: "
                                                        + currentPoolSize
                                                        + "\nMaximum configured pool size is now "
                                                        + maxPoolSize
                                                        + "\nContinuing...\n");
                                createNewConn = false;
                            } else {
                                DODS.getLogChannel().write(
                                                Logger.EMERGENCY,
                                                "ConnectionAllocator: "
                                                        + "failed to allocate a new connection"
                                                            + "\nThe connection pool is empty!\n"
                                                            + e.toString()
                                                            + "Error code: "
                                                            + e.getErrorCode()
                                                            + "\n"
                                                            + "SQLState: "
                                                            + e.getSQLState()
                                                            + "\n"
                                                            + "\nCurrent pool size is: "
                                                            + currentPoolSize
                                                            + "\nMaximum configured pool size is now "
                                                            + maxPoolSize
                                                            + "\nContinuing...\n");
                                throw e;
                            }
                        }
                    }
                }
                try {
                    if (conn == null) {
                        conn = (ExtendedDBConnection) allocatePool.pool.removeFirst();
                    }
                } catch (NoSuchElementException  e) {
                }

                if (conn != null) {
                    connected = true;
                    if (maxConnectionUsages > 0) {
            int connUsages = ((ExtendedDBConnection)conn).getConnectionUsageCounter();
                        if (connUsages > 0) {
                            ((ExtendedDBConnection) conn)
                                 .setConnectionUsageCounter(--connUsages);
                        } else {
                            // the usages limit is over, close the connection
                            conn.close();
                            currentPoolSize--;
                            DODS.getLogChannel().write(
                                    Logger.DEBUG,
                                    "ConnectionAllocator: connection closed due to usage counter. currentPoolSize="
                                            + currentPoolSize + "\n");
                            conn = null;
                        }
                    }
                    if (conn != null) {
                        if (((ExtendedDBConnection) conn).isDroped()
                                || conn.getConnection().isClosed()
                                || (conn.getGeneration() < generation)
                                || conn.isMarkedForDrop()) {
                            conn.close();
                            currentPoolSize--;
                            DODS.getLogChannel().write(
                                            Logger.DEBUG,
                                            "ConnectionAllocator: Inactiv connection closed due allocate() operation. Geting new one. currentPoolSize="
                                                    + currentPoolSize + "\n");
                            conn = null;
                        }
                    }
                    if ((conn != null) && (connectionIdileTimeout > 0)) {
                        if ((System.currentTimeMillis() - ((ExtendedDBConnection) conn)
                                .getConnectionEnterPoolTime()) > connectionIdileTimeout) {
                            conn.close();
                            currentPoolSize--;
                            DODS.getLogChannel().write(
                                            Logger.DEBUG,
                                            "ConnectionAllocator: Connection closed due allocate() operation - long connection idile time. Geting new one. currentPoolSize="
                                                    + currentPoolSize + "\n");
                            conn = null;
                        }
                    }
                }
            }
            if (conn == null) {
                try {
                    if (timeToWait == 0) {
                        DODS.getLogChannel().write(
                                        Logger.EMERGENCY,
                                        "ConnectionAllocator:"
                                                + "allocation of a new connection timed out."
                                                + "Possible dead lock avoided.");
                        String  msg = "Connections are currently unavailable.\n"
                                + "Possible dead lock avoided.";
                        throw new SQLException (msg);

                    } else if (timeToWait > 0) {
                        timeToWait = waitConnectionMonitor.countWait(timeToWait);
                    } else {
                        waitConnectionMonitor.countWait();
                    }

                } catch (InterruptedException  intEx) {
                }
            }
        }
        conn.allocate();
        return conn;
    }

    /**
     *  Mutually switch allocatePool with releasePool 
     */
    private void switchPools() {
        synchronized (releaseMonitor) {
            if (releasePool.pool.size() > allocatePool.pool.size()) {
                LinkedList  tmp = allocatePool.pool;
                allocatePool.pool = releasePool.pool;
                releasePool.pool = tmp;
            }
        }
    }

    /**
     * Release connection back to pool.
     * 
     * @see com.lutris.appserver.server.sql.ConnectionAllocator#release(com.lutris.appserver.server.sql.DBConnection)
     */
    public void release(DBConnection dbConnection) {
        if (dbConnection != null) {
            int releaseSize = 0;
            synchronized (releaseMonitor) {
                if (connectionIdileTimeout > 0) {
                    ((ExtendedDBConnection) dbConnection).setConnectionEnterPoolTime(System.currentTimeMillis());
                }
                releasePool.pool.addLast(dbConnection);
                releaseSize = releasePool.pool.size();
            }
            waitConnectionMonitor.countNotify(releaseSize);
        }

    }

    /**
     * Called when a connection in this pool has an SQL error.
     * All unallocated connections in the pool are dropped if
     * they have a generation number less
     * than or equal to the error connection.
     * If the current generation number
     * is the same as this generation number, increment it.
     * This way so that all outstanding connections
     * (including the one passed in) of the same generation
     * are dropped when returned.
     *
     * @param dbConnection The connection object to drop.
     *   The connection should be
     *   returned to the pool after this function returns.
     */
    public void drop(DBConnection dbConnection) {
        int releasedCounter = 1;
        LinkedList  newAllocatePool = new LinkedList ();
        LinkedList  newReleasePool = new LinkedList ();
        LinkedList  tmpPool;
        synchronized (allocateMonitor) {
            if (generation <= dbConnection.getGeneration()) {
                generation++; // new generation
            }
            synchronized (releaseMonitor) {
                tmpPool = releasePool.pool;
                releasePool.pool = newReleasePool;
            }
            try {
                while (!tmpPool.isEmpty()) {
                    DBConnection connect = (DBConnection) tmpPool.removeFirst();
                    if (connect.getGeneration() < generation) {
                        connect.close();
                        releasedCounter++;
                        currentPoolSize--;
                    } else {
                        newAllocatePool.addLast(connect);
                    }
                }
            } catch (NoSuchElementException  e) {
            }
            try {
                while (!allocatePool.pool.isEmpty()) {
                    DBConnection connect = (DBConnection) allocatePool.pool
                            .removeFirst();
                    if (connect.getGeneration() < generation) {
                        connect.close();
                        releasedCounter++;
                        currentPoolSize--;
                    } else {
                        newAllocatePool.addLast(connect);
                    }
                }
            } catch (NoSuchElementException  e) {
            }
            allocatePool.pool = newAllocatePool;
            dbConnection.close();
            currentPoolSize--;
        }
        for (int i = 0; i < releasedCounter; i++) {
            waitConnectionMonitor.countNotify(releasedCounter);
        }
    }

    /**
     * Called when the database manager is shutting down:
     * Close all connections immediately.
     */
    public void dropAllNow() {
        if (shutDownStr!=null) {
            try {
                DBConnection tmpConn = allocate();
                tmpConn.execute(shutDownStr);
                tmpConn.release();
            } catch (SQLException  e1) {}
        }
        synchronized (releaseMonitor) {
            synchronized (allocateMonitor) {
                try {
                    while (!allocatePool.pool.isEmpty()) {
                        DBConnection connect = (DBConnection) allocatePool.pool.removeFirst();
                        connect.close();
                        currentPoolSize--;
                    }
                } catch (NoSuchElementException  e) {}
            }
            try {
                while (!releasePool.pool.isEmpty()) {
                    DBConnection connect = (DBConnection) releasePool.pool.removeFirst();
                    connect.close();
                    currentPoolSize--;
                }
            } catch (NoSuchElementException  e) {}
        }
    }

    /**
     * Return the number of currently active connections.
     *
     * @return The number of connections.
     */
    public int getActiveCount() {
        return currentPoolSize;
    }

    /**
     * Return the maximum number of connections active at one time.
     *
     * @return The number of connections.
     */
    public int getMaxCount() {
        return biggestPoolSize;
    }

    /**
     * Return the time when the maximum connection count occured.
     *
     * @return The <CODE>Date</CODE> when the maximum connection
     *  count occured.
     */
    public Date  getMaxCountDate() {
        return biggestPoolDate;
    }

    /**
     * Reset the maximum connection count and date.
     */
    public void resetMaxCount() {
        biggestPoolSize = currentPoolSize;
        biggestPoolDate = new Date ();
    }

    /**
     * Return the number of database requests.
     *
     * @return The number of database requests (queries or transactions).
     */
    public long getRequestCount() {
        return numRequests;
    }

    /**
     * Finalizer.
     * If any connections allocated by this object have not been closed,
     * this method ensures that garbage collection does so.
     */
    protected void finalize() {
        dropAllNow();
    }

    /**
     * Database name of logicalDatabase that is owner of pool.
     * @return Database name of logicalDatabase that is owner of pool.
     */
    public String  getDatabaseName() {
        return logicalDatabase.getName();
    }

    
    /* 
     * Increments, numRequests, request counter
     */
    public void IncrementRequesteCount() {
        numRequests++;
    }
}