
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 * Contributor(s):
 *
 * $Id: StandardDBConnection.java,v 1.3 2005/03/24 13:44:51 predrag Exp $
 *
 * formatted with JxBeauty (c) johann.langhofer@nextra.at
 */
package  com.lutris.appserver.server.sql.standard;

import com.lutris.appserver.server.sql.*;
import java.sql.*;

import com.lutris.logging.Logger;
import java.util.Enumeration ;
import java.util.Hashtable ;
import org.enhydra.dods.Common;
import org.enhydra.dods.DODS;

/**
 * Standard implementation of the DBConnection object.
 *
 * @see         DBConnection
 * @author      Mark Diekhans
 * @author      Kyle Clark
 * @since       LBS1.7
 * @version     $Revision: 1.3 $
 */
public class StandardDBConnection implements ExtendedDBConnection {

    /** JDBC connection object. */
    protected Connection connection;

    /** Database URL, used for error messages. */
    protected String  url;

    /** Database user, for error messages. */
    protected String  user;

    /** Database user password. */
    protected String  password;
    // Identifier for this connection.
 protected int id;
    // Next available identifier for a connection.
 protected static int nextId = 0;
    // Table of prepared (compiled) statements, keyed by SQL.
 protected Hashtable  preparedStmtCache;
    // The last statement used by this connection.
 protected Statement currentStmt = null;
    // The connection allocator this connection is associated with.
 protected ConnectionAllocator connectionAllocator;
    // Indicates if logging is enabled.
 protected boolean logging;
    // Generation number assigned and used by StandardConnectionAllocator.
 protected int generation;
    // Has this connection been closed.
 protected boolean closed = false;
    // Do we need to drop this connection?
 protected boolean dropConnection = false;
    // Has this connection been dropped?
 protected boolean dropped = false;
    // Has this connectio been closed.
 protected boolean reset = true;
    // Is this connection currently in use.
 protected boolean allocated = false;
    // Maximum number of open statements.
 protected int maxPreparedStmts;
    // the number of times a particular DBConnection was allocated;
 protected int connectionUsageCounter=0;
    // Time when connection is released  to pool;
 protected long connectionEnterPoolTime = -1;
    /**
     * The log channel.
     */
    //    private LogChannel channel = DODS.getLogChannel();

    /**
     * Initialize the connection to a database.
     *
     * @param url JDBC URL of database.
     * @param user SQL user name.
     * @param password SQL password.
     * @param connectionAllocatorObj The connection allocator that this
     *  connection belongs to.
     * @param maxPreparedStatements Maximum number of preparse statements.
     *  a value of less than zero queries JDBC for the value.
     * @param logging Specifying <CODE>true</CODE> enables SQL logging.
     * @param generation A generation number used to drop old connection
     *  when they are released.
     * @exception java.sql.SQLException If a connection can't be established.
     */
    public StandardDBConnection(ConnectionAllocator connectionAllocatorObj,
                                String  url, String  user, String  password, int maxPreparedStatements,
                                boolean logging, int generation)
        throws SQLException {
        id = nextId++;
        this.preparedStmtCache = new Hashtable ();
        this.connectionAllocator = connectionAllocatorObj;
        this.url = url;
        this.user = user;
        this.password = password;
        this.logging = logging;
        this.generation = generation;
        connection = DriverManager.getConnection(url, user, password);
        currentStmt = connection.createStatement();
        if (maxPreparedStatements < 0) {
            // One UN-prepared statement is used up.
 this.maxPreparedStmts = connection.getMetaData().getMaxStatements()
                - 1;
            if (this.maxPreparedStmts == 0) {
                // Some (Microsoft) JDBC drivers return apparently incorrect
 // value of 1, so we set this to 1, instead of throwing an
 // error since it still works.
 this.maxPreparedStmts = 1;
            }
            if (this.maxPreparedStmts < 0) {
                // Take a wild guess here and assume that if
 // getMaxStatements returns '0', then it really
 // means 'unlimited'.  Assign an arbitrary number.
 this.maxPreparedStmts = 256;
            }
        } else {
            this.maxPreparedStmts = maxPreparedStatements;
        }
        logDebug("DBConnection[" + id + "]: " + url
                     + "\nNew connection allocated.\n");
        connectionEnterPoolTime = System.currentTimeMillis();
    }

    /**
     * Method called when this connection object is allocated from the
     * connection allocator.
     *
     * @exception java.sql.SQLException If <CODE>reset</CODE> had no
     *  been called on the previous operation.
     */
    public synchronized void allocate() throws SQLException {
        logDebug("allocate");
        try {
            closedCheck();
            reset();
        } catch (SQLException e) {
            handleException(e);
            //
 // If this connection is no longer valid we need
 // to drop it.
 //
 if (dropConnection) {
                drop();
            } else {
                release();
            }
            throw  e;
        }
        reset = false;
        allocated = true;
    }

    /**
     * Check to see that the previous database connection
     * has had reset called.  This ensures that the database
     * resources have always been freed in a timely manner.
     *
     * @exception java.sql.SQLException
     *   If <CODE>reset</CODE> has not
     *   been called on the previous operation.
     */
    protected void resetCheck() throws SQLException {
        logDebug("resetCheck()");
        if (!reset) {
            throw new java.sql.SQLException ("DBConnection.reset() was not called after the previous "
                                                 + "operation completed");
        }
    }

    /**
     * Check to see that this database connection has
     * has been properly allocated before is is used.
     *
     * @exception java.sql.SQLException
     *   If it was not properly allocated.
     */
    protected void allocatedCheck() throws SQLException {
        logDebug("allocatedCheck()");
        if (!allocated) {
            throw new java.sql.SQLException ("attempt to access connection which was released.");
        }
    }

    /**
     * Check to see that this database connection has
     * has not been closed.
     *
     * @exception java.sql.SQLException
     *   If it is closed.
     */
    protected void closedCheck() throws SQLException {
        logDebug("closedCheck()");
        if (closed || getConnection().isClosed()) {
            throw new java.sql.SQLException ("attempt to access a closed connection.");
        }
    }

    /**
     * Validates this connection.  Check to see that it
     * is not closed and has been properly allocated.
     *
     * @exception java.sql.SQLException
     *   If it is not valid.
     */
    public void validate() throws SQLException {
        logDebug("validate()");
        allocatedCheck();
        closedCheck();
    }

    /**
     * Closes down all query-specific resources.
     *
     * @exception java.sql.SQLException If a database error occurs.
     */
    public synchronized void reset() throws SQLException {
        // Try to cleanup everything, but save the first error.
        SQLException saveExcept = null;

        if (reset) {
            return;
        }
        logDebug("reset");
        try {
            connection.clearWarnings();
        } catch (SQLException except) {
            if (saveExcept == null) {
                saveExcept = except;
            }
        }
        try {
            connection.setAutoCommit(false);
        } catch (SQLException except) {
            if (saveExcept == null) {
                saveExcept = except;
            }
        }
        reset = true;
        //
 // Throw any exceptions that were generated.
 //
 if (saveExcept != null) {
            throw  saveExcept;
        }
    }

    /**
     * Get a prepared statement given an SQL string.  If the statement is
     * cached, return that statement, otherwise prepare and save in the
     * cache.
     *
     * @param sql The SQL statement to prepared.
     * @return The prepared statement, which is associated only with this
     *  connection and must not be used once the connection is released.
     * @exception java.sql.SQLException If a SQL error occured compiling the
     *  statement.
     */
    public synchronized PreparedStatement prepareStatement(String  sql)
        throws SQLException {
            return _prepareStatement(sql, -100, -100, false);
    }

    private synchronized PreparedStatement _prepareStatement(String  sql, int iResultSetType, int iResultSetConcurrency, boolean tuned)
        throws SQLException {
        PreparedStatement preparedStmt = null;
        String  mapKey = sql+"=+=+"+iResultSetType+"=+=+"+iResultSetConcurrency+"=+=+";
        if (tuned){
             mapKey=mapKey+"tuned"; 
        }
        logDebug("Prepare statement: " + sql);
        validate();
        boolean selectQuery = sql.trim().toLowerCase().startsWith("select");
        if(maxPreparedStmts>0 && !selectQuery){
            preparedStmt = (PreparedStatement) preparedStmtCache.get(mapKey);
            // PWFL-1574 skopiowane z SharkLibs
            /*
             * <b>JIRA #PWFL-1574</b>
             * <p>
             * Warunek {@code preparedStmt.getConnection().isClosed()} jest
             * niepoprawny, ponieważ już sama metoda {@link Statement#getConnection()} rzuca {@link SQLException} jeżeli ten
             * statement jest zamknięty.
             * </p>
             *
             * @author Cezary Kozar 18 lut 2015
             */
            // zmiana: (!preparedStmt.getConnection().isClosed()) -> (!preparedStmt.isClosed())
            if ((preparedStmt != null) && (!preparedStmt.isClosed())) {
                preparedStmt.clearParameters();
            } else {        
                if (preparedStmtCache.size() >= maxPreparedStmts) {
                    String  key = (String ) preparedStmtCache.keys().nextElement();
                    ((PreparedStatement) preparedStmtCache.remove(key)).close();
                }
                preparedStmt = getNewPrepStatemet(sql, iResultSetType, iResultSetConcurrency, tuned);
                preparedStmtCache.put(mapKey, preparedStmt);
            }
        }else{
            preparedStmt = getNewPrepStatemet(sql, iResultSetType, iResultSetConcurrency, tuned);
        }
        return  preparedStmt;
    }

    

    /**
     * @param sql The SQL statement to prepared.
     * @param iResultSetType a result set type; one of ResultSet.TYPE_FORWARD_ONLY, ResultSet.TYPE_SCROLL_INSENSITIVE, or ResultSet.TYPE_SCROLL_SENSITIVE.
     * @param iResultSetConcurrency a concurrency type; one of ResultSet.CONCUR_READ_ONLY or ResultSet.CONCUR_UPDATABLE.
     * @param tuned Showing is PrepareStatment user tuned or use default setings 
     * @return New PreparedStatement
     * @throws SQLException
     */
    private PreparedStatement getNewPrepStatemet(String  sql, int iResultSetType, int iResultSetConcurrency, boolean tuned) throws SQLException {
        PreparedStatement preparedStmt;
        if(!tuned){
            if(getResultSetType()==StandardLogicalDatabase.DEFAULT_RESULT_SET_TYPE ||
               getResultSetConcurrency()==StandardLogicalDatabase.DEFAULT_RESULT_SET_CONCURRENCY){
                preparedStmt = connection.prepareStatement(sql);
            }
            else {
                preparedStmt = connection.prepareStatement(sql,getResultSetType(),getResultSetConcurrency());
            }
        }else{
            preparedStmt = connection.prepareStatement(sql,iResultSetType,iResultSetConcurrency);
        }
        return preparedStmt;
    }

    /**
     * Get a prepared statement given an SQL string.  If the statement is
     * cached, return that statement, otherwise prepare and save in the
     * cache.
     *
     * @param sql The SQL statement to prepared.
     * @param iResultSetType a result set type; one of ResultSet.TYPE_FORWARD_ONLY, ResultSet.TYPE_SCROLL_INSENSITIVE, or ResultSet.TYPE_SCROLL_SENSITIVE.
     * @param iResultSetConcurrency a concurrency type; one of ResultSet.CONCUR_READ_ONLY or ResultSet.CONCUR_UPDATABLE.
     * @return The prepared statement, which is associated only with this
     *  connection and must not be used once the connection is released.
     * @exception java.sql.SQLException If a SQL error occured compiling the
     *  statement.
     */
    public synchronized PreparedStatement prepareStatement(String  sql, int iResultSetType, int iResultSetConcurrency)
        throws SQLException {
        return _prepareStatement(sql, iResultSetType, iResultSetConcurrency, true);
    }


    protected int getResultSetType(){

        int resultSetType;
        try {
            resultSetType =((StandardLogicalDatabase) DODS
                                .getDatabaseManager()
                                .findLogicalDatabase(getDatabaseName()))
                .getResultSetType();
        } catch (DatabaseManagerException e) {
            DODS.getLogChannel().write(Logger.DEBUG,"Error unknown logical database. Using default value for 'resultSetType' parameter");
            resultSetType = StandardLogicalDatabase.DEFAULT_RESULT_SET_TYPE;
        }
        return resultSetType;
    }

    protected int getResultSetConcurrency(){

        int resultSetConcurrency;
        try {
            resultSetConcurrency =((StandardLogicalDatabase) DODS
                                       .getDatabaseManager()
                                       .findLogicalDatabase(getDatabaseName()))
                .getResultSetConcurrency();
        } catch (DatabaseManagerException e) {
            DODS.getLogChannel().write(Logger.DEBUG,"Error unknown logical database. Using default value for 'resultSetConcurrency' parameter");
            resultSetConcurrency = StandardLogicalDatabase.DEFAULT_RESULT_SET_CONCURRENCY;
        }
        return resultSetConcurrency;
    }


    /**
     * Creates a CallableStatement object for calling database
     * stored procedures. Refer to jdk api refernece.
     *
     * @param sql The SQL statement to be called.
     * @return a new CallableStatement object containing the
     *  pre-compiled SQL statement.
     * @exception java.sql.SQLException If a database access error occurs
     *  statement.
     */
    public synchronized CallableStatement prepareCall(String  sql) throws SQLException {
        return  connection.prepareCall(sql);
    }

    /**
     * Execute a prepared query statement.  Once the query has completed,
     * <CODE>reset()</CODE> should be called.
     *
     * @param preparedStmt The statement to execute.
     * @param msg for logging/debug purposes
     * @return Query result set, do not call close, use reset(),
     * @exception java.sql.SQLException If a SQL error occured executing the
     *  statement.
     */
    public synchronized ResultSet executeQuery(PreparedStatement preparedStmt, String  msg)
        throws SQLException {
        logDebug("Execute prepared statement: " + msg, preparedStmt);
        validate();
        return  preparedStmt.executeQuery();
    }

    /**
     * Execute a SQL query statement.  This is a wrapper that adds logging.
     * Once the query has completed, <CODE>reset()</CODE> should be called.
     *
     * @param sql The SQL query statement
     * @return Query result set, do not call close, use reset(),
     * @exception java.sql.SQLException If a SQL error occured executing the
     *  statement.
     */
    public synchronized ResultSet executeQuery(String  sql) throws SQLException {
        logDebug(sql);
        validate();
        return  currentStmt.executeQuery(sql);
    }

    /**
     * Execute a SQL update statement.  This is a wrapper that adds logging.
     * Once the update has completed, <CODE>reset()</CODE> should be called.
     *
     * @param sql
     *   The SQL query statement
     * @return
     *   Either the row count for UPDATE, INSERT, DELETE or 0 for
     *   SQL statements that return nothing.
     * @exception java.sql.SQLException
     *   If a SQL error occured executing the update.
     */
    public synchronized int executeUpdate(String  sql) throws SQLException {
        logDebug(sql);
        validate();
        return  currentStmt.executeUpdate(sql);
    }

    /**
     * Execute a prepared update statement.  Once the update has completed,
     * <CODE>reset()</CODE> should be called.
     *
     * @param preparedStmt The statement to execute.
     * @param msg for logging/debug purposes
     * @return Either the row count for UPDATE, INSERT, DELETE or 0 for
     *   SQL statements that return nothing.
     * @exception java.sql.SQLException If a SQL error occured executing the
     *  statement.
     */
    public int executeUpdate(PreparedStatement preparedStmt, String  msg) throws SQLException {
        logDebug("Execute prepared statement: " + msg, preparedStmt);
        validate();
        return  preparedStmt.executeUpdate();
    }

    /**
     * Execute a SQL statement that does not return a resultset.  This is a
     * wrapper that adds logging.  Once the query has completed,
     * <CODE>reset()</CODE> should be called.
     *
     * @param sql The SQL query statement
     * @return True if the next result is a ResultSet; false if it is
     *   an update count or there are no more results.
     * @exception java.sql.SQLException If a SQL error occured executing the
     *  statement.
     */
    public synchronized boolean execute(String  sql) throws SQLException {
        logDebug("execute: " + sql);
        validate();
        return  currentStmt.execute(sql);
    }

    /**
     * Check for warnings in a result set.
     *
     * @param resultSet The result set to check for warnings.
     * @exception java.sql.SQLException If a SQL error occured compiling the
     *  statement.
     */
    public void warningCheck(ResultSet resultSet) throws SQLException {
        logDebug("warningCheck()");
        SQLWarning warning = resultSet.getWarnings();

        if (warning != null) {
            throw  warning;
        }
    }

    /**
     * Return this connection to its allocator.  This object should not be
     * used after calling this function.
     */
    public synchronized void release() {
        if (allocated) {
        logDebug("release");
            allocated = false;
            if (dropConnection) {
                drop();
            }
            if (!dropped){
                connectionAllocator.release(this);
            }
        }
    }

    /**
     * Drop this connection from the connection allocator.
     */
    protected synchronized void drop() {
        if (!dropped) {
            close();
            logDebug("drop");
            connectionAllocator.drop(this);
            dropped = true;
            logDebug("DBConnection[" + id + "]: " + url
                         + "Connection has been dropped from the connection allocator.");
        }
    }

    /**
     * Check if a connection is valid after an SQL exception
     * is thrown.  If it is not usable, then the connection is
     * dropped from the connection allocator and closed.  The connection
     * is then no longer usable.
     *
     * @param sqlExcept
     *   The SQL exception that occured.
     * @return true if the exception does not affect this
     *   connection object.  False otherwise - in which case
     *   this connection object is no longer usable.
     */
    public synchronized boolean handleException(SQLException sqlExcept) {
        String  sqlState = sqlExcept.getSQLState();

        logDebug("handleException: " + sqlExcept.getMessage());
        //
 // TODO
 // Need to query complete set of fatal exceptions,
 // or complete set of non-fatal exceptions.
 // Until then we assume all are fatal.
 // Note: that code '11111111' is an internal exception
 // and does not require the connection pool to be
 // reset. This is a temporary fix. (paul)
 // Note: There may be a bug in the pool closure logic
 // if connections in the pool repeatidly throw exceptions
 // that close the pool while the pool was already in the
 // process of being closed. (paul)
 //
 if (sqlExcept.getErrorCode() != 11111111) {
            if (!dropConnection) {
                logDebug("DBConnection[" + id + "]: " + url
                             + "\nScheduled to be dropped from connection allocator."
                             + "\nUnable to handle exception: \""
                             + sqlExcept.toString() + " \nErrorCode : "
                             + sqlExcept.getErrorCode() + " \nSQLState : "
                             + sqlExcept.getSQLState() + "\"\n");
                //
 // We need to drop this connection.  N.B. We only
 // set a flag that we want the connection dropped.
 // We don't want the connection dropped yet (which
 // implies closed()) because we may be in the
 // middle of a transaction which we may want to
 // rollback.
 //
                dropConnection = true;
            }
            return false;
        } else {
            return true;
        }
    }

    /**
     * Get the generation number specified when the connection was created.
     *
     * @return The generation number.
     */
    public int getGeneration() {
        return  generation;
    }

    /**
     * Close this connection.  Use by the connection allocator when this
     * object is no longer used.  Errors are ignored.
     */
    public synchronized void close() {
        if (!closed) {
            logDebug("close");

            /*
             * If we are closing the connection, then
             * the prepared statements should automatically
             * be closed.  Of course, the doc isn't clear.
             * So we attempt to close the statements.
             * If any fail, then we ignore the rest.
             * The reason we do this is that if the original
             * connection is broken, then the closing of
             * each statement needs to time out.  This
             * is very time consuming.....
             */
            boolean closeStmts = true;
            //
 // Close the prepared statements.
 //
 /**/
            Enumeration  e = preparedStmtCache.keys();

            while (e.hasMoreElements() && closeStmts) {
                String  key = (String ) e.nextElement();

                try {
                    ((PreparedStatement) preparedStmtCache.remove(key)).close();
                } catch (SQLException except) {
                    // Ignore errors, we maybe handling one.
                    closeStmts = false;
                    logDebug("DBConnection[" + id + "]: " + url
                                 + "\nUnable to close statements.  Continuing....\n");
                }
            }
            /**/
            if (closeStmts) {
                try {
                    if (currentStmt != null) {
                        currentStmt.close();
                    }
                } catch (Exception  except) {
                    // Ignore errors, we maybe handling one.
                    closeStmts = false;
                }
                currentStmt=null;
            }
            try {
                if(!connection.isClosed()){
                    connection.close();
                }
            } catch (Exception  except) {// Ignore errors, we maybe handling one.
            }
            logDebug("DBConnection[" + id + "]: " + url
                         + "\nConnection has been closed.\n");
        }
        closed = true;
        connection = null;
    }

    /**
     * Autocommit on/off.
     *
     * @param on - False to disable auto commit mode.  True to enable.
     * @exception java.sql.SQLException If a database access error occurs.
     */
    public void setAutoCommit(boolean on) throws SQLException {
        validate();
        if(Common.isChangeAutocommitEnabled(getDatabaseName())){
            logDebug("set autocommit: " + on);
            connection.setAutoCommit(on);
        }else{
            logDebug("set autocommit is disabled, can't change to value :" + on);
        }
    }

    /**
     * Commit a transaction.
     *
     * @exception java.sql.SQLException If a database access error occurs.
     */
    public void commit() throws SQLException {
        validate();
        logDebug("commit");
        connection.commit();
    }

    /**
     * Rollback a transaction.  Should only be used when
     * <A HREF=#setAutoCommit>auto-commit</A> mode
     * has been disabled.
     *
     * @exception java.sql.SQLException If a database access error occurs.
     */
    public void rollback() throws SQLException {
        validate();
        logDebug("rollback");
        connection.rollback();
    }

    /**
     * Debug logging.
     *
     * @param str
     *   The data to log.
     */
    protected void logDebug(String  str) {
        if (logging) {
            DODS.getLogChannel().write(Logger.DEBUG,
                                       "\nDBConnection[" + id + "]:" + str + "\n");
        }
    }

    /**
     * Debug logging.
     *
     * @param str
     *   The data to log.
     * @param stmt
     *   The statement to log.
     */
    protected void logDebug(String  str, Statement stmt) {
        if (logging) {
            DODS.getLogChannel().write(Logger.DEBUG,
                                       "\nDBConnection[" + id + "]:" + str + "\n" + "SQL: "
                                           + stmt.toString());
        }
    }

    /**
     * Increment the count of the number of requests against
     * this connection.
     */
    public void incrRequestCount() {
        ((ExtendedConnectionAllocator) connectionAllocator).IncrementRequesteCount();
    }

    /**
     * Get the database URL.
     * @return The database URL.
     */
    public String  getUrl() {
        return  url;
    }

    /**
     * Get the database user name.  Normally user for error messages.
     * @return The database user name.
     */
    public String  getUser() {
        return  user;
    }

    /**
     * Get the underlying <code>Connection</code> object.
     * Use with extreme caution.
     * @return the connection object
     */
    public Connection getConnection() {
        return  connection;
    }

    /**
     * @return true if this connection is marked to be dropped out
     * of the connection pool and closed.
     */
    public boolean isMarkedForDrop() {
        return  dropConnection;
    }

    /**
     * @return database name of current connection
     *
     */
    public String  getDatabaseName() {
        return connectionAllocator.getDatabaseName();
    }
    /**
     * @return connectionUsageCounter
     */
    public int getConnectionUsageCounter() {
        return connectionUsageCounter;
    }

    /**
     * @param i new value for connectionUsageCounter
     */
    public void setConnectionUsageCounter(int i) {
        connectionUsageCounter = i;
    }
    /**
     * @return dropped
     */
    public boolean isDroped() {
        return dropped;
    }
    /**
     * @return dropped
     */
    public boolean isClosed() {
        return closed;
    }

    /* (non-Javadoc)
     * @see com.lutris.appserver.server.sql.ExtendedDBConnection#setConnectionEnterPoolTime(int)
     */
    public void setConnectionEnterPoolTime(long i) {
        connectionEnterPoolTime=i;

    }

    /* (non-Javadoc)
     * @see com.lutris.appserver.server.sql.ExtendedDBConnection#getConnectionEnterPoolTime()
     */
    public long getConnectionEnterPoolTime() {
        return connectionEnterPoolTime;
    }

    /**
     * @return Returns the maxPreparedStmts.
     */
    public int getMaxPreparedStmts() {
        return maxPreparedStmts;
    }
    
    public long curtime=0;

    public boolean _statementCacheFixed() {
        return true;
    }

}
