
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package org.enhydra.dods.cache;

import java.util.HashMap ;
import java.util.Map ;

import org.enhydra.dods.DODS;
import org.enhydra.dods.cache.base.BaseCacheManager;
import org.enhydra.dods.cache.base.DODSCache;
import org.enhydra.dods.exceptions.CacheObjectException;
import org.enhydra.dods.statistics.CacheStatistics;
import org.enhydra.dods.statistics.Statistics;
import org.enhydra.dods.statistics.TableStatistics;

import com.lutris.appserver.server.sql.CoreDataStruct;
import com.lutris.appserver.server.sql.standard.DatabaseConfiguration;
import com.lutris.appserver.server.sql.standard.StandardLogicalDatabase;
import com.lutris.util.Config;

/**
 * This class contains data and mechanisms needed for caching data objects (or
 * DataStruct objects) by their OIDs and provides cache configuration and
 * administration.
 *
 * @author    Tanja Jovanovic
 * @author    Nenad Vico
 * @author    Zorica Suvajdzin
 * @version   2.0  15.06.2003.
 */
public class DataStructCacheImpl extends DataStructCache {

    /**
     * LRU cache for storing data (or DataStruct) objects. LRU cache keys are
     * cache handles (String "<database_name>.<String_presentation_of_oid>"),
     * and LRU cache values are data (or DataStruct) objects.
     */
    protected DODSCache cache = null;

    /**
     * cacheAdministration attribute handles configuration settings about data
     * (or DataStruct) object cache.
     */
    protected CacheAdministration cacheAdministration = null;

    /**
     * TableConfiguration attribute handles configuration settings about table
     * of this cache.
     */
    protected TableConfiguration tableConf = new TableConfiguration();

    /**
     * Initial query statement. This attribute contains "where" clause which is
     * used during data (or DataStruct) object cache initialization.
     * If this attribute is set to "*", all rows from the table (in database)
     * will be put in the cache.
     */
    protected String  initialQueryCache = null;

    /**
     * This attribute contains information if multi databases are used.
     */
    protected boolean multi = false;

    /**
     * This attribute is true if the cache is full, otherwise false.
     */
    protected boolean fullCachingOn = false;
    
    /**
     * Table and cache statictics.
     */
    protected Statistics statistics = null;

    /**
     * List of objects that are unvisible in the cache at the moment. Keys of
     * this map sre cache handles
     * (String "<database_name>.<String_presentation_of_oid>"),
     * and values are counters of how many transactions have made object with
     * that cache handle invisible.
     */
    protected HashMap  nonVisibleList = null;

    /**
     * Reserve factor used in in query caching. Since there is only data object
     * (or DataStruct object) cache, without query caches, this parameter is not
     * used.
     */
    protected double reserveFactor = 0;
    protected double cachePercentage = -1;

    /**
     * True if the cache is disabled, otherwise false.
     */
    private boolean isDisabled = false;

    /**
     * Contains maximal cache size before it was disabled.
     */
    private int disabledMaxCacheSize = 0;
    
    

    private int initialCacheFetchSize = CacheConstants.DEFAULT_INITIAL_CACHE_FETCH_SIZE;

    private int initialDSCacheSize = CacheConstants.DEFAULT_INITIAL_DS_CACHE_SIZE;



    /**
     * Constructor(int).
     * Creates data (or DataStruct) object cache with maximal size <code>maxCSize</code>.
     *
     * @param  maxCSize maximal data (or DataStruct) object cache size.
     */
    public DataStructCacheImpl(int maxCSize) throws CacheObjectException {
        if (isDisabled) {
            throw new CacheObjectException("Caching is disabled");
        }
        cache = BaseCacheManager.getDODSCache(maxCSize);
        statistics = new DataStructCacheImplStatistics();
        nonVisibleList = new HashMap (); 
        init();
    }

    /**
     * Constructor().
     * Creates data (or DataStruct) object cache with its default maximal size
     * (the value of CacheConstants.DEFAULT_MAX_CACHE_SIZE is 10000).
     */
    public DataStructCacheImpl() throws CacheObjectException {
        this(CacheConstants.DEFAULT_MAX_CACHE_SIZE);
    }

    /**
     * Returns CacheAdministration for data (or DataStruct) object cache.
     * Parameter cacheType can have one of these values:
     *  0 - for CacheAdministration of data (or DataStruct) object cache
     *  1 - for CacheAdministration of simple query cache
     *  2 - for CacheAdministration of complex query cache
     * There is only one cache (data (or DataStruct) object), so there is only
     * one CacheAdministration. This method always returns this
     * CacheAdministration, no matter what value does parameter cacheType have.
     *
     * @param cacheType Type of caching. In this case, it is 0,  but, since there
     *  is only one cache (data (or DataStruct) object), parameter is not checked,
     * always is the same CacheAdministration object returned.
     * @return CacheAdministration for data (or DataStruct) object cache.
     */
    public CacheAdministration getCacheAdministration(int cacheType) {
        return cacheAdministration;
    }

    /**
     * Returns initialQueryCache. This attribute contains "where" clause
     * which is used during data object (or DataStruct object) cache
     * initialization.
     *
     * @return initialQueryCache.
     */
    public String  getInitialQueryCache() {
        return initialQueryCache;
    }

    /**
     * Sets initialQueryCache attribute. This attribute contains "where" clause
     * which is used during data object (or DataStruct object) cache
     * initialization.
     *
     * @param initQ New value of initialQueryCache attribute.
     */
    protected void setInitialQueryCache(String  initQ) {
        initialQueryCache = initQ;
    }

    public void makeInvisible(String  cacheHandle) {
        Integer  intObj = (Integer ) nonVisibleList.get(cacheHandle);
        int num;

        if (intObj != null) {
            num = intObj.intValue();
            num++;
            nonVisibleList.put(cacheHandle, new Integer (num));
        } else {
            nonVisibleList.put(cacheHandle, new Integer (1));
        }
    }

    public void makeVisible(String  cacheHandle) {
        Integer  intObj = (Integer ) nonVisibleList.get(cacheHandle);
        int num;

        if (intObj != null) {
            num = intObj.intValue();
            num--;
            if (num == 0) {
                nonVisibleList.remove(cacheHandle);
            } else {
                nonVisibleList.put(cacheHandle, new Integer (num));
            }
        }
    }

    /**
     * Returns statistics of used table and cache.
     *
     * @return statistics of used table and cache.
     */
    public Statistics getStatistics() {
        statistics.stopTime();
        return statistics;
    }

    /**
     * Refreshes statistics.
     */
    public void refreshStatistics() {
        statistics.clear();
    }

    /**
     * Returns information if data object (or DataStruct object) cache if "full".
     * "Full" cache contains all data objects (or DataStruct objects) from
     * the table. The cache is "full", if data (or DataStruct) object cache is
     * unbounded, if the cached table is read-only, and if initialQueryCache
     * attribute is set to "*".
     *
     * @return true if data object (or DataStruct object) cache if "full",
     * otherwise false.
     */
    public void checkFull() {
        if ((cacheAdministration.getMaxCacheSize() < 0)
                && (getInitialQueryCache() != null)
                && (getInitialQueryCache().equals("*"))) {
            fullCachingOn = true;
        } else {
            fullCachingOn = false;
        }
    }

    /**
     * Returns information if data object (or DataStruct object) cache if "full".
     * "Full" cache contains all data objects (or DataStruct objects) from
     * the table. The cache is "full", if data (or DataStruct) object cache is
     * unbounded, and if initialQueryCache attribute is set to "*".
     *
     * @return true if data object (or DataStruct object) cache if "full",
     * otherwise false.
     */
    public boolean isFull() {
        if (fullCachingOn) {
            checkFull();
        }       
        return fullCachingOn;
    }

    /**
     * Returns data object (or DataStruct object) cache type.
     * Possible values are:
     *   "none" - no caching available
     *   "lru"  - cache with LRU (least-recently used) algorithm
     *   "full" - special case of LRU cache - the whole table is cached
     *
     * @return data object (or DataStruct object) cache type.
     */
    public String  getCacheType() {
        if (cacheAdministration.getMaxCacheSize() == 0) {
            return "none";
        } else {
            if (isFull()) {
                return "full";
            } else {
                return "lru";
            }
        }
    }

    /**
     * Returns caching level.
     * Possible values:
     *  org.enhydra.dods.cache.CacheConstants.DATA_CACHING (value 1) for data (or
     *      DataStruct) object caching without query caching, and
     *  org.enhydra.dods.cache.CacheConstants.QUERY_CACHING (value 2) for data (or
     *      DataStruct) object caching with query caching.
     * This is data (or DataStruct) object cache without query caching, so this
     * method returns CacheConstants.DATA_CACHING (value 1).
     *
     * @return value 1 (org.enhydra.dods.cache.CacheConstants.DATA_CACHING).
     */
    public int getLevelOfCaching() {
        return CacheConstants.DATA_CACHING;
    }

    /**
     * Returns TableConfiguration.
     *
     * @return TableConfiguration attribute.
     */
    public TableConfiguration getTableConfiguration() {
        return tableConf;
    }

    /**
     * Returns reserveFactor.
     * Reserve factor used in in query caching. Since there is only data object
     * (or DataStruct object) cache, without query caches, this method returns 0.
     *
     * @return value 0.
     */
    public double getReserveFactor() {
        return 0;
    }

    /**
     * Sets reserveFactor.
     * Reserve factor used in in query caching. Since there is only data object
     * (or DataStruct object) cache, without query caches, this method does not
     * do anything.
     *
     * @param res New reserveFactor.
     */
    protected void setReserveFactor(double res) {}

    /**
     *
     */
    protected void setCachePercentage(double percent) {
        cachePercentage = percent;
    }

    /**
     *
     */
    public double getCachePercentage() {
        return cachePercentage;
    }

    /**
     * Returns information whether the cache is disabled.
     *
     * @return true is the cache is disabled, otherwise false.
     */
    public boolean isDisabled() {
        return isDisabled;
    }

    /**
     * Reads table and cache configuration from application's configuration file.
     *
     * @param tableConfig configuration for table of this cache.
     * @param cacheConfig configuration for this cache.
     */
    public void readConfiguration(Config tableConfig, Config cacheConfig, String  dbName)  throws CacheObjectException {
        if (isDisabled) {
            throw new CacheObjectException("Caching is disabled");
        }
        int maxSize = CacheConstants.DEFAULT_MAX_CACHE_SIZE;
        boolean initialAllCaches = CacheConstants.DEFAULT_INITIAL_ALL_CACHES;        
        Config defaultCacheConfig = null;

        this.tableConf.readTableConfiguration(tableConfig, dbName);
        DatabaseConfiguration dbConf;

        try {
            dbConf = ((StandardLogicalDatabase) (DODS.getDatabaseManager().findLogicalDatabase(dbName))).getDatabaseConfiguration();
        } catch (Exception  ex) { 
            throw new CacheObjectException("Error reading database configuration");
        } 
        if (dbConf != null) {
            try {
                maxSize = dbConf.getMaxCacheSize();
            } catch (Exception  e) {}
            try {
                reserveFactor = dbConf.getReserveFactor();
            } catch (Exception  e) {}
            try {
                cachePercentage = dbConf.getCachePercentage();
            } catch (Exception  e) {}
            try {
                initialAllCaches = dbConf.getInitAllCaches();
            } catch (Exception  e) {}          
            try {
                initialCacheFetchSize = dbConf.getInitialCacheFetchSize ();
            } catch (Exception  e) {}
            try {
                initialDSCacheSize = dbConf.getInitialDSCacheSize();
            } catch (Exception  e) {}
        }
        if (cacheConfig != null) {
            try {
                maxSize = cacheConfig.getInt(CacheConstants.PARAMNAME_MAX_CACHE_SIZE);
            } catch (Exception  e) {}
            try {
                initialQueryCache = cacheConfig.getString(CacheConstants.PARAMNAME_INITIAL_CONDITION);
            } catch (Exception  e) {
                if (initialAllCaches) {
                    initialQueryCache = "*";
                }               
            }
            try {
                reserveFactor = cacheConfig.getDouble(CacheConstants.PARAMNAME_RESERVE_FACTOR);
            } catch (Exception  e) {}        
            try {
                cachePercentage = cacheConfig.getDouble(CacheConstants.PARAMNAME_CACHE_PERCENTAGE);
            } catch (Exception  e) {}   
            try {
                initialCacheFetchSize = cacheConfig.getInt(CacheConstants.PARAMNAME_INITIAL_CACHE_FETCH_SIZE);
            } catch (Exception  e) {}
            try {
                initialDSCacheSize = cacheConfig.getInt(CacheConstants.PARAMNAME_INITIAL_DS_CACHE_SIZE);
            } catch (Exception  e) {}     
        } else if (initialAllCaches) {
            initialQueryCache = "*";
        }
        cacheAdministration.setMaxCacheSize(maxSize);
    }

    /**
     * Creates DataStructCacheImpl instance.
     *
     * @return created DataStructCacheImpl instance as DataStructCache.
     */
    public DataStructCache newInstance() throws CacheObjectException {
        if (isDisabled) {
            throw new CacheObjectException("Caching is disabled");
        }
        return new DataStructCacheImpl();
    }

    /**
     * Creates cacheAdministration object for data (or DataStruct) object cache.
     */
    protected void init() {
        cacheAdministration = new CacheAdministration() {

            /**
             * Returns maximum cache size.
             *
             * @return Maximum cache size.
             */
            public int getMaxCacheSize() {
                if (cache != null) {
                    return cache.getMaxEntries();
                }
                return 0;
            }

            /**
             * Returns maximum cache size. If the cache is unbounded (the
             * maximum size is negative), the current cache size is returned.
             *
             * @param real If this parameter is true, method returns real maximum
             * cache size, otherwise returns appropriate value for statistics.
             * @return Maximum cache size.
             */
            public int getMaxCacheSize(boolean real) {
                int size = getMaxCacheSize();

                if (size < 0) {
                    if (real) {
                        return -1;
                    } else {
                        return getCacheSize();
                    }
                }
                return size;
            }

            /**
             * Returns size of currently used cache (number of objects in the
             * cache).
             *
             * @return Size of currently used cache.
             */
            public int getCacheSize() {
                if (cache != null) {
                    return cache.size();
                }
                return 0;
            }

            /**
             * Sets maximum cache size.
             *
             * @param maxSize Maximum cache size.
             */
            protected void setMaxCacheSize(int maxSize) throws CacheObjectException {
                if (isDisabled) {
                    throw new CacheObjectException("Caching is disabled");
                }
                if (maxSize == 0) {
                    cache = null;
                    statistics.clear();
                    return;
                }
                if (cache == null) {
                    cache = BaseCacheManager.getDODSCache(maxSize);
                } else {
                    cache.setMaxEntries(maxSize);
                }
            }

            /**
             * Refreshes cache.
             */
            public void refresh() {
                if (cache != null) {
                    cache = BaseCacheManager.getDODSCache(cache.getMaxEntries());
                    statistics.clear();
                }
            }

            /**
             * Disables cache.
             */
            public void disable() {
                if (!isDisabled) {
                    isDisabled = true;
                    if (cache != null) {
                        disabledMaxCacheSize = cache.getMaxEntries();
                        statistics.clear();
                    }
                }
            }

            /**
             * Enables cache.
             */
            public void enable() {
                if (isDisabled) {
                    if (disabledMaxCacheSize != 0) {
                        cache = BaseCacheManager.getDODSCache(disabledMaxCacheSize);
                    }
                    statistics.clear();
                    isDisabled = false;
                }
            }
        };
    }

    /**
     * Returns cache (data or DataStruct) content.
     *
     * @return Cache content as <code>Map</code> of data (or DataStruct) object.
     */
    public Map  getCacheContent() {
        return cache;
    }

    /**
     * Returns information if multi databases are supported.
     *
     * @return true if multi databases are used, otherwise false.
     */
    public boolean isMulti() {
        return multi;
    }

    /**
     * Checks wheather cache reconfiguration needs to be done.
     *
     * @return true if cache reconfiguration needs to be done, otherwise false.
     */
    public boolean toReconfigure() {
        return false;
    }

    /**
     * Adds DataStruct object to the cache.
     *
     * @param newDS DataStruct object that will be added to the cache.
     *
     * @return Added DataStruct object.
     */
    public synchronized CoreDataStruct addDataStruct(CoreDataStruct newDS) {
        if (cache == null) {
            return newDS;
        }
        try {
            String  handle = newDS.get_CacheHandle();

            cache.add(handle, newDS);
            return newDS;
        } catch (Exception  e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Removes DataStruct object from the cache.
     *
     * @param data DataStruct object that will be removed from the cache.
     *
     * @return Removed DataStruct object, or <tt>null</tt> if there was no
     * object removed from the cache.
     */
    public synchronized CoreDataStruct removeDataStruct(CoreDataStruct data) {
        if (cache != null) {
            try {
                String  handle = data.get_CacheHandle();

                return (CoreDataStruct) cache.remove(handle);
            } catch (Exception  e) {}
        }
        return null;
    }

    /**
     * Removes DataStruct object from the cache.
     *
     * @param handle Cache handle of DataStruct object that will be removed from
     * the cache. The form of cache handle is:
     * "<database_name>.<String_presentation_of_oid>".
     *
     * @return Removed DataStruct object, or <tt>null</tt> if there was no
     * object removed from the cache.
     */
    public synchronized CoreDataStruct removeDataStruct(String  handle) {
        if (cache != null) {
            try {
                return (CoreDataStruct) cache.remove(handle);
            } catch (Exception  e) {}
        }
        return null;
    }

    /**
     * Updates cached DataStruct object, or inserts it in the cache if it didn't
     *  exist in the cache.
     *
     * @param data DataStruct object that will be updated (or inserted if didn't
     * exist in the cache).
     *
     * @return Updated or inserted DataStruct object.
     */
    public CoreDataStruct updateDataStruct(CoreDataStruct data) {
        data = addDataStruct(data);
        return data;
    }

    /**
     * Deletes DataStruct object from the cache.
     *
     * @param data DataStruct object that will be deleted from the cache.
     *
     * @return Deleted DataStruct object, or <tt>null</tt> if there was no
     * object deleted from the cache.
     */
    public CoreDataStruct deleteDataStruct(CoreDataStruct data) {
        CoreDataStruct oldData = removeDataStruct(data);

        return oldData;
    }

    /**
     * Returns DataStruct object whose String representation of OID is parameter
     * handle.
     *
     * @param handle String representation of OID of DataStruct object that is
     * being searched in the cache.
     *
     * @return DataStruct object whose String representation of OID is handle.
     */
    public CoreDataStruct getDataStructByHandle(String  handle) {
        if (cache == null) {
            return null;
        }
        CoreDataStruct cdt = null;
        if(cache.isNeedToSynchronize()) {
            synchronized (cache) {
                cdt = getDataCacheItem(handle, cdt);
            }       
        }else {
            cdt = getDataCacheItem(handle, cdt);
        }
        return cdt;
    }

    /**
     * @param handle
     * @param cdt
     * @return
     */
    private CoreDataStruct getDataCacheItem(String  handle, CoreDataStruct cdt) {
        if (!nonVisibleList.containsKey(handle)) {
            cdt = (CoreDataStruct) cache.get(handle);
        }
        return cdt;
    }

    /**
     * Shows content of this class.
     * Can be used for debugging.
     */
    public void show() {
        System.out.println("-------------------------------------------------");
        System.out.println(" maxCacheSize : " + cache.getMaxEntries());
        System.out.println(" cacheReadOnly : " + tableConf.isReadOnly());
        System.out.println(" initialQueryCache : " + initialQueryCache);
        System.out.println(" fullCaching : " + isFull());
    }

    /**
     * Shows content of this class.
     * Can be used for debugging.
     */
    public String  toString() {
        StringBuffer  ret = new StringBuffer ();

        ret.append("\n DataStructCacheImpl: ");
        ret.append("\n cacheReadOnly : " + tableConf.isReadOnly());
        ret.append("\n initialQueryCache : " + initialQueryCache);
        ret.append("\n fullCaching : " + isFull());
        return ret.toString();
    }

    
    /**
     * @return
     */
    public int getInitialCacheFetchSize() {
        return initialCacheFetchSize;
    }

    /**
     * @return
     */
    public int getInitialDSCacheSize() {
        return initialDSCacheSize;
    }

    /**
     * @param i
     */
    public void setInitialCacheFetchSize(int i) {
        initialCacheFetchSize = i;
    }

    /**
     * @param i
     */
    public void setInitialDSCacheSize(int i) {
        initialDSCacheSize = i;
    }


    /**
     * Inner class that implements TableStatistics class for administrating
     * table and cache.
     */
    class DataStructCacheImplStatistics extends TableStatistics {

        /**
         * Constructor().
         */
        public DataStructCacheImplStatistics() {
            super();
        }

        /**
         * Returns type of the statistics. In this case, this is CACHE_STATISTICS.
         *
         * @return Type of the statistics: CACHE_STATISTICS.
         */
        public int getStatisticsType() {
            return CACHE_STATISTICS;
        }

        /**
         * Returns query statistics for DO cache.
         * Since there is only one cache - DO cache, always is returned DO query
         * statistics, no metter what is the value of parameter type.
         *
         * @param type should be 0
         * (org.enhydra.dods.cache.CacheConstants.DATA_CACHE),
         * but, since there is only one cache - DO cache, always is returned DO
         *  query statistics, parameter is not used.
         *
         * @return DO cache query statistics.
         */
        public CacheStatistics getCacheStatistics(int type) {
            switch (type) {
            case CacheConstants.DATA_CACHE:
                return cache;

            default:
                return null;
            }
        }
    }
}
