
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package org.enhydra.dods.cache.hash;

import java.util.Collection ;
import java.util.HashSet ;
import java.util.Iterator ;
import java.util.Map ;
import java.util.Set ;

/**
 * @author    Tanja Jovanovic
 *
 * LinkedHashCache class implements LRU cache for storing objects.
 * @version   2.0  07.01.2003.
 *
 */
 
public class LinkedHashCache implements Map  {

    private Map  impl;
    private static Map  createMap(int initialCapacity, float loadFactor, boolean flag) {
        Map  ret = null;
        try {
            ret = (Map ) Class.forName("java.util.LinkedHashMap")
                .getConstructor(new Class [] {int.class, float.class, boolean.class})
                .newInstance(new Object [] {new Integer (initialCapacity), new Float (loadFactor), new Boolean (flag)});
        } catch (Throwable  t) {
        }
        return ret;
    }
    /**
     * Maximal number of objects in LinkedHashCache - 1024 is default value of this 
     * variable.
     */
    protected int maxEntries = 1024;
    
    /**
     * Constructor (int, float, int).
     * Constructs an empty access-ordered <tt>LinkedHashCache</tt> instance with the 
     * specified initial capacity, load factor and maximal number of
     * objects.
     *
     * @param  initialCapacity The initial capacity.
     * @param  loadFactor The load factor.
     * @param  maxEnt Maximal number of objects in LinkedHashCache.
     */
    public LinkedHashCache(int initialCapacity, float loadFactor, int maxEnt) {
        impl = createMap(initialCapacity, loadFactor, false);
        maxEntries = maxEnt;
    }
    
    /**
     * Constructor (int, float). 
     * Constructs an empty access-ordered <tt>LinkedHashCache</tt> instance with the
     * specified initial capacity and load factor.
     *
     * @param  initialCapacity The initial capacity.
     * @param  loadFactor The load factor.
     */
    public LinkedHashCache(int initialCapacity, float loadFactor) {
        impl = createMap(initialCapacity, loadFactor, false);
    }
   
    /**
     * Constructor (int). 
     * Constructs an empty access-ordered <tt>LinkedHashCache</tt> instance with a 
     * default capacity (16) and load factor (0.75) and with maximal number of 
     * objects.
     *
     * @param  maxEnt Maximal number of objects in LinkedHashCache.
     */
    public LinkedHashCache(int maxEnt) {
        impl = createMap(16, (float) 0.75, false);
        maxEntries = maxEnt;
    }
    
    /**
     * Constructor (). 
     * Constructs an empty access-ordered <tt>LinkedHashCache</tt> instance with a 
     * default capacity (16) and load factor (0.75).
     */
    public LinkedHashCache() {
        impl = createMap(16, (float) 0.75, false);
    }

    /**
     * Constructor (Map, int). 
     * Constructs an access-ordered <tt>LinkedHashCache</tt> instance with the same 
     * mappings as the specified map. The <tt>LinkedHashCache</tt> instance is created
     * with maximal number of objects, default load factor (0.75) and an initial
     * capacity sufficient to hold the mappings in the specified map.
     *
     * @param  m The map whose mappings are to be placed in this map.
     * @param  maxEnt Maximal number of objects in LinkedHashCache.
     */
    public LinkedHashCache(Map  m, int maxEnt) {
        impl = createMap(16, (float) 0.75, false);
        maxEntries = maxEnt;
        impl.putAll(m);
    }
    
    /**
     * Constructor (Map). 
     * Constructs an access-ordered <tt>LinkedHashCache</tt> instance with the same 
     * mappings as the specified map. The <tt>LinkedHashCache</tt> instance is created
     * with default load factor (0.75) and an initial capacity sufficient to 
     * hold the mappings in the specified map.
     *
     * @param m The map whose mappings are to be placed in this map.
     */
    public LinkedHashCache(Map  m) {
        impl = createMap(16, (float) 0.75, false);
        impl.putAll(m);
    }

    /**
     * Adds a pair (key, value) to the cache. If key already exists, the method 
     * returns previous value asociated with this key. If maximum number of 
     * objects is achieved, the eldest entry will be removed and returned.
     *
     * @param  key Key asociated with the value.
     * @param  value Value that will be added to the cache.
     * @return Previous value asociated with this key, if the key already 
     * exists, or the eldest entry which is removed, if maximum number of 
     * objects is achieved, otherwise null. 
     */
    public Object  add(Object  key, Object  value) {
        Object  ret = impl.put(key, value);

        if (ret == null) {
            if (maxEntries < size() && maxEntries > 0) {
                Iterator  iter = impl.keySet().iterator();
                Object  rem = iter.next();

                ret = impl.remove(rem);
            }
        }
        return ret;
    }

    /**
     * Returns maximal number of objects in LinkedHashCache.
     *
     * @return Maximal number of objects in LinkedHashCache.
     */
    public int getMaxEntries() {
        return maxEntries;
    }
    
    /**
     * Sets maximal number of objects in LinkedHashCache.
     *
     * @param max New value of maximal number of objects in LinkedHashCache.
     */
    public void setMaxEntries(int max) {
        if (max < size() && max > 0) {
            Iterator  iter = (new HashSet (impl.keySet())).iterator();
            int dif = size() - max;

            for (int i = 0; i < dif; i++) {
                impl.remove(iter.next());
            }
        }
        maxEntries = max;
    }

    /**
     * For debug reason.
     */
    public String  toString() {
        Object [] v = impl.entrySet().toArray();
        String  ret = "LRU content:\n";

        for (int i = 0; i < v.length; i++) {
            Map.Entry  map = (Map.Entry ) v[i];

            ret += i + ". key = " + map.getKey() + "  value = " + map.getValue()
                    + "\n";
        } 
        return ret;
    }

    public int size() {
        return impl.size();
    }
    public boolean isEmpty() {
        return impl.isEmpty();
    }
    public boolean containsKey(Object  key) throws ClassCastException , NullPointerException  {
        return impl.containsKey(key);
    }
    public boolean containsValue(Object  value) throws ClassCastException , NullPointerException  {
        return impl.containsValue(value);
    }
    public Object  get(Object  key) throws ClassCastException , NullPointerException  {
        return impl.get(key);
    }
    public Object  put(Object  key, Object  value) throws ClassCastException , NullPointerException , UnsupportedOperationException , IllegalArgumentException  {
        return impl.put(key, value);
    }
    public synchronized Object  remove(Object  key) throws ClassCastException , NullPointerException , UnsupportedOperationException  {
        return impl.remove(key);
    }
    public void putAll(Map  t) throws ClassCastException , NullPointerException , UnsupportedOperationException , IllegalArgumentException  {
        impl.putAll(t);
    }
    public void clear() throws UnsupportedOperationException  {
        impl.clear();
    }
    public Set  keySet() {
        return impl.keySet();
    }
    public Collection  values() {
        return impl.values();
    }
    public Set  entrySet() {
      return impl.entrySet();
    }
    public boolean equals(Object  o) {
        return (o instanceof LinkedHashCache) && impl.equals(((LinkedHashCache)o).impl);
    }
    public int hashCode() {
        return impl.hashCode();
    }
}
