
/*
 * Enhydra Java Application Server Project
 * 
 * The contents of this file are subject to the Enhydra Public License Version
 * 1.1 (the "License"); you may not use this file except in compliance with the
 * License. You may obtain a copy of the License on the Enhydra web site (
 * http://www.enhydra.org/ ).
 * 
 * Software distributed under the License is distributed on an "AS IS" basis,
 * WITHOUT WARRANTY OF ANY KIND, either express or implied. See the License for
 * the specific terms governing rights and limitations under the License.
 * 
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created by
 * Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc. All Rights
 * Reserved.
 *  
 */
package org.enhydra.dods.cache;

import java.util.Collection ;
import java.util.Date ;
import java.util.Enumeration ;
import java.util.HashMap ;
import java.util.HashSet ;
import java.util.Iterator ;
import java.util.Map ;
import java.util.Vector ;

import org.enhydra.dods.DODS;
import org.enhydra.dods.cache.base.BaseCacheManager;
import org.enhydra.dods.cache.base.DODSCache;
import org.enhydra.dods.exceptions.CacheObjectException;
import org.enhydra.dods.statistics.CacheStatistics;
import org.enhydra.dods.statistics.Statistics;
import org.enhydra.dods.statistics.TableStatistics;

import com.lutris.appserver.server.sql.CoreDataStruct;
import com.lutris.appserver.server.sql.DatabaseManagerException;
import com.lutris.appserver.server.sql.standard.DatabaseConfiguration;
import com.lutris.appserver.server.sql.standard.StandardLogicalDatabase;
import com.lutris.util.Config;

/**
 * This class contains data and mechanisms needed for caching data objects (or
 * DataStruct objects) and queries and provides cache configuration and
 * administration.
 * 
 * @author Tanja Jovanovic
 * @author Nenad Vico
 * @author Zorica Suvajdzin
 * @version 2.0 15.06.2003.
 */
public class QueryCacheImpl extends QueryCache {

    /**
     * LRU cache for storing data (or DataStruct) objects. LRU cache keys are
     * cache handles (String " <database_name>. <String_presentation_of_oid>"),
     * and LRU cache values are data (or DataStruct) objects.
     */
    protected DODSCache             cache                               = null;

    /**
     * LRU cache for storing simple queries. LRU cache keys are Strings in form
     * of " <query_database_name>. <String_presentation_of_query>", and LRU
     * cache values are objects of org.enhydra.dods.cache.QueryCacheItem class.
     * QueryCacheItem class stores one query and its necessary data.
     */
    protected DODSCache             simpleQCache                        = null;

    /**
     * LRU cache for storing complex queries. LRU cache keys are Strings in form
     * of " <query_database_name>. <String_presentation_of_query>", and LRU
     * cache values are objects of org.enhydra.dods.cache.QueryCacheItem class.
     * QueryCacheItem class stores one query and its necessary data.
     */
    protected DODSCache             complexQCache                       = null;

    /**
     * LRU cache for storing multi join queries. LRU cache keys are Strings in
     * form of " <query_database_name>. <String_presentation_of_query>", and LRU
     * cache values are objects of org.enhydra.dods.cache.QueryCacheItem class.
     * QueryCacheItem class stores one query and its necessary data.
     */
    protected DODSCache             multiJoinQCache                     = null;

    /**
     * Attribute cacheAdministration is array of three objects of the class
     * CacheAdministration. CacheAdministration[0] handles configuration
     * settings about data (or DataStruct object) cache. CacheAdministration[1]
     * handles configuration settings about simple query cache.
     * CacheAdministration[2] handles configuration settings about complex query
     * cache. CacheAdministration[3] handles configuration settings about multi
     * join query cache.
     */
    protected CacheAdministration[] cacheAdministration                 = new CacheAdministration[4];

    /**
     * TableConfiguration attribute handles configuration settings about table
     * of this cache.
     */
    protected TableConfiguration    tableConf                           = new TableConfiguration();

    /**
     * Initial query statement. This attribute contains "where" clause which is
     * used during data (or DataStruct) object cache initialization. If this
     * attribute is set to "*", all rows from the table (in database) will be
     * put in the cache.
     */
    protected String                 initialQueryCache                   = null;

    /**
     * This attribute contains information if multi databases are used.
     */
    protected boolean               multi                               = false;

    /**
     * This attribute is true if the cache is full, otherwise false.
     */
    protected boolean               fullCachingOn                       = false;

    /**
     * Table and cache statictics.
     */
    protected Statistics            statistics                          = null;

    /**
     * List of objects that are unvisible in the cache at the moment. Keys of
     * this map sre cache handles (String " <database_name>.
     * <String_presentation_of_oid>"), and values are counters of how many
     * transactions have made object with that cache handle invisible.
     */
    protected HashMap                nonVisibleList                      = null;

    /**
     * Old max simple cache size. When data (or DataStruct) cache's maxCacheSize
     * is set to zero, so is simple query's. This attribute is needed for
     * retrieving max simple cache size after data (or DataStruct) cache's
     * maxCacheSize has been set again to any value different from zero.
     */
    protected int                   oldSimpleMaxCacheSize;

    /**
     * Old max complex cache size. When data (or DataStruct) cache's
     * maxCacheSize is set to zero, so is complex query's. This attribute is
     * needed for retrieving max complex cache size after data (or DataStruct)
     * cache's maxCacheSize has been set again to any value different from zero.
     */
    protected int                   oldComplexMaxCacheSize;

    /**
     * Old max multi join cache size. When data (or DataStruct) cache's
     * maxCacheSize is set to zero, so is multi join query's. This attribute is
     * needed for retrieving max multi join cache size after data (or
     * DataStruct) cache's maxCacheSize has been set again to any value
     * different from zero.
     */
    protected int                   oldMultiJoinMaxCacheSize;

    /**
     * This attribute is used in query caching. It is percent of how many more
     * object are taken for evaluation. If <code>num</code> is number of
     * needed results, then it is used <code>num</code>+
     * DEFAULT_RESERVE_FACTOR *<code>num</code> of objects for estimating
     * what is quicker: go to database for all object that are not in the cache,
     * or run again query on database. This value is given in percents, as
     * number between 0 and 1 (0.25 means 25%). For example, if
     * DEFAULT_RESERVE_FACTOR is 0.5, and wanted number of results is 50, the
     * estimation will be done on 75 (50 + 0.5 * 50) objects. The value of
     * CacheConstants.DEFAULT_RESERVE_FACTOR is 0.5.
     */
    protected double                reserveFactor                       = CacheConstants.DEFAULT_RESERVE_FACTOR;

    /**
     * Indicator for disabled data (or DataStruct) cache cache.
     */
    private boolean                 isDisabled                          = false;

    /**
     * Indicator for disabled simple query cache.
     */
    private boolean                 isDisabledSimple                    = false;

    /**
     * Indicator for disabled complex query cache.
     */
    private boolean                 isDisabledComplex                   = false;

    /**
     * Indicator for disabled multi join query cache.
     */
    private boolean                 isDisabledMultiJoin                 = false;

    private double                  cachePercentage                     = CacheConstants.DEFAULT_CACHE_PERCENTAGE;

    private int                     initialCacheFetchSize               = CacheConstants.DEFAULT_INITIAL_CACHE_FETCH_SIZE;

    private int                     initialDSCacheSize                  = CacheConstants.DEFAULT_INITIAL_DS_CACHE_SIZE;

    /**
     * Maximal cache size of data (or DataStruct) cache cache before it had been
     * disabled.
     */
    private int                     disabledMaxCacheSize                = 0;

    /**
     * Maximal cache size of simple query cache before it had been disabled.
     */
    private int                     disabledMaxSimpleQueryCacheSize     = 0;

    /**
     * Maximal cache size of complex query cache before it had been disabled.
     */
    private int                     disabledMaxComplexQueryCacheSize    = 0;

    /**
     * Maximal cache size of multi join query cache before it had been disabled.
     */
    private int                     disabledMaxMultiJoinQueryCacheSize  = 0;

    /**
     * Constructor(int). Creates data (or DataStruct) object cache with maximal
     * size <code>maxCSize</code> and simple and complex query caches with
     * their default maximal sizes. The value of
     * CacheConstants.DEFAULT_MAX_SIMPLE_QUERY_CACHE_SIZE is 0. The value of
     * CacheConstants.DEFAULT_MAX_COMPLEX_QUERY_CACHE_SIZE is 0.
     * 
     * @param maxCSize
     *            maximal data (or DataStruct) object cache size.
     */
    public QueryCacheImpl(int maxCSize) throws CacheObjectException {
        if (isDisabled) {
            throw new CacheObjectException("Caching is disabled");
        }
        cache = BaseCacheManager.getDODSCache(maxCSize);
        if (cache != null) {
            simpleQCache = BaseCacheManager.getDODSCache(CacheConstants.DEFAULT_MAX_SIMPLE_QUERY_CACHE_SIZE);
            complexQCache = BaseCacheManager.getDODSCache(CacheConstants.DEFAULT_MAX_COMPLEX_QUERY_CACHE_SIZE);
            multiJoinQCache = BaseCacheManager.getDODSCache(CacheConstants.DEFAULT_MAX_MULTI_JOIN_QUERY_CACHE_SIZE);
        }
        statistics = new QueryCacheImplStatistics();
        nonVisibleList = new HashMap ();
        init();
    }

    /**
     * Constructor(). Creates data (or DataStruct) object cache with its default
     * maximal size and simple and complex query caches with their default
     * maximal sizes. The value of CacheConstants.DEFAULT_MAX_CACHE_SIZE is 0.
     * The value of CacheConstants.DEFAULT_MAX_SIMPLE_QUERY_CACHE_SIZE is 0. The
     * value of CacheConstants.DEFAULT_MAX_COMPLEX_QUERY_CACHE_SIZE is 0.
     */
    public QueryCacheImpl() throws CacheObjectException {
        this(CacheConstants.DEFAULT_MAX_CACHE_SIZE);
    }

    /**
     * Returns CacheAdministration for data object (or DataStruct object) cache,
     * simple, or complex query cache. Object CacheAdministration handles
     * configuration settings about these caches. Parameter cacheType can have
     * one of these values: 0 - for CacheAdministration of data object (or
     * DataStruct object) cache 1 - for CacheAdministration of simple query
     * cache 2 - for CacheAdministration of complex query cache 3 - for
     * CacheAdministration of multi join query cache
     * 
     * @param cacheType
     *            0 - for data object (or DataStruct object), 1 for simple
     *            query, 2 for complex query cache and 3 for complex query
     *            cache.
     */
    public CacheAdministration getCacheAdministration(int cacheType) {
        if (cacheType < 0 || cacheType > 3) {
            return null;
        }
        return cacheAdministration[cacheType];
    }

    /**
     * Returns initialQueryCache. This attribute contains "where" clause which
     * is used during data object (or DataStruct object) cache initialization.
     * 
     * @return initialQueryCache.
     */
    public String  getInitialQueryCache() {
        return initialQueryCache;
    }

    /**
     * Sets initialQueryCache attribute. This attribute contains "where" clause
     * which is used during data object (or DataStruct object) cache
     * initialization.
     * 
     * @param initQ
     *            New value of initialQueryCache attribute.
     */
    protected void setInitialQueryCache(String  initQ) {
        initialQueryCache = initQ;
    }

    public void makeInvisible(String  cacheHandle) {
        if (getCacheAdministration(CacheConstants.DATA_CACHE).getMaxCacheSize() != 0) {
            Integer  intObj = (Integer ) nonVisibleList.get(cacheHandle);
            int num;

            if (intObj != null) {
                num = intObj.intValue();
                num++;
                nonVisibleList.put(cacheHandle, new Integer (num));
            } else {
                nonVisibleList.put(cacheHandle, new Integer (1));
            }
        }
    }

    public void makeVisible(String  cacheHandle) {
        if (getCacheAdministration(CacheConstants.DATA_CACHE).getMaxCacheSize() != 0) {
            Integer  intObj = (Integer ) nonVisibleList.get(cacheHandle);
            int num;

            if (intObj != null) {
                num = intObj.intValue();
                num--;
                if (num == 0) {
                    nonVisibleList.remove(cacheHandle);
                } else {
                    nonVisibleList.put(cacheHandle, new Integer (num));
                }
            }
        }
    }

    /**
     * Returns statistics of used table and caches.
     * 
     * @return statistics of used table and caches.
     */
    public Statistics getStatistics() {
        statistics.stopTime();
        return statistics;
    }

    /**
     * Refreshes statistics.
     */
    public void refreshStatistics() {
        statistics.clear();
    }

    /**
     * Returns information if data object (or DataStruct object) cache if
     * "full". "Full" cache contains all data objects (or DataStruct objects)
     * from the table. The cache is "full", if data (or DataStruct) object cache
     * is unbounded, and if initialQueryCache attribute is set to "*".
     * 
     * @return true if data object (or DataStruct object) cache if "full",
     *         otherwise false.
     */
    public void checkFull() {
        if ((cacheAdministration[0].getMaxCacheSize() < 0) && (getInitialQueryCache() != null)
                        && (getInitialQueryCache().equalsIgnoreCase("*"))) {
            fullCachingOn = true;
        } else {
            fullCachingOn = false;
        }
    }

    /**
     * Returns information if data object (or DataStruct object) cache if
     * "full". "Full" cache contains all data objects (or DataStruct objects)
     * from the table. The cache is "full", if data (or DataStruct) object cache
     * is unbounded, and if initialQueryCache attribute is set to "*".
     * 
     * @return true if data object (or DataStruct object) cache if "full",
     *         otherwise false.
     */
    public boolean isFull() {
        if (fullCachingOn) {
            checkFull();
        }
        return fullCachingOn;
    }

    /**
     * Returns data object (or DataStruct object) cache type. Possible values
     * are: "none" - no caching available "lru" - cache with LRU (least-recently
     * used) algorithm "full" - special case of LRU cache - the whole table is
     * cached
     * 
     * @return data object (or DataStruct object) cache type.
     */
    public String  getCacheType() {
        if (cacheAdministration[CacheConstants.DATA_CACHE].getMaxCacheSize() == 0) {
            return "none";
        } else {
            if (isFull()) {
                return "full";
            } else {
                return "lru";
            }
        }
    }

    /**
     * Returns caching level. Possible values:
     * org.enhydra.dods.cache.CacheConstants.DATA_CACHING (value 1) for data
     * object (or DataStruct object) caching without query caching, and
     * org.enhydra.dods.cache.CacheConstants.QUERY_CACHING (value 2) for data
     * object (or DataStruct object) caching with query caching.
     * 
     * @return Value 2 (org.enhydra.dods.cache.CacheConstants.QUERY_CACHING).
     */
    public int getLevelOfCaching() {
        return CacheConstants.QUERY_CACHING;
    }

    /**
     * Returns object TableConfiguration. TableConfiguration contains parameters
     * (and their set and get methods) for table configuration.
     * 
     * @return TableConfiguration.
     */
    public TableConfiguration getTableConfiguration() {
        return tableConf;
    }

    /**
     * Returns reserveFactor. This attribute is used in query caching. It
     * defines how many more (exept needed) objects are taken for evaluation. If
     * <code>num</code> is number of needed results, then it is used
     * <code>num</code>+ DEFAULT_RESERVE_FACTOR *<code>num</code> of
     * objects for estimating what is quicker: go to database for all object
     * that are not in the cache, or run again query on database. This value is
     * given in percents, as number between 0 and 1 (0.25 means 25%). For
     * example, if DEFAULT_RESERVE_FACTOR is 0.5, and wanted number of results
     * is 50, the estimation will be done on 75 (50 + 0.5 * 50) objects.
     * 
     * @return reserveFactor.
     */
    public double getReserveFactor() {
        return reserveFactor;
    }

    /**
     * Sets reserveFactor. This attribute is used in query caching. It defines
     * how many more (exept needed) objects are taken for evaluation. If
     * <code>num</code> is number of needed results, then it is used
     * <code>num</code>+ DEFAULT_RESERVE_FACTOR *<code>num</code> of
     * objects for estimating what is quicker: go to database for all object
     * that are not in the cache, or run again query on database. This value is
     * given in percents, as number between 0 and 1 (0.25 means 25%). For
     * example, if DEFAULT_RESERVE_FACTOR is 0.5, and wanted number of results
     * is 50, the estimation will be done on 75 (50 + 0.5 * 50) objects.
     * 
     * @param res
     *            New reserveFactor.
     */
    protected void setReserveFactor(double res) {
        reserveFactor = res;
    }

    /**
     *  
     */
    protected void setCachePercentage(double percent) {
        cachePercentage = percent;
    }

    /**
     *  
     */
    public double getCachePercentage() {
        return cachePercentage;
    }

    /**
     * Returns information whether caching is disabled.
     * 
     * @return true is caching is disabled, otherwise false.
     */
    public boolean isDisabled() {
        return isDisabled;
    }

    /**
     * Reads table and cache configuration from application's configuration
     * file.
     * 
     * @param tableConfig
     *            configuration for table of this cache.
     * @param cacheConfig
     *            configuration for this cache.
     */
    public void readConfiguration(Config tableConfig, Config cacheConfig, String  dbName) throws CacheObjectException {
        if (isDisabled) {
            throw new CacheObjectException("Caching is disabled");
        }
        boolean initialAllCaches = CacheConstants.DEFAULT_INITIAL_ALL_CACHES;
        int maxSize = CacheConstants.DEFAULT_MAX_CACHE_SIZE;
        int maxSimple = CacheConstants.DEFAULT_MAX_SIMPLE_QUERY_CACHE_SIZE;
        int maxComplex = CacheConstants.DEFAULT_MAX_COMPLEX_QUERY_CACHE_SIZE;
        int maxMultiJoin = CacheConstants.DEFAULT_MAX_MULTI_JOIN_QUERY_CACHE_SIZE;
        Config defaultCacheConfig = null;

        this.tableConf.readTableConfiguration(tableConfig, dbName);

        DatabaseConfiguration dbConf;

        try {
            dbConf = ((StandardLogicalDatabase) (DODS.getDatabaseManager().findLogicalDatabase(dbName)))
                            .getDatabaseConfiguration();
        } catch (Exception  ex) {
            throw new CacheObjectException("Error reading database configuration");
        }
        if (dbConf != null) {
            try {
                maxSize = dbConf.getMaxCacheSize();
            } catch (Exception  e) {
            }
            try {
                maxSimple = dbConf.getMaxSimpleCacheSize();
            } catch (Exception  e) {
            }
            try {
                maxComplex = dbConf.getMaxComplexCacheSize();
            } catch (Exception  e) {
            }
            try {
                maxMultiJoin = dbConf.getMaxMultiJoinCacheSize();
            } catch (Exception  e) {
            }
            try {
                reserveFactor = dbConf.getReserveFactor();
            } catch (Exception  e) {
            }
            try {
                cachePercentage = dbConf.getCachePercentage();
            } catch (Exception  e) {
            }
            try {
                initialAllCaches = dbConf.getInitAllCaches();
            } catch (Exception  e) {
            }
            try {
                initialCacheFetchSize = dbConf.getInitialCacheFetchSize();
            } catch (Exception  e) {
            }
            try {
                initialDSCacheSize = dbConf.getInitialDSCacheSize();
            } catch (Exception  e) {
            }

        }
        if (cacheConfig != null) {
            try {
                maxSize = cacheConfig.getInt(CacheConstants.PARAMNAME_MAX_CACHE_SIZE);
            } catch (Exception  e) {
            }
            try {
                maxSimple = cacheConfig.getInt(CacheConstants.PARAMNAME_MAX_SIMPLE_CACHE_SIZE);
            } catch (Exception  e) {
            }
            try {
                maxComplex = cacheConfig.getInt(CacheConstants.PARAMNAME_MAX_COMPLEX_CACHE_SIZE);
            } catch (Exception  e) {
            }
            try {
                maxMultiJoin = cacheConfig.getInt(CacheConstants.PARAMNAME_MAX_MULTI_JOIN_CACHE_SIZE);
            } catch (Exception  e) {
            }
            try {
                initialQueryCache = cacheConfig.getString(CacheConstants.PARAMNAME_INITIAL_CONDITION);
            } catch (Exception  e) {
                if (initialAllCaches) {
                    initialQueryCache = "*";
                }
            }
            try {
                reserveFactor = cacheConfig.getDouble(CacheConstants.PARAMNAME_RESERVE_FACTOR);
            } catch (Exception  e) {
            }
            try {
                cachePercentage = cacheConfig.getDouble(CacheConstants.PARAMNAME_CACHE_PERCENTAGE);
            } catch (Exception  e) {
            }
            try {
                initialCacheFetchSize = cacheConfig.getInt(CacheConstants.PARAMNAME_INITIAL_CACHE_FETCH_SIZE);
            } catch (Exception  e) {
            }
            try {
                initialDSCacheSize = cacheConfig.getInt(CacheConstants.PARAMNAME_INITIAL_DS_CACHE_SIZE);
            } catch (Exception  e) {
            }
        } else {
            if (initialAllCaches) {
                initialQueryCache = "*";
            }
        }
        cacheAdministration[CacheConstants.DATA_CACHE].setMaxCacheSize(maxSize);
        cacheAdministration[CacheConstants.SIMPLE_QUERY_CACHE].setMaxCacheSize(maxSimple);
        cacheAdministration[CacheConstants.COMPLEX_QUERY_CACHE].setMaxCacheSize(maxComplex);
        cacheAdministration[CacheConstants.MULTI_JOIN_QUERY_CACHE].setMaxCacheSize(maxMultiJoin);
    }

    /**
     * Creates QueryCacheImpl instance.
     * 
     * @return created QueryCacheImpl instance as DataStructCache.
     */
    public DataStructCache newInstance() throws CacheObjectException {
        if (isDisabled) {
            throw new CacheObjectException("Caching is disabled");
        }
        return new QueryCacheImpl();
    }

    /**
     * Creates array of three CacheAdministration objects.
     * CacheAdministration[0] handles configuration settings about data (or
     * DataStruct) object cache. CacheAdministration[1] handles configuration
     * settings about simple query cache. CacheAdministration[2] handles
     * configuration settings about complex query cache.
     */
    protected void init() {
        cacheAdministration[CacheConstants.SIMPLE_QUERY_CACHE] = new CacheAdministration() {

            /**
             * Returns maximum size of simple query cache.
             * 
             * @return Maximum size of simple query cache.
             */
            public int getMaxCacheSize() {
                if (simpleQCache != null) {
                    return simpleQCache.getMaxEntries();
                }
                return 0;
            }

            /**
             * Returns maximum simple query cache size. If the cache is
             * unbounded (the maximum size is negative), the current cache size
             * is returned.
             * 
             * @param real
             *            If this parameter is true, method returns real maximum
             *            cache size, otherwise returns appropriate value for
             *            the statistic.
             * @return Maximum simple query cache size.
             */
            public int getMaxCacheSize(boolean real) {
                int size = getMaxCacheSize();

                if (size < 0) {
                    if (real) {
                        return -1;
                    } else {
                        return getCacheSize();
                    }
                }
                return size;
            }

            /**
             * Sets maximum size of simple query cache.
             * 
             * @param maxSize
             *            Maximum size of simple query cache.
             */
            protected void setMaxCacheSize(int maxSize) throws CacheObjectException {
                try {
                    if (isDisabled) {
                        throw new CacheObjectException("Caching is disabled");
                    }
                    if (maxSize == 0) {
                        if (simpleQCache != null) {
                            statistics.getCacheStatistics(CacheConstants.SIMPLE_QUERY_CACHE).clearStatistics();
                            simpleQCache = null;
                        }
                        return;
                    }
                    if (simpleQCache == null) {
                        if (cache != null) {
                            simpleQCache = BaseCacheManager.getDODSCache(maxSize);
                        }
                    } else {
                        simpleQCache.setMaxEntries(maxSize);
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in setMaxCacheSize - simple cache");
                }
            }

            /**
             * Returns size of currently used simple query cache.
             * 
             * @return Size of currently used simple query cache.
             */
            public int getCacheSize() {
                if (simpleQCache != null) {
                    return simpleQCache.size();
                }
                return 0;
            }

            /**
             * Refreshes simple query cache.
             */
            public void refresh() {
                if (cache != null) {
                    if (simpleQCache != null) {
                        simpleQCache = BaseCacheManager.getDODSCache(simpleQCache.getMaxEntries());
                        statistics.getCacheStatistics(CacheConstants.SIMPLE_QUERY_CACHE).setCacheHitsNum(0);
                    }
                }
            }

            /**
             * Disables simple query cache.
             */
            public void disable() {
                if (!isDisabledSimple) {
                    isDisabledSimple = true;
                    if (simpleQCache != null) {
                        disabledMaxSimpleQueryCacheSize = simpleQCache.getMaxEntries();
                        simpleQCache = null;
                    } else {
                        disabledMaxSimpleQueryCacheSize = 0;
                    }
                }
            }

            /**
             * Enables simple query cache.
             */
            public void enable() {
                try {
                    if (isDisabledSimple) {
                        if (disabledMaxSimpleQueryCacheSize != 0) {
                            if (disabledMaxCacheSize != 0) {
                                simpleQCache = BaseCacheManager.getDODSCache(disabledMaxSimpleQueryCacheSize);
                            }
                        }
                        isDisabledSimple = false;
                    }
                    DODSCache obj = (DODSCache) statistics.getCacheStatistics(CacheConstants.SIMPLE_QUERY_CACHE);

                    if (obj != null) {
                        obj.clearStatistics();
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in enable simple");
                }
            }
        };
        cacheAdministration[CacheConstants.COMPLEX_QUERY_CACHE] = new CacheAdministration() {

            /**
             * Returns maximum size of complex query cache.
             * 
             * @return Maximum size of complex query cache.
             */
            public int getMaxCacheSize() {
                if (complexQCache != null) {
                    return complexQCache.getMaxEntries();
                }
                return 0;
            }

            /**
             * Returns maximum complex query cache size. If the cache is
             * unbounded (the maximum size is negative), the current cache size
             * is returned.
             * 
             * @param real
             *            If this parameter is true, method returns real maximum
             *            cache size, otherwise returns appropriate value for
             *            the statistic.
             * 
             * @return Maximum complex query cache size.
             */
            public int getMaxCacheSize(boolean real) {
                int size = getMaxCacheSize();

                if (size < 0) {
                    if (real) {
                        return -1;
                    } else {
                        return getCacheSize();
                    }
                }
                return size;
            }

            /**
             * Sets maximum size of complex query cache.
             * 
             * @param maxSize
             *            Maximum size of complex query cache.
             */
            protected void setMaxCacheSize(int maxSize) throws CacheObjectException {
                try {
                    if (isDisabled) {
                        throw new CacheObjectException("Caching is disabled");
                    }
                    if (maxSize == 0) {
                        if (complexQCache != null) {
                            statistics.getCacheStatistics(CacheConstants.COMPLEX_QUERY_CACHE).clearStatistics();
                            complexQCache = null;
                        }
                        return;
                    }
                    if (complexQCache == null) {
                        if (cache != null) {
                            complexQCache = BaseCacheManager.getDODSCache(maxSize);
                        }
                    } else {
                        complexQCache.setMaxEntries(maxSize);
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in setMaxCacheSize - complex cache");
                }
            }

            /**
             * Returns size of currently used complex query cache.
             * 
             * @return Size of currently used complex query cache.
             */
            public int getCacheSize() {
                if (complexQCache != null) {
                    return complexQCache.size();
                }
                return 0;
            }

            /**
             * Refreshes complex query cache.
             */
            public void refresh() {
                if (cache != null) {
                    if (complexQCache != null) {
                        complexQCache = BaseCacheManager.getDODSCache(complexQCache.getMaxEntries());
                        statistics.getCacheStatistics(CacheConstants.COMPLEX_QUERY_CACHE).setCacheHitsNum(0);
                    }
                }
            }

            /**
             * Disables complex query cache.
             */
            public void disable() {
                if (!isDisabledComplex) {
                    isDisabledComplex = true;
                    if (complexQCache != null) {
                        disabledMaxComplexQueryCacheSize = complexQCache.getMaxEntries();
                        complexQCache = null;
                    } else {
                        disabledMaxComplexQueryCacheSize = 0;
                    }
                }
            }

            /**
             * Enables complex query cache.
             */
            public void enable() {
                try {
                    if (isDisabledComplex) {
                        if (disabledMaxComplexQueryCacheSize != 0) {
                            if (disabledMaxCacheSize != 0) {
                                complexQCache = BaseCacheManager.getDODSCache(disabledMaxComplexQueryCacheSize);
                            }
                        }
                        isDisabledComplex = false;
                    }
                    DODSCache obj = (DODSCache) statistics.getCacheStatistics(CacheConstants.COMPLEX_QUERY_CACHE);

                    if (obj != null) {
                        obj.clearStatistics();
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in enable complex");
                }
            }
        };

        cacheAdministration[CacheConstants.MULTI_JOIN_QUERY_CACHE] = new CacheAdministration() {

            /**
             * Returns maximum size of multi join query cache.
             * 
             * @return Maximum size of multi join query cache.
             */
            public int getMaxCacheSize() {
                if (multiJoinQCache != null) {
                    return multiJoinQCache.getMaxEntries();
                }
                return 0;
            }

            /**
             * Returns maximum multi join query cache size. If the cache is
             * unbounded (the maximum size is negative), the current cache size
             * is returned.
             * 
             * @param real
             *            If this parameter is true, method returns real maximum
             *            cache size, otherwise returns appropriate value for
             *            the statistic.
             * 
             * @return Maximum multi join query cache size.
             */
            public int getMaxCacheSize(boolean real) {
                int size = getMaxCacheSize();

                if (size < 0) {
                    if (real) {
                        return -1;
                    } else {
                        return getCacheSize();
                    }
                }
                return size;
            }

            /**
             * Sets maximum size of multi join query cache.
             * 
             * @param maxSize
             *            Maximum size of multi join query cache.
             */
            protected void setMaxCacheSize(int maxSize) throws CacheObjectException {
                try {
                    if (isDisabled) {
                        throw new CacheObjectException("Caching is disabled");
                    }
                    if (maxSize == 0) {
                        if (multiJoinQCache != null) {
                            statistics.getCacheStatistics(CacheConstants.MULTI_JOIN_QUERY_CACHE).clearStatistics();
                            multiJoinQCache = null;
                        }
                        return;
                    }
                    if (multiJoinQCache == null) {
                        if (cache != null) {
                            multiJoinQCache = BaseCacheManager.getDODSCache(maxSize);
                        }
                    } else {
                        multiJoinQCache.setMaxEntries(maxSize);
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in setMaxCacheSize - complex cache");
                }
            }

            /**
             * Returns size of currently used multi join query cache.
             * 
             * @return Size of currently used multi join query cache.
             */
            public int getCacheSize() {
                if (multiJoinQCache != null) {
                    return multiJoinQCache.size();
                }
                return 0;
            }

            /**
             * Refreshes multi join query cache.
             */
            public void refresh() {
                if (cache != null) {
                    if (multiJoinQCache != null) {
                        multiJoinQCache = BaseCacheManager.getDODSCache(multiJoinQCache.getMaxEntries());
                        statistics.getCacheStatistics(CacheConstants.MULTI_JOIN_QUERY_CACHE).setCacheHitsNum(0);
                    }
                }
            }

            /**
             * Disables multi join query cache.
             */
            public void disable() {
                if (!isDisabledMultiJoin) {
                    isDisabledMultiJoin = true;
                    if (multiJoinQCache != null) {
                        disabledMaxMultiJoinQueryCacheSize = multiJoinQCache.getMaxEntries();
                        multiJoinQCache = null;
                    } else {
                        disabledMaxMultiJoinQueryCacheSize = 0;
                    }
                }
            }

            /**
             * Enables multi join query cache.
             */
            public void enable() {
                try {
                    if (isDisabledMultiJoin) {
                        if (disabledMaxMultiJoinQueryCacheSize != 0) {
                            if (disabledMaxCacheSize != 0) {
                                multiJoinQCache = BaseCacheManager.getDODSCache(disabledMaxMultiJoinQueryCacheSize);
                            }
                        }
                        isDisabledMultiJoin = false;
                    }
                    DODSCache obj = (DODSCache) statistics.getCacheStatistics(CacheConstants.MULTI_JOIN_QUERY_CACHE);

                    if (obj != null) {
                        obj.clearStatistics();
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in enable complex");
                }
            }
        };

        cacheAdministration[CacheConstants.DATA_CACHE] = new CacheAdministration() {

            /**
             * Returns maximum size of data (or DataStruct) object cache.
             * 
             * @return Maximum data (or DataStruct) object cache size.
             */
            public int getMaxCacheSize() {
                if (cache != null) {
                    return cache.getMaxEntries();
                }
                return 0;
            }

            /**
             * Returns maximum size of data (or DataStruct) object cache. If the
             * cache is unbounded (the maximum size is negative), the current
             * cache size is returned.
             * 
             * @param real
             *            If this parameter is true, method returns real maximum
             *            cache size, otherwise returns appropriate value for
             *            the statistic.
             * 
             * @return Maximum data (or DataStruct) object cache size.
             */
            public int getMaxCacheSize(boolean real) {
                int size = getMaxCacheSize();

                if (size < 0) {
                    if (real) {
                        return -1;
                    } else {
                        return getCacheSize();
                    }
                }
                return size;
            }

            /**
             * Returns size of currently used data (or DataStruct) object cache.
             * 
             * @return Size of currently used data (or DataStruct) object cache.
             */
            public int getCacheSize() {
                if (cache != null) {
                    return cache.size();
                }
                return 0;
            }

            /**
             * Sets maximum size of data (or DataStruct) object cache.
             * 
             * @param maxSize
             *            Maximum DO cache size.
             */
            protected void setMaxCacheSize(int maxSize) throws CacheObjectException {
                try {
                    if (isDisabled) {
                        throw new CacheObjectException("Caching is disabled");
                    }
                    if (maxSize == 0) {
                        cache = null;
                        if (simpleQCache != null) {
                            oldSimpleMaxCacheSize = getCacheAdministration(CacheConstants.SIMPLE_QUERY_CACHE)
                                            .getMaxCacheSize();
                            simpleQCache = null;
                        } else {
                            oldSimpleMaxCacheSize = 0;
                        }
                        if (complexQCache != null) {
                            oldComplexMaxCacheSize = getCacheAdministration(CacheConstants.COMPLEX_QUERY_CACHE)
                                            .getMaxCacheSize();
                            complexQCache = null;
                        } else {
                            oldComplexMaxCacheSize = 0;
                        }
                        if (multiJoinQCache != null) {
                            oldMultiJoinMaxCacheSize = getCacheAdministration(CacheConstants.MULTI_JOIN_QUERY_CACHE)
                                            .getMaxCacheSize();
                            multiJoinQCache = null;
                        } else {
                            oldMultiJoinMaxCacheSize = 0;
                        }
                        statistics.clear();
                        return;
                    }
                    if (cache == null) {
                        cache = BaseCacheManager.getDODSCache(maxSize);
                        if (oldSimpleMaxCacheSize != 0) {
                            simpleQCache = BaseCacheManager.getDODSCache(oldSimpleMaxCacheSize);
                        }
                        if (oldComplexMaxCacheSize != 0) {
                            complexQCache = BaseCacheManager.getDODSCache(oldComplexMaxCacheSize);
                        }
                        if (oldMultiJoinMaxCacheSize != 0) {
                            multiJoinQCache = BaseCacheManager.getDODSCache(oldMultiJoinMaxCacheSize);
                        }
                    } else {
                        cache.setMaxEntries(maxSize);
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in setMaxCacheSize - DO cache");
                }
            }

            /**
             * Refreshes caches (data (or DataStruct) object, simple query and
             * complex query).
             */
            public void refresh() {
                if (cache != null) {
                    cache = BaseCacheManager.getDODSCache(cache.getMaxEntries());
                }
                if (simpleQCache != null) {
                    simpleQCache = BaseCacheManager.getDODSCache(simpleQCache.getMaxEntries());
                }
                if (complexQCache != null) {
                    complexQCache = BaseCacheManager.getDODSCache(complexQCache.getMaxEntries());
                }
                if (multiJoinQCache != null) {
                    multiJoinQCache = BaseCacheManager.getDODSCache(multiJoinQCache.getMaxEntries());
                }
                statistics.clear();
            }

            /**
             * Disables data (or DataStruct) object cache. When this cache is
             * disabled, so are and simple and complex query caches.
             */
            public void disable() {
                if (!isDisabled) {
                    isDisabled = true;
                    if (cache != null) {
                        disabledMaxCacheSize = cache.getMaxEntries();
                        cache = null;
                        if (simpleQCache != null) {
                            getCacheAdministration(CacheConstants.SIMPLE_QUERY_CACHE).disable();
                        }
                        if (complexQCache != null) {
                            cacheAdministration[CacheConstants.COMPLEX_QUERY_CACHE].disable();
                        }
                        if (multiJoinQCache != null) {
                            cacheAdministration[CacheConstants.MULTI_JOIN_QUERY_CACHE].disable();
                        }
                        statistics.clear();
                    }
                }
            }

            /**
             * Enables data (or DataStruct) object cache. When this cache is
             * enabled, so are and simple and complex query caches (if they
             * existed before DO cache had been disabled).
             */
            public void enable() {
                try {
                    if (isDisabled) {
                        if (disabledMaxCacheSize != 0) {
                            cache = BaseCacheManager.getDODSCache(disabledMaxCacheSize);
                        }
                        if (isDisabledSimple) {
                            getCacheAdministration(CacheConstants.SIMPLE_QUERY_CACHE).enable();
                        }
                        if (isDisabledComplex) {
                            getCacheAdministration(CacheConstants.COMPLEX_QUERY_CACHE).enable();
                        }
                        if (isDisabledMultiJoin) {
                            getCacheAdministration(CacheConstants.MULTI_JOIN_QUERY_CACHE).enable();
                        }
                        statistics.clear();
                        isDisabled = false;
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in enable DO");
                }
            }
        };
    }

    /**
     * Returns cache (data or DataStruct) content.
     * 
     * @return Cache content as <code>Map</code> of data (or DataStruct)
     *         objects.
     */
    public Map  getCacheContent() {
        return cache;
    }

    /**
     * Returns information if multi databases are supported.
     * 
     * @return true if multi databases are used, otherwise false.
     */
    public boolean isMulti() {
        return multi;
    }

    /**
     * Checks wheather cache reconfiguration needs to be done.
     * 
     * @return true if cache reconfiguration needs to be done, otherwise false.
     */
    public boolean toReconfigure() {
        return false;
    }

    /**
     * Adds DataStruct object to the cache.
     * 
     * @param newDS
     *            DataStruct object that will be added to the cache.
     * 
     * @return Added DataStruct object.
     */
    public CoreDataStruct addDataStruct(CoreDataStruct newDS) {
        String  handle;
        try {
            handle = newDS.get_CacheHandle();
        } catch (Exception  e) {
            handle = null;
        }
        if (cache == null) {
            return newDS;
        }
        try {
            if (null != handle) {
                synchronized (cache) {
                    cache.add(handle, newDS);
                }
                return newDS;
            }
        } catch (Exception  e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Removes DataStruct object from the cache.
     * 
     * @param data
     *            DataStruct object that will be removed from the cache.
     * 
     * @return Removed DataStruct object, or <tt>null</tt> if there was no
     *         object removed from the cache.
     */
    public CoreDataStruct removeDataStruct(CoreDataStruct data) {
        try {
            String  handle = data.get_CacheHandle();
            if (cache != null) {
                synchronized (cache) {
                    return (CoreDataStruct) cache.remove(handle);
                }
            }
        } catch (Exception  e) {
        }
        return null;
    }

    /**
     * Removes DataStruct object from the cache.
     * 
     * @param handle
     *            Cache handle of DataStruct object that will be removed from
     *            the cache. The form of cache handle is: " <database_name>.
     *            <String_presentation_of_oid>".
     * 
     * @return Removed DataStruct object, or <tt>null</tt> if there was no
     *         object removed from the cache.
     */
    public CoreDataStruct removeDataStruct(String  handle) {
        if (cache != null) {
            synchronized (cache) {
                try {
                    return (CoreDataStruct) cache.remove(handle);
                } catch (Exception  e) {
                }
            }
        }
        return null;
    }

    /**
     * Updates cached DataStruct object, or inserts it in the cache if it didn't
     * exist in the cache.
     * 
     * @param data
     *            DataStruct object that will be updated (or inserted if didn't
     *            exist in the cache).
     * 
     * @return Updated or inserted DataStruct object.
     */
    public CoreDataStruct updateDataStruct(CoreDataStruct data) {
        try {
            data = addDataStruct(data);
            QueryCacheItem queryItem = null;
            HashSet  del;
            Iterator  iter = null;

            if (complexQCache != null) {
                del = new HashSet ();
                synchronized (complexQCache) {
                    for (iter = complexQCache.values().iterator(); iter.hasNext();) {
                        queryItem = (QueryCacheItem) iter.next();
                        if (data.get_Database().equals(queryItem.get_OriginDatabase())) {
                            del.add(queryItem);
                        }
                    }
                } // end synchronized
                iter = null;
                for (iter = del.iterator(); iter.hasNext();) {
                    removeComplexQuery((QueryCacheItem) iter.next());
                }
            }
            if (multiJoinQCache != null) {
                del = new HashSet ();
                synchronized (multiJoinQCache) {
                    for (iter = multiJoinQCache.values().iterator(); iter.hasNext();) {
                        queryItem = (QueryCacheItem) iter.next();
                        if (data.get_Database().equals(queryItem.get_OriginDatabase())) {
                            del.add(queryItem);
                        }
                    }
                } // end synchronized
                iter = null;
                for (iter = del.iterator(); iter.hasNext();) {
                    removeMultiJoinQuery((QueryCacheItem) iter.next());
                }
            }
            if (simpleQCache != null) {
                iter = null;
                synchronized (simpleQCache) {
                    for (iter = simpleQCache.values().iterator(); iter.hasNext();) {
                        queryItem = (QueryCacheItem) iter.next();
                        String  db = data.get_Database();

                        if (db.equals(queryItem.get_OriginDatabase())) {
                            if (queryItem.checkConditions(data)) {
                                if (queryItem.getOIds().contains(data.get_Handle())) {
                                    queryItem.setModifiedQuery(true);
                                } else {
                                    if (queryItem.isCompleteResult()) {
                                        queryItem.add(data);
                                    }
                                    queryItem.setModifiedQuery(true);

                                }
                            } // queryItem.checkConditions(data)
 else {
                                if (queryItem.getOIds().contains(data.get_Handle())) {
                                    queryItem.delete(data);
                                    queryItem.setModifiedQuery(true);
                                }
                            } // else of queryItem.checkConditions(data)
                        } // db.equals(queryItem.get_OriginDatabase())
                    } // for
                } // end synchronized
            } // simpleQCache != null
        } catch (DatabaseManagerException ex) {
        }
        return data;
    }

    /**
     * Deletes DataStruct object from the cache.
     * 
     * @param data
     *            DataStruct object that will be deleted from the cache.
     * 
     * @return Deleted DataStruct object, or <tt>null</tt> if there was no
     *         object deleted from the cache.
     */
    public CoreDataStruct deleteDataStruct(CoreDataStruct data) {
        if (data != null) {
            CoreDataStruct oldDS = removeDataStruct(data);
            Iterator  iter = null;
            QueryCacheItem queryItem = null;

            if (simpleQCache != null) {
                synchronized (simpleQCache) {
                    for (iter = simpleQCache.values().iterator(); iter.hasNext();) {
                        queryItem = (QueryCacheItem) iter.next();
                        try {
                            String  db = data.get_Database();

                            if (db.equals(queryItem.get_OriginDatabase())) {
                                if (queryItem.getOIds().contains(data.get_Handle())) {
                                    queryItem.delete(data);
                                    queryItem.setModifiedQuery(true);
                                }
                            }
                        } catch (Exception  e) {
                            System.out.println("Error in deleteDataStruct of QueryCacheImpl");
                        }
                    }
                } // end synchronized
            }
            iter = null;
            if (complexQCache != null) {
                synchronized (complexQCache) {
                    for (iter = complexQCache.values().iterator(); iter.hasNext();) {
                        queryItem = (QueryCacheItem) iter.next();
                        try {
                            String  db = data.get_Database();

                            if (db.equals(queryItem.get_OriginDatabase())) {
                                if (queryItem.getOIds().contains(data.get_Handle())) {
                                    queryItem.delete(data);
                                    queryItem.setModifiedQuery(true);
                                }
                            }
                        } catch (Exception  e) {
                            System.out.println("Error in deleteDataStruct of QueryCacheImpl");
                        }
                    }
                } // end synchronized
            }
            if (multiJoinQCache != null) {
                synchronized (multiJoinQCache) {
                    for (iter = multiJoinQCache.values().iterator(); iter.hasNext();) {
                        queryItem = (QueryCacheItem) iter.next();
                        try {
                            String  db = data.get_Database();

                            if (db.equals(queryItem.get_OriginDatabase())) {
                                if (queryItem.getOIds().contains(data.get_Handle())) {
                                    queryItem.delete(data);
                                    queryItem.setModifiedQuery(true);
                                }
                            }
                        } catch (Exception  e) {
                            System.out.println("Error in deleteDataStruct of QueryCacheImpl");
                        }
                    }
                } // end synchronized
            }
            return oldDS;
        }
        return null;
    }

    /**
     * Returns DataStruct object whose String representation of OID is parameter
     * handle.
     * 
     * @param cacheHandle
     *            String representation of OID of DataStruct object that is
     *            being searched in the cache.
     * 
     * @return DataStruct object whose String representation of OID is handle.
     */
    public CoreDataStruct getDataStructByHandle(String  cacheHandle) {
        if (cache == null) {
            return null;
        }
        if (isLocked()) {
            return null;
        }
        CoreDataStruct tmpDO = null;
        if(cache.isNeedToSynchronize()) {
            synchronized (cache) {
                tmpDO = getCacheItem(cacheHandle, tmpDO);
            }   
        }else {
            tmpDO = getCacheItem(cacheHandle, tmpDO);
        }
        return tmpDO;
    }

    /**
     * @param cacheHandle
     * @param tmpDO
     * @return
     */
    private CoreDataStruct getCacheItem(String  cacheHandle, CoreDataStruct tmpDO) {
        cache.incrementCacheAccessNum(1);
        if (!nonVisibleList.containsKey(cacheHandle)) {
            tmpDO = (CoreDataStruct) cache.get(cacheHandle);
            if (tmpDO != null) {
                cache.incrementCacheHitsNum(1);
            }
        }
        return tmpDO;
    }

    /**
     * Creates new QueryCacheItem instance.
     * 
     * @param dbName
     *            Database name.
     * @return Created QueryCacheItem instance.
     */
    public QueryCacheItem newQueryCacheItemInstance(String  dbName) {
        return new QueryCacheItemImpl(dbName);
    }

    /**
     * Returns QueryCacheItem object for specified database and simple query, if
     * exists, otherwise null.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query in form of String.
     * @return QueryCacheItem object.
     */
    public QueryCacheItem getSimpleQueryCacheItem(String  dbName, String  query) {
        if (simpleQCache != null && !isLockedSimpleComplexQCache()) {
            return _getItem(simpleQCache, dbName + "." + query);
        }
        return null;
    }

    private QueryCacheItem _getItem(DODSCache whereFrom, String  whichOne) {
       if(whereFrom.isNeedToSynchronize()) {
           synchronized (whereFrom) {
               return (QueryCacheItem) whereFrom.get(whichOne);
           }
       } else {
           return (QueryCacheItem) whereFrom.get(whichOne);
       }
    }
    /**
     * Returns QueryCacheItem object for specified database and complex query,
     * if exists, otherwise null.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query in form of String.
     * @return QueryCacheItem object.
     */
    public QueryCacheItem getComplexQueryCacheItem(String  dbName, String  query) {
        if (complexQCache != null && !isLockedSimpleComplexQCache()) {
           return _getItem(complexQCache, dbName + "." + query);
        }
        return null;
    }

    /**
     * Returns QueryCacheItem object for specified database and multi join
     * query, if exists, otherwise null.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query in form of String.
     * @return QueryCacheItem object.
     */
    public QueryCacheItem getMultiJoinQueryCacheItem(String  dbName, String  query) {
        if (multiJoinQCache != null && !isLockedMultiJoinQCache()) {
           return _getItem(multiJoinQCache, dbName + "." + query);
        }
        return null;
    }

    /**
     * Adds simple query to simple query cache.
     * 
     * @param queryItem
     *            Query that will be added to simple query cache.
     * @return Query added to simple query cache.
     */
    public QueryCacheItem addSimpleQuery(QueryCacheItem queryItem) {
        if (simpleQCache != null) {
            synchronized (simpleQCache) {
                return (QueryCacheItem) simpleQCache.add(queryItem.get_OriginDatabase() + "." + queryItem.getQueryId(),
                                queryItem);
            }
        }
        return null;
    }

    /**
     * Removes simple query from simple query cache.
     * 
     * @param queryItem
     *            Query that will be removed from simple query cache.
     * @return Query removed from simple query cache.
     */
    public QueryCacheItem removeSimpleQuery(QueryCacheItem queryItem) {
        if (simpleQCache != null) {
            synchronized (simpleQCache) {
                return (QueryCacheItem) simpleQCache.remove(queryItem.get_OriginDatabase() + "."
                                + queryItem.getQueryId());
            }
        }
        return null;
    }

    /**
     * Adds complex query to complex query cache.
     * 
     * @param queryItem
     *            Query that will be added to complex query cache.
     * @return Query added to complex query cache.
     */
    public QueryCacheItem addComplexQuery(QueryCacheItem queryItem) {
        if (complexQCache != null) {
            synchronized (complexQCache) {
                return (QueryCacheItem) complexQCache.add(
                                queryItem.get_OriginDatabase() + "." + queryItem.getQueryId(), queryItem);
            }
        }
        return null;
    }

    /**
     * Removes complex query from complex query cache.
     * 
     * @param queryItem
     *            Query that will be removed from complex query cache.
     * @return Query removed from complex query cache.
     */
    public QueryCacheItem removeComplexQuery(QueryCacheItem queryItem) {
        if (complexQCache != null) {
            synchronized (complexQCache) {
                return (QueryCacheItem) complexQCache.remove(queryItem.get_OriginDatabase() + "."
                                + queryItem.getQueryId());
            }
        }
        return null;
    }

    /**
     * Adds multi join query to multi join query cache.
     * 
     * @param queryItem
     *            Query that will be added to multi join query cache.
     * @return Query added to multi join query cache.
     */
    public QueryCacheItem addMultiJoinQuery(QueryCacheItem queryItem) {
        if (multiJoinQCache != null) {
            synchronized (multiJoinQCache) {
                return (QueryCacheItem) multiJoinQCache.add(queryItem.get_OriginDatabase() + "."
                                + queryItem.getQueryId(), queryItem);
            }
        }
        return null;
    }

    /**
     * Removes multi join query from multi join query cache.
     * 
     * @param queryItem
     *            Query that will be removed from multi join query cache.
     * @return Query removed from multi join query cache.
     */
    public QueryCacheItem removeMultiJoinQuery(QueryCacheItem queryItem) {
        if (multiJoinQCache != null) {
            synchronized (multiJoinQCache) {
                return (QueryCacheItem) multiJoinQCache.remove(queryItem.get_OriginDatabase() + "."
                                + queryItem.getQueryId());
            }
        }
        return null;
    }

    /**
     * Returns query results from simple query cache.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query for which are results searched in simple query cache.
     * @return Query results retrieved from simple cache, or null, if there are
     *         no results retrieved from simple query cache.
     */
    public QueryResult getSimpleQueryResults(String  dbName, String  query) {
        return getSimpleQueryResults(dbName, query, 0, 0, false);
    }

    /**
     * Returns query results from simple query cache.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query for which are results searched in simple query cache.
     * @param limit
     *            Specified number of results (database limit and read skip).
     * @param maxdb
     *            Number of rows retrieved from database (or cache).
     * @return Query results retrieved from simple cache, or null, if there are
     *         no results retrieved from simple query cache.
     */
    public QueryResult getSimpleQueryResults(String  dbName, String  query, int limit, int maxdb) {
        return getSimpleQueryResults(dbName, query, limit, maxdb, false);
    }

    /**
     * Returns query results from simple query cache.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query for which are results searched in simple query cache.
     * @param limit
     *            Specified number of results (database limit and read skip).
     * @param maxdb
     *            Number of rows retrieved from database (or cache).
     * @param unique
     *            If true, only unique results are returned.
     * @return Query results retrieved from simple cache, or null, if there are
     *         no results retrieved from simple query cache.
     */
    public QueryResult getSimpleQueryResults(String  dbName, String  query, int limit, int maxdb, boolean unique) {
        if (simpleQCache == null || isLockedSimpleComplexQCache()) {
            return null;
        }
        QueryResult result = null;
        String  queryHandle = dbName + "." + query;
        QueryCacheItem queryItem = _getItem(simpleQCache, queryHandle);
        int i = 0;

        if (queryItem != null) {
            synchronized (queryItem) {
                result = new QueryResult();
                result.database = queryItem.get_OriginDatabase();
                String  handle = null;
                String  cachePrefix = queryItem.get_OriginDatabase() + ".";
                Iterator  iter = queryItem.getOIds().iterator();

                if (unique) {
                    HashSet  allResultOids = new HashSet ();
                    int skippedNum = 0;

                    while ((maxdb == 0 || i < maxdb) && (iter.hasNext()) && ((limit == 0 || result.DOs.size() < limit))) {
                        handle = (String ) iter.next();
                        if (allResultOids.contains(handle)) {
                            skippedNum++;
                        } else {
                            allResultOids.add(handle);
                            result.DOs.add(handle);

                        }
                        i++;
                    } // while
                    result.skippedUnique = skippedNum;
                } // (unique)
 else {
                    while ((maxdb == 0 || i < maxdb) && (iter.hasNext()) && ((limit == 0 || result.DOs.size() < limit))) {
                        handle = (String ) iter.next();
                        result.DOs.add(handle);
                        i++;
                    } // while
                } // (unique)
            } // (queryItem != null)
        }
        if (result.DOs.size() < limit) {
            if ((maxdb == 0) || (i < maxdb)) {
                result = null;
            }
        }
        return result;
    }

    /**
     * Returns query results from complex query cache.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query for which are results searched in complex query cache.
     * @return Query results retrieved from complex cache, or null, if there are
     *         no results retrieved from complex query cache.
     */
    public QueryResult getComplexQueryResults(String  dbName, String  query) {
        return getComplexQueryResults(dbName, query, 0, 0, false);
    }

    /**
     * Returns query results from complex query cache.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query for which are results searched in complex query cache.
     * @param limit
     *            Specified number of results (database limit and read skip).
     * @param maxdb
     *            Number of rows retrieved from database (or cache).
     * @return Query results retrieved from complex cache, or null, if there are
     *         no results retrieved from complex query cache.
     */
    public QueryResult getComplexQueryResults(String  dbName, String  query, int limit, int maxdb) {
        return getComplexQueryResults(dbName, query, limit, maxdb, false);
    }

    /**
     * Returns query results from complex query cache.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query for which are results searched in complex query cache.
     * @param limit
     *            Specified number of results (database limit and read skip).
     * @param maxdb
     *            Number of rows retrieved from database (or cache).
     * @param unique
     *            If true, only unique results are returned.
     * @return Query results retrieved from complex cache, or null, if there are
     *         no results retrieved from complex query cache.
     */
    public QueryResult getComplexQueryResults(String  dbName, String  query, int limit, int maxdb, boolean unique) {
        if (complexQCache == null || isLockedSimpleComplexQCache()) {
            return null;
        }
        QueryResult result = null;
        String  queryHandle = dbName + "." + query;
        QueryCacheItem queryItem = _getItem(complexQCache, queryHandle);
        int i = 0;

        if (queryItem != null) {
            synchronized (queryItem) {
                result = new QueryResult();
                result.database = queryItem.get_OriginDatabase();
                String  handle = null;
                String  cachePrefix = queryItem.get_OriginDatabase() + ".";
                Iterator  iter = queryItem.getOIds().iterator();

                if (unique) {
                    HashSet  allResultOids = new HashSet ();
                    int skippedNum = 0;

                    while ((maxdb == 0 || i < maxdb) && (iter.hasNext()) && ((limit == 0 || result.DOs.size() < limit))) {
                        handle = (String ) iter.next();
                        if (allResultOids.contains(handle)) {
                            skippedNum++;
                        } else {
                            allResultOids.add(handle);
                            result.DOs.add(handle);
                        }
                        i++;
                    } // while
                    result.skippedUnique = skippedNum;
                } // (unique)
 else {
                    while ((maxdb == 0 || i < maxdb) && (iter.hasNext()) && ((limit == 0 || result.DOs.size() < limit))) {
                        handle = (String ) iter.next();
                        result.DOs.add(handle);
                        i++;
                    } // while
                } // (unique)
            }
        } // (queryItem != null)
 if (result.DOs.size() < limit) {
            if ((maxdb == 0) || (i < maxdb)) {
                result = null;
            }
        }
        return result;
    }

    /**
     * Returns query results from multi join query cache.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query for which are results searched in multi join query
     *            cache.
     * @return Query results retrieved from multi join cache, or null, if there
     *         are no results retrieved from multi join query cache.
     */
    public QueryResult getMultiJoinQueryResults(String  dbName, String  query) {
        return getMultiJoinQueryResults(dbName, query, 0, 0, false);
    }

    /**
     * Returns query results from multi join query cache.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query for which are results searched in multi join query
     *            cache.
     * @param limit
     *            Specified number of results (database limit and read skip).
     * @param maxdb
     *            Number of rows retrieved from database (or cache).
     * @return Query results retrieved from multi join cache, or null, if there
     *         are no results retrieved from multi join query cache.
     */
    public QueryResult getMultiJoinQueryResults(String  dbName, String  query, int limit, int maxdb) {
        return getMultiJoinQueryResults(dbName, query, limit, maxdb, false);
    }

    /**
     * Returns query results from multi join query cache.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query for which are results searched in multi join query
     *            cache.
     * @param limit
     *            Specified number of results (database limit and read skip).
     * @param maxdb
     *            Number of rows retrieved from database (or cache).
     * @param unique
     *            If true, only unique results are returned.
     * @return Query results retrieved from multi join cache, or null, if there
     *         are no results retrieved from multi join query cache.
     */
    public QueryResult getMultiJoinQueryResults(String  dbName, String  query, int limit, int maxdb, boolean unique) {
        if (multiJoinQCache == null || isLockedMultiJoinQCache()) {
            return null;
        }
        QueryResult result = null;
        String  queryHandle = dbName + "." + query;
        QueryCacheItem queryItem = _getItem(multiJoinQCache, queryHandle);
        int i = 0;

        if (queryItem != null) {
            synchronized (queryItem) {
                result = new QueryResult();
                result.database = queryItem.get_OriginDatabase();
                String  handle = null;
                String  cachePrefix = queryItem.get_OriginDatabase() + ".";
                Iterator  iter = queryItem.getOIds().iterator();

                if (unique) {
                    HashSet  allResultOids = new HashSet ();
                    int skippedNum = 0;

                    while ((maxdb == 0 || i < maxdb) && (iter.hasNext()) && ((limit == 0 || result.DOs.size() < limit))) {
                        handle = (String ) iter.next();
                        if (allResultOids.contains(handle)) {
                            skippedNum++;
                        } else {
                            allResultOids.add(handle);
                            result.DOs.add(handle);
                        }
                        i++;
                    } // while
                    result.skippedUnique = skippedNum;
                } // (unique)
 else {
                    while ((maxdb == 0 || i < maxdb) && (iter.hasNext()) && ((limit == 0 || result.DOs.size() < limit))) {
                        handle = (String ) iter.next();
                        result.DOs.add(handle);
                        i++;
                    } // while
                } // (unique)
            }
        } // (queryItem != null)
 if (result.DOs.size() < limit) {
            if ((maxdb == 0) || (i < maxdb)) {
                result = null;
            }
        }
        return result;
    }

    /**
     * Returns query results from simple, complex or multi join query cache.
     * 
     * @param dbName
     *            Database name.
     * @param query
     *            Query for which are results searched in simple, complex or
     *            multi join query cache.
     * @return Query results retrieved from simple, complex or multi join cache,
     *         or null, if there are no results retrieved from simple, complex
     *         or multi join query cache.
     */
    public QueryResult getQueryResults(String  dbName, String  query) {
        QueryResult result = getSimpleQueryResults(dbName, query);

        if (result == null) {
            result = getComplexQueryResults(dbName, query);
        }
        if (result == null) {
            result = getMultiJoinQueryResults(dbName, query);
        }
        return result;
    }

    /**
     * Inner class that implements TableStatistics. This is class for
     * administrating table and cache.
     */
    private class QueryCacheImplStatistics extends TableStatistics {

        /**
         * Constructor().
         */
        public QueryCacheImplStatistics() {
            try {
                this.reset();
                if (cache != null) {
                    getCacheStatistics(CacheConstants.DATA_CACHE).clearStatistics();
                }
                if (simpleQCache != null) {
                    getCacheStatistics(CacheConstants.SIMPLE_QUERY_CACHE).clearStatistics();
                }
                if (complexQCache != null) {
                    getCacheStatistics(CacheConstants.COMPLEX_QUERY_CACHE).clearStatistics();
                }
                if (multiJoinQCache != null) {
                    getCacheStatistics(CacheConstants.MULTI_JOIN_QUERY_CACHE).clearStatistics();
                }
            } catch (Exception  ex) {
            }
        }

        /**
         * Resets parameters.
         */
        public void reset() {
            insertNum = 0;
            updateNum = 0;
            deleteNum = 0;
            lazyLoadingNum = 0;
            startTime = new Date ();
            stopTime = new Date ();
            queryNum = 0;
            queryByOIdNum = 0;
            averageQueryTime = 0;
            averageQueryByOIdTime = 0;
        }

        /**
         * Returns type of the statistics. In this case, this is
         * QUERY_CACHE_STATISTICS.
         * 
         * @return Type of the statistics: QUERY_CACHE_STATISTICS.
         */
        public int getStatisticsType() {
            return QUERY_CACHE_STATISTICS;
        }

        /**
         * Clears statistics.
         */
        public void clear() {
            this.reset();
            if (cache != null) {
                getCacheStatistics(CacheConstants.DATA_CACHE).clearStatistics();
            }
            if (simpleQCache != null) {
                getCacheStatistics(CacheConstants.SIMPLE_QUERY_CACHE).clearStatistics();
            }
            if (complexQCache != null) {
                getCacheStatistics(CacheConstants.COMPLEX_QUERY_CACHE).clearStatistics();
            }
            if (multiJoinQCache != null) {
                getCacheStatistics(CacheConstants.MULTI_JOIN_QUERY_CACHE).clearStatistics();
            }
        }

        /**
         * Returns query statistics for DO cache, simple query or complex query
         * cache.
         * 
         * @param type
         *            Indicator of cache for which query statistics is returned
         *            for. Possible values:
         *            org.enhydra.dods.cache.CacheConstants.DATA_CACHE (value 0)
         *            for query statistics of DO cache
         *            org.enhydra.dods.cache.CacheConstants.SIMPLE_QUERY_CACHE
         *            (value 1) for query statistics of simple query cache
         *            org.enhydra.dods.cache.CacheConstants.COMPLEX_QUERY_CACHE
         *            (value 2) for query statistics of complex query cache
         *            org.enhydra.dods.cache.CacheConstants.MULTI_JOIN_QUERY_CACHE
         *            (value 3) for query statistics of multi join query cache
         * 
         * @return Query statistics for specified cache.
         */
        public CacheStatistics getCacheStatistics(int type) {
            switch (type) {
            case CacheConstants.DATA_CACHE:
                return cache;

            case CacheConstants.SIMPLE_QUERY_CACHE:
                return simpleQCache;

            case CacheConstants.COMPLEX_QUERY_CACHE:
                return complexQCache;

            case CacheConstants.MULTI_JOIN_QUERY_CACHE:
                return multiJoinQCache;

            default:
                return null;
            }
        }
    }

    /**
     * Shows content of this class. Can be used for debugging.
     */
    public String  toString() {
        StringBuffer  ret = new StringBuffer ();

        ret.append("\n QueryCacheImpl: ");
        ret.append("\n cache: " + cache);
        ret.append("\n simpleQCache : " + simpleQCache);
        ret.append("\n complexQCache : " + complexQCache);
        ret.append("\n multiJoinQCache : " + multiJoinQCache);
        ret.append("\n cacheReadOnly : " + tableConf.isReadOnly());
        ret.append("\n initialQueryCache : " + initialQueryCache);
        ret.append("\n fullCaching : " + isFull());
        return ret.toString();
    }

    public void removeEntries(Vector  vec) {
        //System.err.println(getClass()+".removeEntries(Vector)");
        _refreshSimpleQuery();
        _refreshComplexQuery();
        _refreshMultiJoinQuery();

        if (cache.size() > vec.size()) {
            for (Enumeration  e = vec.elements(); e.hasMoreElements();) {
                removeDataStruct((String ) e.nextElement());
            }
        } else {
            Iterator  it = cache.values().iterator();
            Vector  vecToRemove = new Vector ();
            while (it.hasNext()) {
                CoreDataStruct obj = (CoreDataStruct) it.next();
                try {
                    if (vec.contains((obj.get_CacheHandle())))
                        vecToRemove.add(obj);
                } catch (DatabaseManagerException e) {
                }
            }
            for (Enumeration  e = vecToRemove.elements(); e.hasMoreElements();) {
                removeDataStruct((String ) e.nextElement());
            }
        }
    }

    public void removeEntries() {
        //System.err.println(getClass()+".removeEntries()");
        _refreshSimpleQuery();
        _refreshComplexQuery();
        _refreshMultiJoinQuery();

        if (cache != null)
            synchronized (cache) {
                cache.clear();
                cache.clearStatistics();
                // -dp cache = new DODSLinkedHashCache(cache.getMaxEntries());
            }
    }

    public void emptyEntries(Vector  vec, boolean incrementVersion) {
        //System.err.println(getClass()+".emptyEntries(Vector)");
        _refreshSimpleQuery();
        _refreshComplexQuery();
        _refreshMultiJoinQuery();

        for (Enumeration  e = vec.elements(); e.hasMoreElements();) {
            String  cacheHandle = (String ) e.nextElement();
            CoreDataStruct ds = (CoreDataStruct) cache.get(cacheHandle);
            if (ds != null) {
                updateDataStruct(ds.dumpData(incrementVersion));
            }
        }
    }

    public void emptyEntries() {
        //System.err.println(getClass()+".emptyEntries()");
        _refreshSimpleQuery();
        _refreshComplexQuery();
        _refreshMultiJoinQuery();

        Collection  c = cache.values();
        Iterator  it = c.iterator();
        while (it.hasNext()) {
            CoreDataStruct ds = (CoreDataStruct) it.next();
            updateDataStruct(ds.dumpData(false));
        }
    }

    private void _refreshSimpleQuery() {
        if (simpleQCache != null)
            synchronized (simpleQCache) {
                simpleQCache.clear();
                simpleQCache.clearStatistics();
                // -dp simpleQCache = new
 // DODSLinkedHashCache(simpleQCache.getMaxEntries());
            }
    }

    private void _refreshComplexQuery() {
        if (complexQCache != null)
            synchronized (complexQCache) {
                complexQCache.clear();
                complexQCache.clearStatistics();
                // -dp complexQCache = new
 // DODSLinkedHashCache(complexQCache.getMaxEntries());
            }
    }

    private void _refreshMultiJoinQuery() {
        if (multiJoinQCache != null)
            synchronized (multiJoinQCache) {
                multiJoinQCache.clear();
                multiJoinQCache.clearStatistics();
                //                multiJoinQCache =
 //                    new DODSLinkedHashCache(multiJoinQCache.getMaxEntries());
            }
    }

    /**
     * @return
     */
    public int getInitialCacheFetchSize() {
        return initialCacheFetchSize;
    }

    /**
     * @return
     */
    public int getInitialDSCacheSize() {
        return initialDSCacheSize;
    }

    /**
     * @param i
     */
    public void setInitialCacheFetchSize(int i) {
        initialCacheFetchSize = i;
    }

    /**
     * @param i
     */
    public void setInitialDSCacheSize(int i) {
        initialDSCacheSize = i;
    }

}