
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package org.enhydra.dods.cache;

import java.util.HashSet ;
import java.util.Iterator ;
import java.util.Map ;
import java.util.Vector ;
import java.util.Date ;
import java.util.Enumeration ;
import java.util.Collection ;
import org.enhydra.dods.statistics.CacheStatistics;
import org.enhydra.dods.statistics.Statistics;
import org.enhydra.dods.statistics.TableStatistics;
import org.enhydra.dods.cache.CacheAdministration;
import org.enhydra.dods.cache.CacheConstants;
import org.enhydra.dods.cache.DODSHashMap;
import org.enhydra.dods.cache.TableConfiguration;
import org.enhydra.dods.cache.TransactionQueryCache;
import org.enhydra.dods.cache.base.BaseCacheManager;
import org.enhydra.dods.cache.base.DODSCache;
import org.enhydra.dods.cache.lru.DODSLRUCache;
import org.enhydra.dods.exceptions.CacheObjectException;
import com.lutris.dods.builder.generator.dataobject.GenericDO;
import com.lutris.util.Config;
import com.lutris.appserver.server.sql.DatabaseManagerException;

/**
 * This class contains data and mechanisms needed for caching data objects (or
 * DataStruct objects) and queries and provides cache configuration and
 * administration.
 *
 * @author    Tanja Jovanovic
 * @author    Sinisa Milosevic  
 * @version   1.0  05.08.2003.
 */
public class TransactionCacheImpl extends TransactionQueryCache {

    /**
     * HashMap for storing data objects. Cache keys are cache handles (String 
     * "<database_name>.<String_presentation_of_oid>"), and cache values are 
     * data objects.
     */
    protected DODSHashMap cache = null;

    /**
     * LRU cache for storing simple queries. LRU cache keys are Strings in form
     * of "<query_database_name>.<String_presentation_of_query>", and LRU cache
     * values are objects of org.enhydra.dods.cache.QueryCacheItem class.
     * QueryCacheItem class stores one query and its necessary data.
     */
    protected DODSCache simpleQCache = null;

    /**
     * LRU cache for storing complex queries. LRU cache keys are Strings in form
     * of "<query_database_name>.<String_presentation_of_query>", and LRU cache
     * values are objects of org.enhydra.dods.cache.QueryCacheItem class.
     * QueryCacheItem class stores one query and its necessary data.
     */
    protected DODSCache complexQCache = null;

    /**
     * Attribute cacheAdministration is array of three objects of the class
     * CacheAdministration.
     * CacheAdministration[0] handles configuration settings about data (or
     * DataStruct object) cache.
     * CacheAdministration[1] handles configuration settings about simple query
     * cache.
     * CacheAdministration[2] handles configuration settings about complex query
     * cache.
     */
    protected CacheAdministration[] cacheAdministration = new CacheAdministration[3];

    /**
     * TableConfiguration attribute handles configuration settings about table
     * of this cache.
     */
    protected TableConfiguration tableConf = new TableConfiguration();

    /**
     * Initial query statement. This attribute contains "where" clause which is
     * used during data (or DataStruct) object cache initialization.
     * If this attribute is set to "*", all rows from the table (in database)
     * will be put in the cache.
     */
    protected String  initialQueryCache = null;

    /**
     * This attribute is true if the cache is full, otherwise false.
     */
    protected boolean fullCachingOn = false;
    
    /**
     * This attribute contains information if multi databases are used.
     */
    protected boolean multi = false;

    /**
     * Table and cache statictics.
     */
    protected Statistics statistics = null;

    /**
     * Old max simple cache size. When data (or DataStruct) cache's maxCacheSize
     * is set to zero, so is simple query's. This attribute is needed for
     * retrieving max simple cache size after data (or DataStruct) cache's
     *  maxCacheSize has been set again to any value different from zero.
     */
    protected int oldSimpleMaxCacheSize;

    /**
     * Old max complex cache size. When data (or DataStruct) cache's maxCacheSize
     * is set to zero, so is complex query's. This attribute is needed for
     * retrieving max complex cache size after data (or DataStruct) cache's
     *  maxCacheSize has been set again to any value different from zero.
     */
    protected int oldComplexMaxCacheSize;

    /**
     * This attribute is used in query caching. It is percent of how many more
     * object are taken for evaluation. If <code>num</code> is number of needed
     * results, then  it is used
     * <code>num</code> + DEFAULT_RESERVE_FACTOR * <code>num</code> of
     * objects for estimating what is quicker: go to database for all object that
     * are not in the cache, or run again query on database.
     * This value is given in percents, as number between 0 and 1
     * (0.25 means 25%).
     * For example, if DEFAULT_RESERVE_FACTOR is 0.5, and wanted number of
     * results is 50, the estimation will be done on 75 (50 + 0.5 * 50) objects.
     * The value of CacheConstants.DEFAULT_RESERVE_FACTOR is 0.5.
     */
    protected double reserveFactor = CacheConstants.DEFAULT_RESERVE_FACTOR;
    protected double cachePercentage = CacheConstants.DEFAULT_CACHE_PERCENTAGE;


    private int initialCacheFetchSize = CacheConstants.DEFAULT_INITIAL_CACHE_FETCH_SIZE;

    private int initialDSCacheSize = CacheConstants.DEFAULT_INITIAL_DS_CACHE_SIZE;



    /**
     * Indicator for disabled data (or DataStruct) cache cache.
     */
    private boolean isDisabled = false;

    /**
     * Indicator for disabled simple query cache.
     */
    private boolean isDisabledSimple = false;

    /**
     * Indicator for disabled complex query cache.
     */
    private boolean isDisabledComplex = false;

    /**
     * Maximal cache size of data (or DataStruct) cache cache before it had been
     * disabled.
     */
    private int disabledMaxCacheSize = 0;

    /**
     * Maximal cache size of simple query cache before it had been disabled.
     */
    private int disabledMaxSimpleQueryCacheSize = 0;

    /**
     * Maximal cache size of complex query cache before it had been disabled.
     */
    private int disabledMaxComplexQueryCacheSize = 0;

    /**
     * Constructor(int, int).
     * Creates unlimited data object cache.
     * and simple and complex query caches with their default maximal sizes.
     * The value of CacheConstants.DEFAULT_MAX_SIMPLE_QUERY_CACHE_SIZE is 0.
     * The value of CacheConstants.DEFAULT_MAX_COMPLEX_QUERY_CACHE_SIZE is 0.
     *
     * @param  maxSQSize maximal data (or DataStruct) object cache size.
     * @param  maxCQSize maximal data (or DataStruct) object cache size.
     */
    public TransactionCacheImpl(int maxSQSize, int maxCQSize) throws CacheObjectException {
        if (isDisabled) {
            throw new CacheObjectException("Caching is disabled");
        }
        cache = new DODSHashMap();
        simpleQCache = BaseCacheManager.getDODSCache(maxSQSize);
        complexQCache = BaseCacheManager.getDODSCache(maxCQSize);
        statistics = new QueryCacheImplStatistics();
        init();
    }

    /**
     * Constructor().
     * Creates unlimited data object cache.
     * and simple and complex query caches with their default maximal sizes.
     * The value of CacheConstants.DEFAULT_MAX_SIMPLE_QUERY_CACHE_SIZE is 0.
     * The value of CacheConstants.DEFAULT_MAX_COMPLEX_QUERY_CACHE_SIZE is 0.
     *
     */
    public TransactionCacheImpl() throws CacheObjectException {
        if (isDisabled) {
            throw new CacheObjectException("Caching is disabled");
        }
        cache = new DODSHashMap();
        simpleQCache = BaseCacheManager.getDODSCache(0);
        complexQCache = BaseCacheManager.getDODSCache(0);
        statistics = new QueryCacheImplStatistics();
        init();
    }

    /**
     * Returns CacheAdministration for data object (or DataStruct object) cache,
     * simple, or complex query cache. Object CacheAdministration handles
     * configuration settings about these caches.
     * Parameter cacheType can have one of these values:
     *  0 - for CacheAdministration of data object (or DataStruct object) cache
     *  1 - for CacheAdministration of simple query cache
     *  2 - for CacheAdministration of complex query cache
     *
     * @param cacheType  0 - for data object (or DataStruct object),
     * 1 for simple query and 2 for complex query cache.
     */
    public CacheAdministration getCacheAdministration(int cacheType) {
        if (cacheType < 0 || cacheType > 2) {
            return null;
        }
        return cacheAdministration[cacheType];
    }

    /**
     * Returns initialQueryCache. This attribute contains "where" clause
     * which is used during data object (or DataStruct object) cache
     * initialization.
     *
     * @return initialQueryCache.
     */
    public String  getInitialQueryCache() {
        return initialQueryCache;
    }

    /**
     * Sets initialQueryCache attribute. This attribute contains "where" clause
     * which is used during data object (or DataStruct object) cache
     * initialization.
     *
     * @param initQ New value of initialQueryCache attribute.
     */
    protected void setInitialQueryCache(String  initQ) {
        initialQueryCache = initQ;
    }

    /**
     * Returns statistics of used table and caches.
     *
     * @return statistics of used table and caches.
     */
    public Statistics getStatistics() {
        statistics.stopTime();
        return statistics;
    }

    /**
     * Refreshes statistics.
     */
    public void refreshStatistics() {
        statistics.clear();
    }

    /**
     * Returns information if data object (or DataStruct object) cache if "full".
     * "Full" cache contains all data objects (or DataStruct objects) from
     * the table. The cache is "full", if data (or DataStruct) object cache is
     * unbounded, if the cached table is read-only, and if initialQueryCache
     * attribute is set to "*".
     *
     * @return true if data object (or DataStruct object) cache if "full",
     * otherwise false.
     */
    public void checkFull() {
        if ((cacheAdministration[0].getMaxCacheSize() < 0)
                && (getInitialQueryCache() != null)
                && (getInitialQueryCache().equals("*"))) {
            fullCachingOn = true;
        } else {
            fullCachingOn = false;
        }
    }

    /**
     * Returns information if data object (or DataStruct object) cache if "full".
     * "Full" cache contains all data objects (or DataStruct objects) from
     * the table. The cache is "full", if data (or DataStruct) object cache is
     * unbounded, and if initialQueryCache attribute is set to "*".
     *
     * @return true if data object (or DataStruct object) cache if "full",
     * otherwise false.
     */
    public boolean isFull() {
        if (fullCachingOn) {
            checkFull();
        }       
        return fullCachingOn;
    }

    /**
     * Returns data object (or DataStruct object) cache type.
     * Possible values are:
     *   "none" - no caching available
     *   "lru"  - cache with LRU (least-recently used) algorithm
     *   "full" - special case of LRU cache - the whole table is cached
     *
     * @return data object (or DataStruct object) cache type.
     */
    public String  getCacheType() {
        if (cacheAdministration[CacheConstants.DATA_CACHE].getMaxCacheSize()
                == 0) {
            return "none";
        } else {
            if (isFull()) {
                return "full";
            } else {
                return "lru";
            }
        }
    }

    /**
     * Returns caching level.
     * Possible values:
     *  org.enhydra.dods.cache.CacheConstants.DATA_CACHING (value 1) for
     *      data object (or DataStruct object) caching without query caching, and
     *  org.enhydra.dods.cache.CacheConstants.QUERY_CACHING (value 2) for
     *      data object (or DataStruct object) caching with query caching.
     *
     * @return Value 2 (org.enhydra.dods.cache.CacheConstants.QUERY_CACHING).
     */
    public int getLevelOfCaching() {
        return CacheConstants.QUERY_CACHING;
    }

    /**
     * Returns object TableConfiguration.
     * TableConfiguration contains parameters (and their set and get methods)
     * for table configuration.
     *
     * @return TableConfiguration.
     */
    public TableConfiguration getTableConfiguration() {
        return tableConf;
    }

    /**
     * Returns reserveFactor.
     * This attribute is used in query caching. It defines how many more (exept
     * needed) objects are taken for evaluation. If <code>num</code> is number
     * of needed results, then it is used
     * <code>num</code> + DEFAULT_RESERVE_FACTOR * <code>num</code>
     * of objects for estimating what is quicker: go to database for all object
     * that are not in the cache, or run again query on database.
     * This value is given in percents, as number between 0 and 1
     * (0.25 means 25%).
     * For example, if DEFAULT_RESERVE_FACTOR is 0.5, and wanted number of
     * results is 50, the estimation will be done on 75 (50 + 0.5 * 50) objects.
     *
     * @return reserveFactor.
     */
    public double getReserveFactor() {
        return reserveFactor;
    }

    /**
     * Sets reserveFactor.
     * This attribute is used in query caching. It defines how many more (exept
     * needed) objects are taken for evaluation. If <code>num</code> is number
     * of needed results, then it is used
     * <code>num</code> + DEFAULT_RESERVE_FACTOR * <code>num</code>
     * of objects for estimating what is quicker: go to database for all object
     * that are not in the cache, or run again query on database.
     * This value is given in percents, as number between 0 and 1
     * (0.25 means 25%).
     * For example, if DEFAULT_RESERVE_FACTOR is 0.5, and wanted number of
     * results is 50, the estimation will be done on 75 (50 + 0.5 * 50) objects.
     *
     * @param res New reserveFactor.
     */
    protected void setReserveFactor(double res) {
        reserveFactor = res;
    }

    /**
     *
     */
    protected void setCachePercentage(double percent) {
        cachePercentage = percent;
    }

    /**
     *
     */
    public double getCachePercentage() {
        return cachePercentage;
    }

    /**
     * Returns information whether caching is disabled.
     *
     * @return true is caching is disabled, otherwise false.
     */
    public boolean isDisabled() {
        return isDisabled;
    }

    /**
     * Reads table and cache configuration from application's configuration file.
     *
     * @param tableConfig configuration for table of this cache.
     * @param cacheConfig configuration for this cache.
     */
    public void readConfiguration(Config tableConfig, Config cacheConfig, String  dbName) throws CacheObjectException {
        if (isDisabled) {
            throw new CacheObjectException("Caching is disabled");
        }
        int maxSize = -1;
        int maxSimple = 0;
        int maxComplex = 0;

        cacheAdministration[CacheConstants.DATA_CACHE].setMaxCacheSize(maxSize);
        cacheAdministration[CacheConstants.SIMPLE_QUERY_CACHE].setMaxCacheSize(maxSimple);
        cacheAdministration[CacheConstants.COMPLEX_QUERY_CACHE].setMaxCacheSize(maxComplex);
    }

    /**
     * Creates TransactionCacheImpl instance.
     *
     * @return created QueryCacheImpl instance as DataCache.
     */
    public DOCache newInstance() throws CacheObjectException {
        if (isDisabled) {
            throw new CacheObjectException("Caching is disabled");
        }
        return new TransactionCacheImpl();
    }

    /**
     * Creates array of three CacheAdministration objects.
     * CacheAdministration[0] handles configuration settings about data (or
     * DataStruct) object cache.
     * CacheAdministration[1] handles configuration settings about simple query
     * cache.
     * CacheAdministration[2] handles configuration settings about complex query
     * cache.
     */
    protected void init() {
        cacheAdministration[CacheConstants.SIMPLE_QUERY_CACHE] = new CacheAdministration() {

            /**
             * Returns maximum size of simple query cache.
             *
             * @return Maximum size of simple query cache.
             */
            public int getMaxCacheSize() {
                if (simpleQCache != null) {
                    return simpleQCache.getMaxEntries();
                }
                return 0;
            }

            /**
             * Returns maximum simple query cache size. If the cache is unbounded
             * (the maximum size is negative), the current cache size is returned.
             *
             * @param real If this parameter is true, method returns real
             * maximum cache size, otherwise returns appropriate value for the
             * statistic.
             * @return Maximum simple query cache size.
             */
            public int getMaxCacheSize(boolean real) {
                int size = getMaxCacheSize();

                if (size < 0) {
                    if (real) {
                        return -1;
                    } else {
                        return getCacheSize();
                    }
                }
                return size;
            }

            /**
             * Sets maximum size of simple query cache.
             *
             * @param maxSize Maximum size of simple query cache.
             */
            protected void setMaxCacheSize(int maxSize) throws CacheObjectException {
                try {
                    if (isDisabled) {
                        throw new CacheObjectException("Caching is disabled");
                    }
                    if (maxSize == 0) {
                        if (simpleQCache != null) {
                            statistics.getCacheStatistics(CacheConstants.SIMPLE_QUERY_CACHE).clearStatistics();
                            simpleQCache = null;
                        }
                        return;
                    }
                    if (simpleQCache == null) {
                        if (cache != null) {
                            simpleQCache = BaseCacheManager.getDODSCache(maxSize);
                        }
                    } else {
                        simpleQCache.setMaxEntries(maxSize);
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in setMaxCacheSize - simple cache");
                }
            }

            /**
             * Returns size of currently used simple query cache.
             *
             * @return Size of currently used simple query cache.
             */
            public int getCacheSize() {
                if (simpleQCache != null) {
                    return simpleQCache.size();
                }
                return 0;
            }

            /**
             * Refreshes simple query cache.
             */
            public void refresh() {
                if (cache != null) {
                    if (simpleQCache != null) {
                        simpleQCache = BaseCacheManager.getDODSCache(simpleQCache.getMaxEntries());
                        statistics.getCacheStatistics(CacheConstants.SIMPLE_QUERY_CACHE).setCacheHitsNum(0);
                    }
                }
            }

            /**
             * Disables simple query cache.
             */
            public void disable() {
                if (!isDisabledSimple) {
                    isDisabledSimple = true;
                    if (simpleQCache != null) {
                        disabledMaxSimpleQueryCacheSize = simpleQCache.getMaxEntries();
                        simpleQCache = null;
                    } else {
                        disabledMaxSimpleQueryCacheSize = 0;
                    }
                }
            }

            /**
             * Enables simple query cache.
             */
            public void enable() {
                try {
                    if (isDisabledSimple) {
                        if (disabledMaxSimpleQueryCacheSize != 0) {
                            if (disabledMaxCacheSize != 0) {
                                simpleQCache = BaseCacheManager.getDODSCache(disabledMaxSimpleQueryCacheSize);
                            }
                        }
                        isDisabledSimple = false;
                    }
                    DODSLRUCache obj = (DODSLRUCache) statistics.getCacheStatistics(CacheConstants.SIMPLE_QUERY_CACHE);

                    if (obj != null) {
                        obj.clearStatistics();
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in enable simple");
                }
            }
        };
        cacheAdministration[CacheConstants.COMPLEX_QUERY_CACHE] = new CacheAdministration() {

            /**
             * Returns maximum size of complex query cache.
             *
             * @return Maximum size of complex query cache.
             */
            public int getMaxCacheSize() {
                if (complexQCache != null) {
                    return complexQCache.getMaxEntries();
                }
                return 0;
            }

            /**
             * Returns maximum complex query cache size. If the cache is unbounded
             * (the maximum size is negative), the current cache size is returned.
             *
             * @param real If this parameter is true, method returns real
             * maximum cache size, otherwise returns appropriate value for the
             * statistic.
             *
             * @return Maximum complex query cache size.
             */
            public int getMaxCacheSize(boolean real) {
                int size = getMaxCacheSize();

                if (size < 0) {
                    if (real) {
                        return -1;
                    } else {
                        return getCacheSize();
                    }
                }
                return size;
            }

            /**
             * Sets maximum size of complex query cache.
             *
             * @param maxSize Maximum size of complex query cache.
             */
            protected void setMaxCacheSize(int maxSize) throws CacheObjectException {
                try {
                    if (isDisabled) {
                        throw new CacheObjectException("Caching is disabled");
                    }
                    if (maxSize == 0) {
                        if (complexQCache != null) {
                            statistics.getCacheStatistics(CacheConstants.COMPLEX_QUERY_CACHE).clearStatistics();
                            complexQCache = null;
                        }
                        return;
                    }
                    if (complexQCache == null) {
                        if (cache != null) {
                            complexQCache = BaseCacheManager.getDODSCache(maxSize);
                        }
                    } else {
                        complexQCache.setMaxEntries(maxSize);
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in setMaxCacheSize - complex cache");
                }
            }

            /**
             * Returns size of currently used complex query cache.
             *
             * @return Size of currently used complex query cache.
             */
            public int getCacheSize() {
                if (complexQCache != null) {
                    return complexQCache.size();
                }
                return 0;
            }

            /**
             * Refreshes complex query cache.
             */
            public void refresh() {
                if (cache != null) {
                    if (complexQCache != null) {
                        complexQCache = BaseCacheManager.getDODSCache(complexQCache.getMaxEntries());
                        statistics.getCacheStatistics(CacheConstants.COMPLEX_QUERY_CACHE).setCacheHitsNum(0);
                    }
                }
            }

            /**
             * Disables complex query cache.
             */
            public void disable() {
                if (!isDisabledComplex) {
                    isDisabledComplex = true;
                    if (complexQCache != null) {
                        disabledMaxComplexQueryCacheSize = complexQCache.getMaxEntries();
                        complexQCache = null;
                    } else {
                        disabledMaxComplexQueryCacheSize = 0;
                    }
                }
            }

            /**
             * Enables complex query cache.
             */
            public void enable() {
                try {
                    if (isDisabledComplex) {
                        if (disabledMaxComplexQueryCacheSize != 0) {
                            if (disabledMaxCacheSize != 0) {
                                complexQCache = BaseCacheManager.getDODSCache(disabledMaxComplexQueryCacheSize);
                            }
                        }
                        isDisabledComplex = false;
                    }
                    DODSLRUCache obj = (DODSLRUCache) statistics.getCacheStatistics(CacheConstants.COMPLEX_QUERY_CACHE);

                    if (obj != null) {
                        obj.clearStatistics();
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in enable complex");
                }
            }
        };
        cacheAdministration[CacheConstants.DATA_CACHE] = new CacheAdministration() {

            /**
             * Returns maximum size of data (or DataStruct) object cache.
             *
             * @return Maximum data (or DataStruct) object cache size.
             */
            public int getMaxCacheSize() {
                return -1;
            }

            /**
             * Returns maximum size of data (or DataStruct) object cache. If the
             * cache is unbounded (the maximum size is negative), the current cache
             * size is returned.
             *
             * @param real If this parameter is true, method returns real
             * maximum cache size, otherwise returns appropriate value for the
             * statistic.
             *
             * @return Maximum data (or DataStruct) object cache size.
             */
            public int getMaxCacheSize(boolean real) {
                if (real) {
                    return getMaxCacheSize();
                } else {
                    return getCacheSize();
                }
            }

            /**
             * Returns size of currently used data (or DataStruct) object cache.
             *
             * @return Size of currently used data (or DataStruct) object cache.
             */
            public int getCacheSize() {
                if (cache != null) {
                    return cache.size();
                }
                return 0;
            }

            /**
             * Sets maximum size of data (or DataStruct) object cache.
             *
             * @param maxSize Maximum DO cache size.
             */
            protected void setMaxCacheSize(int maxSize) throws CacheObjectException {/* tanja 16.09.2003 kept for future changes
                 try {
                 if (isDisabled) {
                 throw new CacheObjectException("Caching is disabled");
                 }
                 if (maxSize == 0) {
                 cache = null;
                 if (simpleQCache != null) {
                 oldSimpleMaxCacheSize = getCacheAdministration(CacheConstants.SIMPLE_QUERY_CACHE).getMaxCacheSize();
                 simpleQCache = null;
                 }
                 else
                 oldSimpleMaxCacheSize = 0;
                 if (complexQCache != null) {
                 oldComplexMaxCacheSize = getCacheAdministration(CacheConstants.COMPLEX_QUERY_CACHE).getMaxCacheSize();
                 complexQCache = null;
                 }
                 else
                 oldComplexMaxCacheSize = 0;
                 statistics.clear();
                 return;
                 }
                 if (cache == null){
                 cache = new DODSHashMap();
                 if (oldSimpleMaxCacheSize != 0)
                 simpleQCache = new DODSLRUCache(oldSimpleMaxCacheSize);
                 if (oldComplexMaxCacheSize != 0)
                 complexQCache = new DODSLRUCache(oldComplexMaxCacheSize);
                 }
                 //                      else
                 //                        cache.setMaxEntries(maxSize);
                 }
                 catch (Exception ex){
                 System.out.println("Error in setMaxCacheSize - DO cache");
                 }
                 */}

            /**
             * Refreshes caches (data (or DataStruct) object, simple query and
             * complex query).
             */
            public void refresh() {
                if (cache != null) {
                    cache = new DODSHashMap();
                }
                if (simpleQCache != null) {
                    simpleQCache = BaseCacheManager.getDODSCache(simpleQCache.getMaxEntries());
                }
                if (complexQCache != null) {
                    complexQCache = BaseCacheManager.getDODSCache(complexQCache.getMaxEntries());
                }
                statistics.clear();
            }

            /**
             * Disables data (or DataStruct) object cache. When this cache is
             * disabled, so are and simple and complex query caches.
             */
            public void disable() {
                if (!isDisabled) {
                    isDisabled = true;
                    if (cache != null) {
                        cache = null;
                        if (simpleQCache != null) {
                            getCacheAdministration(CacheConstants.SIMPLE_QUERY_CACHE).disable();
                        }
                        if (complexQCache != null) {
                            cacheAdministration[CacheConstants.COMPLEX_QUERY_CACHE].disable();
                        }
                        statistics.clear();
                    }
                }
            }

            /**
             * Enables data (or DataStruct) object cache. When this cache is
             * enabled, so are and simple and complex query caches (if they existed
             * before DO cache had been disabled).
             */
            public void enable() {
                try {
                    if (isDisabled) {
                        cache = new DODSHashMap();
                        if (isDisabledSimple) {
                            getCacheAdministration(CacheConstants.SIMPLE_QUERY_CACHE).enable();
                        }
                        if (isDisabledComplex) {
                            getCacheAdministration(CacheConstants.COMPLEX_QUERY_CACHE).enable();
                        }
                        statistics.clear();
                        isDisabled = false;
                    }
                } catch (Exception  ex) {
                    System.out.println("Error in enable DO");
                }
            }
        };
    }

    /**
     * Returns cache (data or DataStruct) content.
     *
     * @return Cache content as <code>Map</code> of data (or DataStruct) objects.
     */
    public Map  getCacheContent() {
        return cache;
    }

    /**
     * Returns information if multi databases are supported.
     *
     * @return true if multi databases are used, otherwise false.
     */
    public boolean isMulti() {
        return multi;
    }

    /**
     * Checks wheather cache reconfiguration needs to be done.
     *
     * @return true if cache reconfiguration needs to be done, otherwise false.
     */
    public boolean toReconfigure() {
        return false;
    }

    /**
     * Adds DO (data object) to the cache.
     *
     * @param newDO Data object that will be added to the cache.
     *
     * @return Added data object.
     */
    public synchronized GenericDO addDO(GenericDO newDO) {
        if (cache == null) {
            return newDO;
        }
        try {
            String  handle = newDO.get_CacheHandle();
            GenericDO ret = (GenericDO) cache.put(handle, newDO);

            return newDO;
        } catch (Exception  e) {
            e.printStackTrace();
        }
        return null;
    }

    /**
     * Removes DO (data object) from the cache.
     *
     * @param DO Data object that will be removed from the cache.
     *
     * @return Removed data object, or <tt>null</tt> if there was no object
     * removed from the cache.
     */
    public synchronized GenericDO removeDO(GenericDO DO) {
        if (cache != null) {
            try {
                String  handle = DO.get_CacheHandle();

                return (GenericDO) cache.remove(handle);
            } catch (Exception  e) {}
        }
        return null;
    }

    /**
     * Removes DO (data object) from the cache.
     *
     * @param handle Cache handle of DO (data object) that will be removed from
     * the cache. The form of cache handle is:
     * "<database_name>.<String_presentation_of_oid>".
     *
     * @return Removed data object, or <tt>null</tt> if there was no object
     * removed from the cache.
     */
    public synchronized GenericDO removeDO(String  handle) {
        if (cache != null) {
            try {
                return (GenericDO) cache.remove(handle);
            } catch (Exception  e) {}
        }
        return null;
    }

    /**
     * Updates cached DO, or inserts DO in the cache if it didn't exist in the
     * cache.
     *
     * @param DO Data object that will be updated (or inserted if didn't
     * exist in the cache).
     *
     * @return Updated or inserted data object.
     */
    public GenericDO updateDO(GenericDO DO) {
        DO = addDO(DO);
        QueryCacheItem queryItem = null;
        HashSet  del;
        Iterator  iter = null;

        if (complexQCache != null) {
            del = new HashSet ();
            for (iter = complexQCache.values().iterator(); iter.hasNext();) {
                queryItem = (QueryCacheItem) iter.next();
                if (DO.get_OriginDatabase().equals(queryItem.get_OriginDatabase())) {
                    del.add(queryItem);
                }
            }
            iter = null;
            for (iter = del.iterator(); iter.hasNext();) {
                removeComplexQuery((QueryCacheItem) iter.next());
            }
        }
        if (simpleQCache != null) {
            del = new HashSet ();
            iter = null;
            for (iter = simpleQCache.values().iterator(); iter.hasNext();) {
                queryItem = (QueryCacheItem) iter.next();
                if (queryItem.isCompleteResult()) {
                    queryItem.update(DO);
                } else {
                    del.add(queryItem);
                }
            }
            iter = null;
            for (iter = del.iterator(); iter.hasNext();) {
                removeSimpleQuery((QueryCacheItem) iter.next());
            }
        }
        return DO;
    }

    /**
     * Deletes DO from the cache.
     *
     * @param DO Data object that will be deleted from the cache.
     *
     * @return Deleted data object, or <tt>null</tt> if there was no object
     * deleted from the cache.
     */
    public GenericDO deleteDO(GenericDO DO) {
        GenericDO oldDO = removeDO(DO);
        Iterator  iter = null;
        QueryCacheItem queryItem = null;

        if (simpleQCache != null) {
            for (iter = simpleQCache.values().iterator(); iter.hasNext();) {
                queryItem = (QueryCacheItem) iter.next();
                queryItem.delete(DO);
            }
        }
        iter = null;
        if (complexQCache != null) {
            for (iter = complexQCache.values().iterator(); iter.hasNext();) {
                queryItem = (QueryCacheItem) iter.next();
                queryItem.delete(DO);
            }
        }
        return oldDO;
    }

    /**
     * Returns data object whose String representation of OID is parameter handle.
     *
     * @param handle String representation of OID of object that is being
     * searched in the cache.
     *
     * @return Data object whose String representation of OID is handle.
     */
    public synchronized GenericDO getDOByHandle(String  handle) {
        if (cache == null) {
            return null;
        }
        return (GenericDO) cache.get(handle);
    }

    /**
     * Creates new QueryCacheItem instance.
     *
     * @param dbName Database name.
     * @return Created QueryCacheItem instance.
     */
    public QueryCacheItem newQueryCacheItemInstance(String  dbName) {
        return new QueryCacheItemImpl(dbName);
    }

    /**
     * Returns QueryCacheItem object for specified database and simple query,
     * if exists, otherwise null.
     *
     * @param dbName Database name.
     * @param query Query in form of String.
     * @return QueryCacheItem object.
     */
    public QueryCacheItem getSimpleQueryCacheItem(String  dbName, String  query) {
        if (simpleQCache != null) {
            return (QueryCacheItem) simpleQCache.get(dbName + "." + query);
        }
        return null;
    }

    /**
     * Returns QueryCacheItem object for specified database and complex query,
     * if exists, otherwise null.
     *
     * @param dbName Database name.
     * @param query Query in form of String.
     * @return QueryCacheItem object.
     */
    public QueryCacheItem getComplexQueryCacheItem(String  dbName, String  query) {
        if (complexQCache != null) {
            return (QueryCacheItem) complexQCache.get(dbName + "." + query);
        }
        return null;
    }

    /**
     * Adds simple query to simple query cache.
     *
     * @param queryItem Query that will be added to simple query cache.
     * @return Query added to simple query cache.
     */
    public synchronized QueryCacheItem addSimpleQuery(QueryCacheItem queryItem) {
        if (simpleQCache != null) {
            return (QueryCacheItem) simpleQCache.add(queryItem.get_OriginDatabase()
                    + "." + queryItem.getQueryId(),
                    queryItem);
        }
        return null;
    }

    /**
     * Removes simple query from simple query cache.
     *
     * @param queryItem Query that will be removed from simple query cache.
     * @return Query removed from simple query cache.
     */
    public synchronized QueryCacheItem removeSimpleQuery(QueryCacheItem queryItem) {
        if (simpleQCache != null) {
            return (QueryCacheItem) simpleQCache.remove(queryItem.get_OriginDatabase()
                    + "." + queryItem.getQueryId());
        }
        return null;
    }

    /**
     * Adds complex query to complex query cache.
     *
     * @param queryItem Query that will be added to complex query cache.
     * @return Query added to complex query cache.
     */
    public synchronized QueryCacheItem addComplexQuery(QueryCacheItem queryItem) {
        if (complexQCache != null) {
            return (QueryCacheItem) complexQCache.add(queryItem.get_OriginDatabase()
                    + "." + queryItem.getQueryId(),
                    queryItem);
        }
        return null;
    }

    /**
     * Removes complex query from complex query cache.
     *
     * @param queryItem Query that will be removed from complex query cache.
     * @return Query removed from complex query cache.
     */
    public synchronized QueryCacheItem removeComplexQuery(QueryCacheItem queryItem) {
        if (complexQCache != null) {
            return (QueryCacheItem) complexQCache.remove(queryItem.get_OriginDatabase()
                    + "." + queryItem.getQueryId());
        }
        return null;
    }

    /**
     * Returns query results from simple query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in simple query cache.
     * @return Query results retrieved from simple cache, or null, if there are
     * no results retrieved from simple query cache.
     */
    public QueryResult getSimpleQueryResults(String  dbName, String  query) {
        return getSimpleQueryResults(dbName, query, 0, 0, false);
    }

    /**
     * Returns query results from simple query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in simple query cache.
     * @param limit Specified number of results (database limit and read skip).
     * @param maxdb Number of rows retrieved from database (or cache).
     * @return Query results retrieved from simple cache, or null, if there are
     * no results retrieved from simple query cache.
     */
    public QueryResult getSimpleQueryResults(String  dbName, String  query, int limit, int maxdb) {
        return getSimpleQueryResults(dbName, query, limit, maxdb, false);
    }

    /**
     * Returns query results from simple query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in simple query cache.
     * @param limit Specified number of results (database limit and read skip).
     * @param maxdb Number of rows retrieved from database (or cache).
     * @param unique If true, only unique results are returned.
     * @return Query results retrieved from simple cache, or null, if there are
     * no results retrieved from simple query cache.
     */
    public QueryResult getSimpleQueryResults(String  dbName, String  query, int limit, int maxdb, boolean unique) {
        if (simpleQCache == null) {
            return null;
        }
        QueryResult result = null;
        String  queryHandle = dbName + "." + query;
        QueryCacheItem queryItem = (QueryCacheItem) simpleQCache.get(queryHandle);
        int i = 0;

        if (queryItem != null) {
            result = new QueryResult();
            result.database = queryItem.get_OriginDatabase();
            DOShell shell = null;
            String  handle = null;
            String  cachePrefix = queryItem.get_OriginDatabase() + ".";
            GenericDO cacheDO = null;
          
          synchronized(queryItem.getOIds()) {  
            Iterator  iter = queryItem.getOIds().iterator();

            if (unique) {
                HashSet  allResultOids = new HashSet ();
                int skippedNum = 0;

                while ((maxdb == 0 || i < maxdb) && (iter.hasNext())
                        && ((limit == 0 || result.DOs.size() < limit))) {
                    handle = (String ) iter.next();
                    if (allResultOids.contains(handle)) {
                        skippedNum++;
                    } else {
                        allResultOids.add(handle);
                        try {
                            cacheDO = (GenericDO) cache.get(cachePrefix + handle);
                            shell = new DOShell(handle);
                            if (cacheDO != null) {
                                shell.dataObject = cacheDO;
                            } else {
                                result.lazy.add(shell);
                            }
                            result.DOs.add(shell);
                        } catch (Exception  e) {}
                    }
                    i++;
                } // while
                result.skippedUnique = skippedNum;
            } // (unique)
 else {
                while ((maxdb == 0 || i < maxdb) && (iter.hasNext())
                        && ((limit == 0 || result.DOs.size() < limit))) {
                    handle = (String ) iter.next();
                    try {
                        cacheDO = (GenericDO) cache.get(cachePrefix + handle);
                        shell = new DOShell(handle);
                        if (cacheDO != null) {
                            shell.dataObject = cacheDO;
                        } else {
                            result.lazy.add(shell);
                        }
                        result.DOs.add(shell);
                    } catch (Exception  e) {}
                    i++;
                } // while
            } // (unique)
        } // (queryItem != null)
       } 
        if (result.DOs.size() < limit) {
            if ((maxdb == 0) || (i < maxdb)) {
                result = null;
            }
        }
        return result;
    }

    /**
     * Returns query results from complex query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in complex query cache.
     * @return Query results retrieved from complex cache, or null, if there are
     * no results retrieved from complex query cache.
     */
    public QueryResult getComplexQueryResults(String  dbName, String  query) {
        return getComplexQueryResults(dbName, query, 0, 0, false);
    }

    /**
     * Returns query results from complex query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in complex query cache.
     * @param limit Specified number of results (database limit and read skip).
     * @param maxdb Number of rows retrieved from database (or cache).
     * @return Query results retrieved from complex cache, or null, if there are
     * no results retrieved from complex query cache.
     */
    public QueryResult getComplexQueryResults(String  dbName, String  query, int limit, int maxdb) {
        return getComplexQueryResults(dbName, query, limit, maxdb, false);
    }

    /**
     * Returns query results from complex query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in complex query cache.
     * @param limit Specified number of results (database limit and read skip).
     * @param maxdb Number of rows retrieved from database (or cache).
     * @param unique If true, only unique results are returned.
     * @return Query results retrieved from complex cache, or null, if there are
     * no results retrieved from complex query cache.
     */
    public QueryResult getComplexQueryResults(String  dbName, String  query, int limit, int maxdb, boolean unique) {
        if (complexQCache == null) {
            return null;
        }
        QueryResult result = null;
        String  queryHandle = dbName + "." + query;
        QueryCacheItem queryItem = (QueryCacheItem) complexQCache.get(queryHandle);
        int i = 0;

        if (queryItem != null) {
            result = new QueryResult();
            result.database = queryItem.get_OriginDatabase();
            DOShell shell = null;
            String  handle = null;
            String  cachePrefix = queryItem.get_OriginDatabase() + ".";
            GenericDO cacheDO = null;
          
          synchronized(queryItem.getOIds()) {  
            Iterator  iter = queryItem.getOIds().iterator();

            if (unique) {
                HashSet  allResultOids = new HashSet ();
                int skippedNum = 0;

                while ((maxdb == 0 || i < maxdb) && (iter.hasNext())
                        && ((limit == 0 || result.DOs.size() < limit))) {
                    handle = (String ) iter.next();
                    if (allResultOids.contains(handle)) {
                        skippedNum++;
                    } else {
                        allResultOids.add(handle);
                        try {
                            cacheDO = (GenericDO) cache.get(cachePrefix + handle);
                            shell = new DOShell(handle);
                            if (cacheDO != null) {
                                shell.dataObject = cacheDO;
                            } else {
                                result.lazy.add(shell);
                            }
                            result.DOs.add(shell);
                        } catch (Exception  e) {}
                    }
                    i++;
                } // while
                result.skippedUnique = skippedNum;
            } // (unique)
 else {
                while ((maxdb == 0 || i < maxdb) && (iter.hasNext())
                        && ((limit == 0 || result.DOs.size() < limit))) {
                    handle = (String ) iter.next();
                    try {
                        cacheDO = (GenericDO) cache.get(cachePrefix + handle);
                        shell = new DOShell(handle);
                        if (cacheDO != null) {
                            shell.dataObject = cacheDO;
                        } else {
                            result.lazy.add(shell);
                        }
                        result.DOs.add(shell);
                    } catch (Exception  e) {}
                    i++;
                } // while
            } // (unique)
        } // (queryItem != null)
       } 
        if (result.DOs.size() < limit) {
            if ((maxdb == 0) || (i < maxdb)) {
                result = null;
            }
        }
        return result;
    }

    /**
     * Returns query results from simple or complex query cache.
     *
     * @param dbName Database name.
     * @param query Query for which are results searched in simple and complex
     * query cache.
     * @return Query results retrieved from simple or complex cache, or null,
     * if there are no results retrieved from simple and complex query cache.
     */
    public QueryResult getQueryResults(String  dbName, String  query) {
        QueryResult result = getSimpleQueryResults(dbName, query);

        if (result == null) {
            result = getComplexQueryResults(dbName, query);
        }
        return result;
    }

    /**
     * Inner class that implements TableStatistics. This is class for
     * administrating table and cache.
     */
    private class QueryCacheImplStatistics extends TableStatistics {

        /**
         * Constructor().
         */
        public QueryCacheImplStatistics() {
            try {
                this.reset();
                if (cache != null) {
                    getCacheStatistics(CacheConstants.DATA_CACHE).clearStatistics();
                }
                if (simpleQCache != null) {
                    getCacheStatistics(CacheConstants.SIMPLE_QUERY_CACHE).clearStatistics();
                }
                if (complexQCache != null) {
                    getCacheStatistics(CacheConstants.COMPLEX_QUERY_CACHE).clearStatistics();
                }
            } catch (Exception  ex) {}
        }

        /**
         * Resets parameters.
         */
        public void reset() {
            insertNum = 0;
            updateNum = 0;
            deleteNum = 0;
            lazyLoadingNum = 0;
            startTime = new Date ();
            stopTime = new Date ();
            queryNum = 0;
            queryByOIdNum = 0;
            averageQueryTime = 0;
            averageQueryByOIdTime = 0;
        }

        /**
         * Returns type of the statistics. In this case, this is
         * QUERY_CACHE_STATISTICS.
         *
         * @return Type of the statistics: QUERY_CACHE_STATISTICS.
         */
        public int getStatisticsType() {
            return QUERY_CACHE_STATISTICS;
        }

        /**
         * Clears statistics.
         */
        public void clear() {
            this.reset();
            if (cache != null) {
                getCacheStatistics(CacheConstants.DATA_CACHE).clearStatistics();
            }
            if (simpleQCache != null) {
                getCacheStatistics(CacheConstants.SIMPLE_QUERY_CACHE).clearStatistics();
            }
            if (complexQCache != null) {
                getCacheStatistics(CacheConstants.COMPLEX_QUERY_CACHE).clearStatistics();
            }
        }

        /**
         * Returns query statistics for DO cache, simple query or complex query
         * cache.
         *
         * @param type Indicator of cache for which query statistics is returned
         * for. Possible values:
         * org.enhydra.dods.cache.CacheConstants.DATA_CACHE (value 0) for query
         * statistics of DO cache
         * org.enhydra.dods.cache.CacheConstants.SIMPLE_QUERY_CACHE (value 1)
         * for query statistics of simple query cache
         * org.enhydra.dods.cache.CacheConstants.COMPLEX_QUERY_CACHE (value 2)
         * for query statistics of complex query cache
         *
         * @return Query statistics for specified cache.
         */
        public CacheStatistics getCacheStatistics(int type) {
            switch (type) {
            case CacheConstants.DATA_CACHE:
                return cache;

            case CacheConstants.SIMPLE_QUERY_CACHE:
                return simpleQCache;

            case CacheConstants.COMPLEX_QUERY_CACHE:
                return complexQCache;

            default:
                return null;
            }
        }
    }

    /**
     * Shows content of this class.
     * Can be used for debugging.
     */
    public String  toString() {
        StringBuffer  ret = new StringBuffer ();

        ret.append("\n TransactionCacheImpl: ");
        ret.append("\n cache: " + cache);
        ret.append("\n simpleQCache : " + simpleQCache);
        ret.append("\n complexQCache : " + complexQCache);
        ret.append("\n cacheReadOnly : " + tableConf.isReadOnly());
        ret.append("\n initialQueryCache : " + initialQueryCache);
        ret.append("\n fullCaching : " + isFull());
        return ret.toString();
    }
    /**
     *
     */
    public void removeEntries(Vector  vec) {
        //System.err.println(getClass()+".removeEntries(Vector)");
 if (cache.size() > vec.size()) {
            for (Enumeration  e = vec.elements(); e.hasMoreElements();) {
                String  cacheHandle = (String ) e.nextElement();
                removeDO(cacheHandle);
            }
        } else {
            Iterator  it=cache.values().iterator();
            Vector  vecToRemove = new Vector ();
            while (it.hasNext()) {
                GenericDO obj = (GenericDO)it.next();
                try {
                    if (vec.contains((obj.get_CacheHandle())))
                        vecToRemove.add(obj);
                } catch (DatabaseManagerException e) {}
            }
            for (Enumeration  e = vecToRemove.elements(); e.hasMoreElements();) {
                removeDO((GenericDO)e.nextElement());
            }
        }
    }

    /**
     *
     * @param tableClass -
     */
    public void removeEntries(Class  tableClass) {
        //System.err.println(getClass()+".removeEntries(Class)");
 Iterator  it=cache.values().iterator();
        Vector  vecToRemove = new Vector ();
        while (it.hasNext()) {
            GenericDO obj = (GenericDO)it.next();
            if (tableClass.equals(obj.getClass()))
                vecToRemove.add(obj);
        } // while
 for (Enumeration  e = vecToRemove.elements(); e.hasMoreElements();) {
            removeDO((GenericDO)e.nextElement());
        }
    }

    /**
     *
     */
    public void emptyEntries(Vector  vec, boolean incrementVersion) {
        //System.err.println(getClass()+".emptyEntries(Vector)");
 for (Enumeration  e = vec.elements(); e.hasMoreElements();) {
            String  cacheHandle = (String ) e.nextElement();
            GenericDO obj = (GenericDO) cache.get(cacheHandle);
            if (obj != null){
                obj.dumpData(incrementVersion);
            }
        } // for
    }
    
    /**
     * Dumps data structs for all instances of tableClass in transaction
     * cache.
     *
     * @param tableClass - Class object whose instances will be emptied
     */
    public void emptyEntries(Class  tableClass) {
        //System.err.println(getClass()+".emptyEntries(Class)");
 Collection  c = cache.values();
        Iterator  it=c.iterator();
        while (it.hasNext()) {
            GenericDO obj = (GenericDO)it.next();
            if (tableClass.equals(obj.getClass())) {
//                System.err.println("emptiing: "+obj);
                obj.dumpData(false);
            }
        } // while
    }
    
    
    /**
     * @return
     */
    public int getInitialCacheFetchSize() {
        return initialCacheFetchSize;
    }

    /**
     * @return
     */
    public int getInitialDSCacheSize() {
        return initialDSCacheSize;
    }

    /**
     * @param i
     */
    public void setInitialCacheFetchSize(int i) {
        initialCacheFetchSize = i;
    }

    /**
     * @param i
     */
    public void setInitialDSCacheSize(int i) {
        initialDSCacheSize = i;
    }


    
}
