
/*
 * Enhydra Java Application Server Project
 *
 * The contents of this file are subject to the Enhydra Public License
 * Version 1.1 (the "License"); you may not use this file except in
 * compliance with the License. You may obtain a copy of the License on
 * the Enhydra web site ( http://www.enhydra.org/ ).
 *
 * Software distributed under the License is distributed on an "AS IS"
 * basis, WITHOUT WARRANTY OF ANY KIND, either express or implied. See
 * the License for the specific terms governing rights and limitations
 * under the License.
 *
 * The Initial Developer of the Enhydra Application Server is Lutris
 * Technologies, Inc. The Enhydra Application Server and portions created
 * by Lutris Technologies, Inc. are Copyright Lutris Technologies, Inc.
 * All Rights Reserved.
 *
 */
package org.enhydra.dods.cache.lru;

import org.enhydra.dods.cache.base.DODSCache;

/**
 * DODSLRUCache class implements LRU cache (for storing data objects (or
 * DataStruct objects), or simple queries, or complex queries), and provides
 * query statistics about the cache (query number, cache hits number, their
 * get/set/increment methods, percents of used cache, cache hits,...).
 *
 * @author    Tanja Jovanovic
 * @author    Nenad Vico
 * @author    Zorica Suvajdzin
 * @version   2.0  15.06.2003.
 */
public class DODSLRUCache extends LRUCache implements DODSCache {

    /**
     * Total number of times the cache was accessed.
     */
    protected int cacheAccessNum = 0;

    /**
     * Number of queries performed on the cache successfully.
     */
    protected int cacheHitsNum = 0;

    /**
     * Constructor (int).
     *
     * @param maxSize Maximal number of objects in DODSLRUCache.
     */
    DODSLRUCache(int maxSize) {
        super(maxSize);
        clearStatistics();
    }

    /**
     * Returns total number of times the cache was accessed.
     *
     * @return total number of times the cache was accessed.
     */
    public int getCacheAccessNum() {
        return cacheAccessNum;
    }   

    /**
     * Sets total number of times the cache was accessed.
     *
     * @param num Total number of times the cache was accessed.
     */
    public void setCacheAccessNum(int num) {
        this.cacheAccessNum = num;
    }   

    /**
     * Increases total number of times the cache was accessed.
     */
    public void incrementCacheAccessNum(int num) {
        cacheAccessNum += num;
    } 

    /**
     * Returns number of queries performed on the cache successfully.
     *
     * @return Number of queries performed on the cache successfully.
     */
    public int getCacheHitsNum() {
        return cacheHitsNum;
    }

    /**
     * Sets number of queries performed on the cache successfully.
     *
     * @param cacheHitsNum Number of queries performed on the cache successfully.
     */
    public void setCacheHitsNum(int cacheHitsNum) {
        this.cacheHitsNum = cacheHitsNum;
    }

    /**
     * Increases number of queries performed on the cache successfully for one.
     */
    public void incrementCacheHitsNum(int num) {
        cacheHitsNum += num;
    }

    /**
     * Returns how much cache is currently used. This value is given in percents.
     * If cache is unbounded, method returns 100%.
     *
     * @return Percents - how much cache is currently used.
     */
    public double getUsedPercents() {
        int maxCacheSize = this.maxEntries;

        if (maxCacheSize == 0) {
            return 0;
        }
        if (maxCacheSize < 0) {
            return 100;
        }
        int temp = size() * 10000;
        double res = temp / maxCacheSize;

        return res / 100;
    }

    /**
     * Returns how much queries performed on the cache were successful.
     * This value is given in percents.
     *
     * @return Percents - how much queries performed on the cache were
     * successful.
     */
    public double getCacheHitsPercents() {
        if (cacheAccessNum == 0) {
            return 0;
        }
        int temp = cacheHitsNum * 10000;
        double res = temp / cacheAccessNum;

        return res / 100;
    }

    /**
     * Clears statistics.
     */
    public void clearStatistics() {
        this.cacheAccessNum = 0;
        this.cacheHitsNum = 0;
    }
    
    public boolean isNeedToSynchronize() {
        return true;
    }
    
}
