package org.enhydra.shark.repositorypersistence;

import org.enhydra.shark.SharkEngineManager;
import org.enhydra.shark.api.RepositoryTransaction;
import org.enhydra.shark.api.RootException;
import org.enhydra.shark.api.TransactionException;
import org.enhydra.shark.api.internal.repositorypersistence.EntityNotFoundRepositoryException;
import org.enhydra.shark.api.internal.repositorypersistence.RepositoryException;
import org.enhydra.shark.api.internal.repositorypersistence.RepositoryPersistenceManager;
import org.enhydra.shark.api.internal.working.CallbackUtilities;
import org.enhydra.shark.plusworkflow.databasemanager.HibernateConfigBundle;
import org.enhydra.shark.plusworkflow.databasemanager.HibernateDatabaseManager;
import org.enhydra.shark.plusworkflow.databasemanager.HibernateTransaction;
import org.enhydra.shark.plusworkflow.databasemanager.oid.CounterAllocatorException;
import org.enhydra.shark.plusworkflow.databasemanager.oid.ObjectIdAllocatorException;
import org.enhydra.shark.repositorypersistence.data.NextXPDLVersion;
import org.enhydra.shark.repositorypersistence.data.XPDL;
import org.enhydra.shark.repositorypersistence.data.XPDLData;
import org.enhydra.shark.repositorypersistence.data.XPDLHistory;
import org.enhydra.shark.repositorypersistence.data.XPDLHistoryData;
import org.enhydra.shark.repositorypersistence.data.XPDLReference;
import org.hibernate.Session;
import org.hibernate.SessionFactory;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;

@SuppressWarnings( "unused" )
public class HibernateRepositoryPersistenceManager
    implements RepositoryPersistenceManager
{

    private static final String INITIAL_VERSION = "1";

    private SessionFactory sessionFactory;

    @Override
    public void configure( CallbackUtilities cus )
        throws RootException
    {
        HibernateDatabaseManager hibernateDatabaseManager = getHibernateDatabaseManager();

        try
        {
            HibernateConfigBundle configBundle = HibernateConfigBundle.builder()
                .postgresConfigPath( "shark/repository/psql/hibernate.cfg.xml" )
                .mssqlConfigPath( "shark/repository/mssql/hibernate.cfg.xml" )
                .oracleConfigPath( "shark/repository/oracle/hibernate.cfg.xml" )
                .h2ConfigPath( "shark/repository/h2/hibernate.cfg.xml" )
                .build();

            sessionFactory = hibernateDatabaseManager.buildSessionFactory( configBundle );
            cus.info( "Created session factory successfully" );
        }
        catch ( Exception e )
        {
            throw new RepositoryException( "Failed to configure hibernate session factory", e );
        }
    }

    @Override
    public void uploadXPDL( RepositoryTransaction t,
                            String xpdlId,
                            byte[] xpdl, byte[] serializedPkg,
                            long xpdlClassVer )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            String xpdlVer = updateNextVersion( session, xpdlId );

            XPDL xpdlEntity = createXPDL();
            xpdlEntity.setXpdlId( xpdlId );
            xpdlEntity.setXpdlVersion( xpdlVer );
            xpdlEntity.setXpdlClassVersion( xpdlClassVer );
            xpdlEntity.setXpdlUploadTime( new Timestamp( System.currentTimeMillis() ) );

            XPDLData xpdlDataEntity = createXPDLData();
            xpdlDataEntity.setXpdlContent( xpdl );
            xpdlDataEntity.setXpdlClassContent( serializedPkg );
            xpdlDataEntity.setXpdl( xpdlEntity );

            xpdlEntity.getXpdlDatas().add( xpdlDataEntity );

            session.save( xpdlEntity );
            session.save( xpdlDataEntity );
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to upload XPDL by id %s", xpdlId
            ), e );
        }
    }

    private String updateNextVersion(Session session, String xpdlId) throws RepositoryException {
        String curVersion = INITIAL_VERSION;

        try
        {
            NextXPDLVersion nextXPDLVersion = session
                .createQuery( "from NextXPDLVersion where xpdlId = :xpdlId", NextXPDLVersion.class )
                .setParameter( "xpdlId", xpdlId )
                .uniqueResult();

            if ( nextXPDLVersion == null )
            {
                nextXPDLVersion = createNextXPDLVersion();
                nextXPDLVersion.setXpdlId( xpdlId );
                nextXPDLVersion.setNextVersion( INITIAL_VERSION );
            }
            else
            {
                curVersion = nextXPDLVersion.getNextVersion();
            }

            int nver = Integer.parseInt( nextXPDLVersion.getNextVersion() ) + 1;
            String nextVersion = String.valueOf( nver );
            nextXPDLVersion.setNextVersion( nextVersion );

            session.save( nextXPDLVersion );
        }
        catch ( Exception ex )
        {
            throw new RepositoryException( "Internal problem while updating Next XPDL Version", ex );
        }
        return curVersion;
    }

    @Override
    public void updateXPDL( RepositoryTransaction t,
                            String xpdlId, String xpdlVersion,
                            byte[] xpdl, byte[] serializedPkg,
                            long xpdlClassVer )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            XPDL xpdlEntity = getXPDLEntity( session, xpdlId, xpdlVersion );
            xpdlEntity.setXpdlClassVersion( xpdlClassVer );

            XPDLData xpdlDataEntity = xpdlEntity.getXpdlDatas().get( 0 );
            xpdlDataEntity.setXpdlContent( xpdl );
            xpdlDataEntity.setXpdlClassContent( serializedPkg );

            session.save( xpdlEntity );
            session.save( xpdlDataEntity );
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to upload XPDL by id %s", xpdlId
            ), e );
        }
    }

    @Override
    public void deleteXPDL( RepositoryTransaction t, String xpdlId, String xpdlVersion )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            XPDL xpdlEntity = getXPDLEntity( session, xpdlId, xpdlVersion );

            session.delete( xpdlEntity );
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to delete XPDL by id %s", xpdlId
            ), e );
        }
    }

    @Override
    public void moveToHistory( RepositoryTransaction t, String xpdlId, String xpdlVersion )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            XPDL xpdlEntity = getXPDLEntity( session, xpdlId, xpdlVersion );
            XPDLData xpdlDataEntity = xpdlEntity.getXpdlDatas().get( 0 );

            XPDLHistory xpdlHistoryEntity = createXPDLHistory();
            xpdlHistoryEntity.setXpdlId( xpdlEntity.getXpdlId() );
            xpdlHistoryEntity.setXpdlVersion( xpdlEntity.getXpdlVersion() );
            xpdlHistoryEntity.setXpdlUploadTime( xpdlEntity.getXpdlUploadTime() );
            xpdlHistoryEntity.setXpdlHistoryUploadTime( new Timestamp( System.currentTimeMillis() ) );

            XPDLHistoryData xpdlHistoryDataEntity = createXPDLHistoryData();
            xpdlHistoryDataEntity.setXpdlContent( xpdlDataEntity.getXpdlContent() );
            xpdlHistoryDataEntity.setXpdlHistory( xpdlHistoryEntity );

            xpdlHistoryEntity.getXpdlHistoryDatas().add( xpdlHistoryDataEntity );

            session.delete( xpdlEntity );
            session.save( xpdlHistoryEntity );
            session.save( xpdlHistoryDataEntity );
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to move to history XPDL by id %s", xpdlId
            ), e );
        }
    }

    @Override
    public void deleteFromHistory( RepositoryTransaction t, String xpdlId, String xpdlVersion )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            XPDLHistory xpdlHistoryEntity = session
                .createQuery( "from XPDLHistory where xpdlId = :xpdlId and xpdlVersion = :xpdlVersion", XPDLHistory.class )
                .setParameter( "xpdlId", xpdlId )
                .setParameter( "xpdlVersion", xpdlVersion )
                .uniqueResult();

            session.delete( xpdlHistoryEntity );
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to delete from history XPDL by id %s", xpdlId
            ), e );
        }
    }

    @Override
    public void clearRepository( RepositoryTransaction t )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            session
                .createQuery( "delete from XPDL" )
                .executeUpdate();
        }
        catch ( Exception e )
        {
            throw new RepositoryException( "Failed to clear repository", e );
        }
    }

    @Override
    public String getCurrentVersion( RepositoryTransaction t, String xpdlId )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            return getLastVersionXPDL( session, xpdlId )
                .getXpdlVersion();
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get current XPDL version for id %s", xpdlId
            ), e );
        }
    }

    @Override
    public String getNextVersion( RepositoryTransaction t, String xpdlId )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            NextXPDLVersion nextXPDLVersionEntity = session
                .createQuery( "from NextXPDLVersion where xpdlId = :xpdlId", NextXPDLVersion.class )
                .setParameter( "xpdlId", xpdlId )
                .uniqueResult();

            return nextXPDLVersionEntity != null
                ? nextXPDLVersionEntity.getNextVersion()
                : INITIAL_VERSION;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get next XPDL version for id %s", xpdlId
            ), e );
        }
    }

    @Override
    public long getSerializedXPDLObjectVersion( RepositoryTransaction t, String xpdlId, String xpdlVersion )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            return getXPDLEntity( session, xpdlId, xpdlVersion )
                .getXpdlClassVersion();
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get serialized XPDL object version by id %s and version %s",
                xpdlId, xpdlVersion
            ), e );
        }
    }

    @Override
    public byte[] getXPDL( RepositoryTransaction t, String xpdlId )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            return getLastVersionXPDL( session, xpdlId )
                .getXpdlDatas().get( 0 )
                .getXpdlContent();
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get XPDL by id %s", xpdlId
            ), e );
        }
    }

    @Override
    public byte[] getSerializedXPDLObject( RepositoryTransaction t, String xpdlId )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            return getLastVersionXPDL( session, xpdlId )
                .getXpdlDatas().get( 0 )
                .getXpdlClassContent();
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get serialized XPDL by id %s", xpdlId
            ), e );
        }
    }

    @Override
    public byte[] getXPDL( RepositoryTransaction t, String xpdlId, String xpdlVersion )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            // optymalizowane, by zwracać bezpośrednio byte[],
            // by session nie trzymało encji XpdlData w L1 hibernate cache, gdy nie jest to potrzebne do edycji
            // a będzie często odpytywane przez PWFL
            return session
                .createQuery(
                    "select xpdlDataEntity.xpdlContent "
                        + "from XPDL xpdlEntity "
                        + "join XPDLData xpdlDataEntity "
                        + "on xpdlDataEntity.xpdl = xpdlEntity "
                        + "where xpdlEntity.xpdlId = :xpdlId and xpdlEntity.xpdlVersion = :xpdlVersion",
                    byte[].class
                )
                .setParameter( "xpdlId", xpdlId )
                .setParameter( "xpdlVersion", xpdlVersion )
                .uniqueResultOptional()
                .orElseThrow( () -> new EntityNotFoundRepositoryException( String.format(
                    "Could not find xpdl data by id %s and version %s", xpdlId, xpdlVersion
                ) ) );
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get XPDL by id %s and version %s", xpdlId, xpdlVersion
            ), e );
        }
    }

    @Override
    public byte[] getSerializedXPDLObject( RepositoryTransaction t, String xpdlId, String xpdlVersion )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            // optymalizowane, by zwracać bezpośrednio byte[],
            // by session nie trzymało encji XpdlData w L1 hibernate cache, gdy nie jest to potrzebne do edycji
            // a będzie często odpytywane przez PWFL
            return session
                .createQuery(
                    "select xpdlDataEntity.xpdlClassContent "
                        + "from XPDL xpdlEntity "
                        + "join XPDLData xpdlDataEntity "
                        + "on xpdlDataEntity.xpdl = xpdlEntity "
                        + "where xpdlEntity.xpdlId = :xpdlId and xpdlEntity.xpdlVersion = :xpdlVersion",
                    byte[].class
                )
                .setParameter( "xpdlId", xpdlId )
                .setParameter( "xpdlVersion", xpdlVersion )
                .uniqueResultOptional()
                .orElseThrow( () -> new EntityNotFoundRepositoryException( String.format(
                    "Could not find serialized xpdl data by id %s and version %s", xpdlId, xpdlVersion
                ) ) );
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get serialized XPDL by id %s and version %s", xpdlId, xpdlVersion
            ), e );
        }
    }

    @Override
    public List<String> getXPDLVersions( RepositoryTransaction t, String xpdlId )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            List<String> xpdlVersions = session
                .createQuery( "select xpdlVersion from XPDL where xpdlId = :xpdlId", String.class )
                .setParameter( "xpdlId", xpdlId )
                .list();

            return new ArrayList<>( xpdlVersions );
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get XPDL versions by id %s", xpdlId
            ), e );
        }
    }

    @Override
    public boolean doesXPDLExist( RepositoryTransaction t, String xpdlId )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            return findLastVersionXPDL( session, xpdlId )
                .isPresent();
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to check XPDL existence by id %s", xpdlId
            ), e );
        }
    }

    @Override
    public boolean doesXPDLExist( RepositoryTransaction t, String xpdlId, String xpdlVersion )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            return session
                .createQuery(
                    "select 1 from XPDL "
                        + "where xpdlId = :xpdlId "
                        + "and xpdlVersion = :xpdlVersion",
                    Integer.class
                )
                .setParameter( "xpdlId", xpdlId )
                .setParameter( "xpdlVersion", xpdlVersion )
                .uniqueResultOptional()
                .isPresent();
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to check XPDL existence by id %s and version %s", xpdlId, xpdlVersion
            ), e );
        }
    }

    @Override
    public List<String> getExistingXPDLIds( RepositoryTransaction t )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            List<String> xpdlIds = session
                .createQuery( "select distinct xpdlId from XPDL", String.class )
                .list();

            return new ArrayList<>( xpdlIds );
        }
        catch ( Exception e )
        {
            throw new RepositoryException( "Failed to get existing XPDL ids", e );
        }
    }

    @Override
    public void addXPDLReference( RepositoryTransaction t,
                                  String referredXPDLId, String referringXPDLId, String referringXPDLVersion,
                                  int referredXPDLNumber )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            XPDL referringXpdlEntity = getXPDLEntity( session, referringXPDLId, referringXPDLVersion );
            Optional<XPDLReference> xpdlReferenceEntity = session
                .createQuery( "from XPDLReference "
                                  + "where referredXpdlId = :referredXpdlId and referringXpdl = :referringXpdl", XPDLReference.class )
                .setParameter( "referredXpdlId", referredXPDLId )
                .setParameter( "referringXpdl", referringXpdlEntity )
                .uniqueResultOptional();

            if ( !xpdlReferenceEntity.isPresent() )
            {
                XPDLReference newXpdlReferenceEntity = createXPDLReference();
                newXpdlReferenceEntity.setReferredXpdlId( referredXPDLId );
                newXpdlReferenceEntity.setReferringXpdl( referringXpdlEntity );
                newXpdlReferenceEntity.setReferredXpdlNumber( referredXPDLNumber );

                referringXpdlEntity.getXpdlReferences().add( newXpdlReferenceEntity );

                session.save( referringXpdlEntity );
                session.save( newXpdlReferenceEntity );
            }
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to add XPDL reference by referred XPDL id %s and referring XPDL id %s",
                referredXPDLId, referringXPDLId
            ), e );
        }
    }

    @Override
    public List<String> getReferringXPDLIds( RepositoryTransaction t, String referredXPDLId )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            List<String> referringXpdlIds = session
                .createQuery( "select distinct xpdlReference.referringXpdl.xpdlId "
                                  + "from XPDLReference xpdlReference "
                                  + "join xpdlReference.referringXpdl "
                                  + "where xpdlReference.referredXpdlId = :referredXpdlId",
                              String.class )
                .setParameter( "referredXpdlId", referredXPDLId )
                .list();

            return new ArrayList<>( referringXpdlIds );
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get referring XPDL ids by referred XPDL id %s", referredXPDLId
            ), e );
        }
    }

    @Override
    public List<String> getReferringXPDLVersions( RepositoryTransaction t, String referredXPDLId, String referringXPDLId )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            List<String> referringXpdlVersions = session
                .createQuery( "select xpdlReference.referringXpdl.xpdlVersion "
                                  + "from XPDLReference xpdlReference "
                                  + "join xpdlReference.referringXpdl "
                                  + "where xpdlReference.referredXpdlId = :referredXpdlId "
                                  + "and xpdlReference.referringXpdl.xpdlId = :referringXpdlId",
                              String.class )
                .setParameter( "referredXpdlId", referredXPDLId )
                .setParameter( "referringXpdlId", referringXPDLId )
                .list();

            return new ArrayList<>( referringXpdlVersions );
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get referring XPDL versions by referred XPDL id %s and referring XPDL id %s",
                referredXPDLId, referringXPDLId
            ), e );
        }
    }

    @Override
    public List<String> getReferredXPDLIds( RepositoryTransaction t, String referringXPDLId, String referringXPDLVersion )
        throws RepositoryException
    {
        try
        {
            Session session = getExtended( t ).getSession();

            XPDL referringXpdl = getXPDLEntity( session, referringXPDLId, referringXPDLVersion );

            List<XPDLReference> xpdlReferenceEntities = session
                .createQuery( "from XPDLReference where referringXpdl = :referringXpdl", XPDLReference.class )
                .setParameter( "referringXpdl", referringXpdl )
                .list();

            Map<Integer, String> xpdlNumberToXpdlIdMap = new HashMap<>();
            for ( XPDLReference xpdlReferenceEntity : xpdlReferenceEntities )
            {
                xpdlNumberToXpdlIdMap.put( xpdlReferenceEntity.getReferredXpdlNumber(),
                                           xpdlReferenceEntity.getReferredXpdlId() );
            }

            List<Integer> xpdlNumbers = new ArrayList<>( xpdlNumberToXpdlIdMap.keySet() );
            Collections.sort( xpdlNumbers );

            List<String> referredXpdlIds = new ArrayList<>();
            for ( Integer xpdlNumber : xpdlNumbers )
            {
                referredXpdlIds.add( xpdlNumberToXpdlIdMap.get( xpdlNumber ) );
            }

            return referredXpdlIds;
        }
        catch ( RepositoryException e )
        {
            throw e;
        }
        catch ( Exception e )
        {
            throw new RepositoryException( String.format(
                "Failed to get referred XPDL ids by referring XPDL id %s and referring XPDL version %s",
                referringXPDLId, referringXPDLVersion
            ), e );
        }
    }

    private Optional<XPDL> findXPDLEntity( Session session, String xpdlId, String xpdlVersion )
    {
        return session
            .createQuery( "from XPDL where xpdlId = :xpdlId and xpdlVersion = :xpdlVersion", XPDL.class )
            .setParameter( "xpdlId", xpdlId )
            .setParameter( "xpdlVersion", xpdlVersion )
            .uniqueResultOptional();
    }

    private XPDL getXPDLEntity( Session session, String xpdlId, String xpdlVersion )
        throws RepositoryException
    {
        return findXPDLEntity( session, xpdlId, xpdlVersion )
            .orElseThrow( () -> new EntityNotFoundRepositoryException( String.format(
                "Could not find xpdl by id %s and version %s", xpdlId, xpdlVersion
            ) ) );
    }

    private Optional<XPDL> findLastVersionXPDL( Session session, String xpdlId )
    {
        return session
            .createQuery( "from XPDL where xpdlId = :xpdlId order by cast(xpdlVersion as long) desc", XPDL.class )
            .setMaxResults( 1 )
            .setParameter( "xpdlId", xpdlId )
            .uniqueResultOptional();
    }

    private XPDL getLastVersionXPDL( Session session, String xpdlId )
        throws RepositoryException
    {
        return findLastVersionXPDL( session, xpdlId )
            .orElseThrow( () -> new EntityNotFoundRepositoryException( String.format(
                "Could not find last xpdl by id %s", xpdlId
            ) ) );
    }

    @Override
    public RepositoryTransaction createTransaction()
        throws TransactionException
    {
        HibernateTransaction hibernateTransaction = getHibernateDatabaseManager()
            .createHibernateTransaction( sessionFactory );

        return new HibernateRepositoryTransaction( hibernateTransaction );
    }

    private HibernateRepositoryTransaction getExtended( RepositoryTransaction transaction )
    {
        return ( HibernateRepositoryTransaction ) transaction;
    }

    private BigDecimal allocateObjectId()
        throws RepositoryException
    {
        try
        {
            return getHibernateDatabaseManager().allocateObjectId();
        }
        catch ( ObjectIdAllocatorException e )
        {
            throw new RepositoryException( "Failed to allocate object id", e );
        }
    }

    private XPDL createXPDL()
        throws ObjectIdAllocatorException
    {
        XPDL xpdl = new XPDL();
        xpdl.setObjectId( getHibernateDatabaseManager().allocateObjectId() );
        xpdl.setObjectVersion( 0 );
        xpdl.setXpdlDatas( new ArrayList<>( 1 ) );

        return xpdl;
    }

    private XPDLData createXPDLData()
        throws ObjectIdAllocatorException, CounterAllocatorException
    {
        XPDLData xpdlData = new XPDLData();
        xpdlData.setObjectId( getHibernateDatabaseManager().allocateObjectId() );
        xpdlData.setObjectVersion( 0 );
        xpdlData.setCnt( getHibernateDatabaseManager().allocateNextXpdlDataCnt() );

        return xpdlData;
    }

    private NextXPDLVersion createNextXPDLVersion()
        throws ObjectIdAllocatorException
    {
        NextXPDLVersion nextXPDLVersion = new NextXPDLVersion();
        nextXPDLVersion.setObjectId( getHibernateDatabaseManager().allocateObjectId() );
        nextXPDLVersion.setObjectVersion( 0 );

        return nextXPDLVersion;
    }

    private XPDLHistory createXPDLHistory()
        throws ObjectIdAllocatorException
    {
        XPDLHistory xpdlHistory = new XPDLHistory();
        xpdlHistory.setObjectId( getHibernateDatabaseManager().allocateObjectId() );
        xpdlHistory.setObjectVersion( 0 );
        xpdlHistory.setXpdlClassVersion( 0L );
        xpdlHistory.setXpdlHistoryDatas( new ArrayList<>( 1 ) );

        return xpdlHistory;
    }

    private XPDLHistoryData createXPDLHistoryData()
        throws ObjectIdAllocatorException, CounterAllocatorException
    {
        XPDLHistoryData xpdlHistoryData = new XPDLHistoryData();
        xpdlHistoryData.setObjectId( getHibernateDatabaseManager().allocateObjectId() );
        xpdlHistoryData.setObjectVersion( 0 );
        xpdlHistoryData.setCnt( getHibernateDatabaseManager().allocateNextXpdlHistoryDataCnt() );
        xpdlHistoryData.setXpdlClassContent( new byte[]{ 0 } );

        return xpdlHistoryData;
    }

    private XPDLReference createXPDLReference()
        throws ObjectIdAllocatorException
    {
        XPDLReference xpdlReference = new XPDLReference();
        xpdlReference.setObjectId( getHibernateDatabaseManager().allocateObjectId() );
        xpdlReference.setObjectVersion( 0 );

        return xpdlReference;
    }

    private HibernateDatabaseManager getHibernateDatabaseManager()
    {
        return SharkEngineManager
            .getInstance()
            .getHibernateDatabaseManager();
    }

}
