package org.enhydra.shark.plusworkflow.util;

import lombok.AccessLevel;
import lombok.AllArgsConstructor;
import lombok.NonNull;

import java.util.ArrayDeque;
import java.util.Deque;
import java.util.Optional;

public class ThreadLocalContext<T>
{

    private final ThreadLocal<Deque<T>> localContextStack = ThreadLocal.withInitial( () -> new ArrayDeque<>( 4 ) );

    private final String name;

    private final boolean stackable;

    public ThreadLocalContext( @NonNull String name, boolean stackable )
    {
        this.name = name;
        this.stackable = stackable;
    }

    public T current()
    {
        return currentOptional()
            .orElseThrow( () -> new ContextNotActiveException( String.format(
                "Context [%s] is not active.", name
            ) ) );
    }

    public Optional<T> currentOptional()
    {
        T topContext = localContextStack.get().peek();
        return Optional.ofNullable( topContext );
    }

    public boolean isActive()
    {
        return !localContextStack.get().isEmpty();
    }

    public ExecuteStep withNewContext( @NonNull T context )
    {
        if ( !stackable && isActive() )
        {
            throw new ContextNotStackableException( String.format(
                "Context [%s] is not stackable", name
            ) );
        }
        return new ExecuteStep( context );
    }

    public void pushContext( @NonNull T context )
    {
        if ( !stackable && isActive() )
        {
            throw new ContextNotStackableException( String.format(
                "Context [%s] is not stackable", name
            ) );
        }

        Deque<T> contextStack = localContextStack.get();
        contextStack.push( context );
    }

    public void popContext()
    {
        Deque<T> contextStack = localContextStack.get();
        contextStack.pop();
    }

    @AllArgsConstructor( access = AccessLevel.PRIVATE )
    public class ExecuteStep
    {

        private T context;

        public void execute( @NonNull WithContextRunnable<T> runnable )
        {
            Deque<T> contextStack = localContextStack.get();
            try
            {
                contextStack.push( context );
                runnable.execute( context );
            }
            finally
            {
                contextStack.pop();
            }
        }

        public <R> R execute( @NonNull WithContextFunction<T, R> supplier )
        {
            Deque<T> contextStack = localContextStack.get();
            try
            {
                contextStack.push( context );
                return supplier.execute( context );
            }
            finally
            {
                contextStack.pop();
            }
        }

    }

    @FunctionalInterface
    public interface WithContextRunnable<T>
    {
        void execute( T context );
    }

    @FunctionalInterface
    public interface WithContextFunction<T, R>
    {
        R execute( T context );
    }

}
