
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext;

import org.apache.xalan.extensions.ExpressionContext;

/**
 * Ejen version utility (static methods).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:ver="org.ejen.ext.Version"</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="xml" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;ejen-version&gt;
 *        &lt;xsl:value-of select="ver:{@link #toString(ExpressionContext) toString}()"/&gt;
 *      &lt;/ejen-version&gt;
 *      &lt;ejen-high-version&gt;
 *        &lt;xsl:value-of select="ver:{@link #high(ExpressionContext) high}()"/&gt;
 *      &lt;/ejen-high-version&gt;
 *      &lt;ejen-low-version&gt;
 *        &lt;xsl:value-of select="ver:{@link #low(ExpressionContext) low}()"/&gt;
 *      &lt;/ejen-low-version&gt;
 *      &lt;ejen-version-status&gt;
 *        &lt;xsl:value-of select="ver:{@link #status(ExpressionContext) status}()"/&gt;
 *      &lt;/ejen-version-status&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class Version {

    /**
     * Protected constructor (prevents instanciation).
     */
    protected Version() {}

    /** High version number: "1". */
    public static final int HIGH_VERSION = 1;

    /** Low version number: "0". */
    public static final int LOW_VERSION = 0;

    /** Version status: ".0". */
    public static final String  VERSION_STATUS = ".0";

    /**
     * Returns the Ejen high version number.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="high-version" select="ver:high()"/&gt;
     * </pre></td></tr></table>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return the high version number.
     */
    public static int high(ExpressionContext context) {
        return HIGH_VERSION;
    }

    /**
     * Returns the Ejen low version number.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="low-version" select="ver:low()"/&gt;
     * </pre></td></tr></table>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return the low version number.
     */
    public static int low(ExpressionContext context) {
        return LOW_VERSION;
    }

    /**
     * Returns the Ejen low version number.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="version-status" select="ver:status()"/&gt;
     * </pre></td></tr></table>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return the version status String.
     */
    public static String  status(ExpressionContext context) {
        return VERSION_STATUS;
    }

    /**
     * Returns the Ejen high version number.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="version" select="ver:version()"/&gt;
     * </pre></td></tr></table>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return the version number in the form of
     *         "&lt;HIGH_VERSION&gt;.&lt;LOW_VERSION&gt;&lt;VERSION_STATUS&gt;".
     */
    public static String  toString(ExpressionContext context) {
        return "" + HIGH_VERSION + "." + LOW_VERSION + VERSION_STATUS;
    }
}
