
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext;

import org.ejen.util.XSLUtil;
import org.apache.xalan.extensions.ExpressionContext;
import java.text.SimpleDateFormat ;
import java.util.Date ;

/**
 * String operations utilities (static methods).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:stu="org.ejen.ext.StringUtil"</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="text" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      Date: &lt;xsl:value-of select="stu:{@link #dateFormat(ExpressionContext,String) dateFormat}('yyyy/MM/dd')"/&gt;
 *      &lt;xsl:if test="stu:{@link #equals(ExpressionContext,String,String) equals}('abc','abc')"&gt;
 *        ...
 *      &lt;/xsl:if&gt;
 *      &lt;xsl:if test="stu:{@link #equalsIgnoreCase(ExpressionContext,String,String) equalsIgnoreCase}('abc','AbC')"&gt;
 *        ...
 *      &lt;/xsl:if&gt;
 *      Name (lower case): &lt;xsl:value-of select="stu:{@link #toLowerCase(ExpressionContext,String) toLowerCase}('aTtiLa')"/&gt;
 *      Name (upper case): &lt;xsl:value-of select="stu:{@link #toUpperCase(ExpressionContext,String) toUpperCase}('aTtiLa')"/&gt;
 *      Name (...): &lt;xsl:value-of select="stu:{@link #toULowerCase(ExpressionContext,String) toULowerCase}('aTtiLa')"/&gt;
 *      &lt;xsl:value-of select="stu:{@link #indent(ExpressionContext,int,int) indent}(2,3)"/&gt;
 *      Error: &lt;xsl:value-of select="stu:{@link #replace(ExpressionContext,String,String,String) replace}('in of memory','in','out')"/&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class StringUtil {
    public static final String  LINE_SEPARATOR = System.getProperty("line.separator",
            "\n");

    /**
     * Protected constructor (prevents instanciation).
     */
    protected StringUtil() {}

    /**
     * Formats and returns current date.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="stu:dateFormat('yyyy/MM/dd')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * See {@link java.text.SimpleDateFormat}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> time pattern to be used to format the date.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param format time pattern.
     * @return the formated current date.
     */
    public static String  dateFormat(ExpressionContext context, String  format) {
        return new SimpleDateFormat (XSLUtil.evaluate(context, format)).format(new Date ());
    }

    /**
     * Compares the String s1 to the String s2.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:if test="stu:equals('abc','abc')"&gt;
     *    ...
     *  &lt;/xsl:if&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> the first String.
     *   <dd><b>[Mandatory/AVT]</b> the second string.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param s1 the first String.
     * @param s2 the second String.
     * @return true if s1 equals s2, false otherwise.
     */
    public static boolean equals(ExpressionContext context, String  s1, String  s2) {
        return XSLUtil.evaluate(context, s1).equals(XSLUtil.evaluate(context, s2));
    }

    /**
     * Compares the String s1 to the String s2, ignoring case considerations.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:if test="stu:equals('abc','ABc')"&gt;
     *    ...
     *  &lt;/xsl:if&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> the first String.
     *   <dd><b>[Mandatory/AVT]</b> the second string.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param s1 <b>[AVT]</b> - the first String.
     * @param s2 <b>[AVT]</b> - the second String.
     * @return true if s1 equals s2, ignoring case, false otherwise.
     */
    public static boolean equalsIgnoreCase(ExpressionContext context, String  s1, String  s2) {
        return XSLUtil.evaluate(context, s1).equalsIgnoreCase(XSLUtil.evaluate(context,
                s2));
    }

    /**
     * Converts all of the characters in the String s to lower case using the rules of
     * the default locale.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="stu:toLowerCase('aBc')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> the String to be converted to lower case.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param s the String to be converted.
     * @return the String s converted to lower case.
     */
    public static String  toLowerCase(ExpressionContext context, String  s) {
        return XSLUtil.evaluate(context, s).toLowerCase();
    }

    /**
     * Converts all of the characters in the String s to upper case using the rules of
     * the default locale.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="stu:toUpperCase('aBc')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> the String to be converted to upper case.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param s the String to be converted.
     * @return the String s converted to upper case.
     */
    public static String  toUpperCase(ExpressionContext context, String  s) {
        return XSLUtil.evaluate(context, s).toUpperCase();
    }

    /**
     * Converts the first character in the String s to uppercase and the remaining
     * characters to lower case using the rules of the default locale.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="stu:toULowerCase('aBc')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> the String to be converted.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param s the String to be converted.
     * @return the String s converted.
     */
    public static String  toULowerCase(ExpressionContext context, String  s) {
        s = XSLUtil.evaluate(context, s);
        if (s.length() == 0) {
            return s;
        }
        if (s.length() == 1) {
            return s.toUpperCase();
        } else {
            return s.substring(0, 1).toUpperCase()
                    + s.substring(1).toLowerCase();
        }
    }

    /**
     * Returns a String starting with 'newLines' new lines followed by 'tabs' tabs.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="stu:indent(1,2)"/&gt;
     * </pre></td></tr></table>
     * <p>
     * This call will return <code>"\n\t\t"</code>.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory]</b> number of new lines.
     *   <dd><b>[Mandatory]</b> number of tabs.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param newLines number of new lines.
     * @param tabs number of tabs.
     * @return the indent String.
     */
    public static String  indent(ExpressionContext context, int newLines, int tabs) {
        StringBuffer  sb = new StringBuffer ();

        for (int i = 0; i < newLines; i++) {
            sb.append('\n');
        }
        for (int i = 0; i < tabs; i++) {
            sb.append('\t');
        }
        return sb.toString();
    }

    /**
     * Returns a new string resulting from replacing all occurrences of s2 in s1 with s3.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="stu:replace('ab-ab-ab','ab','cde')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * This call will return <code>"cde-cde-cde"</code>.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory]</b> the first String.
     *   <dd><b>[Mandatory]</b> the second string (will be unescaped).
     *   <dd><b>[Mandatory]</b> the third string (will be unescaped).
     * </dl></dd>
     * <p>
     * "unescaped" means: common escape sequences ("\t", "\n", "\r", "\f")
     * converted to their respective values.
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param s1 the first String.
     * @param s2 the second String (will be unescaped).
     * @param s3 the third String (will be unescaped).
     * @return true if s1 equals s2, ignoring case, false otherwise.
     */
    public static String  replace(ExpressionContext context, String  s1, String  s2, String  s3) {
        s2 = unescape(s2);
        s3 = unescape(s3);
        int iS2inS1 = s1.indexOf(s2);

        if (iS2inS1 == -1) {
            return s1;
        }
        StringBuffer  sb = new StringBuffer (s1.substring(0, iS2inS1)).append(s3);
        int oldIS2inS1 = iS2inS1;
        int s1Length = s1.length();
        int s2Length = s2.length();

        while (true) {
            iS2inS1 = s1.indexOf(s2, iS2inS1 + s2Length);
            if (iS2inS1 == -1) {
                if (oldIS2inS1 + s2Length < s1Length) {
                    sb.append(s1.substring(oldIS2inS1 + s2Length));
                }
                break;
            }
            sb.append(s1.substring(oldIS2inS1 + s2Length, iS2inS1)).append(s3);
            oldIS2inS1 = iS2inS1;
        }
        return sb.toString();
    }

    /**
     * Returns a String with common escape sequences ("\t", "\n", "\r", "\f")
     * converted to their respective values. The rule is: "\\" becomes "\";
     * "\*" (with * in "tnrf") becomes the character '\*'; "\*" (with * not in
     * "tnrf") becomes "*"; if a "\" is the last character of the String, it
     * is removed (except if the String ends with "\\").
     * <p>
     * @param s String to unescape.
     * @return the String s unescaped.
     */
    protected static String  unescape(String  s) {
        int ibs = s.indexOf('\\');

        if (ibs == -1) {
            return s;
        }
        StringBuffer  sb = new StringBuffer ();
        
        int lastChar = s.length() - 1;
        int from = 0;

        while (true) {
            if (from < ibs) {
                sb.append(s.substring(from, ibs));
            }
            if (ibs >= lastChar) {
                break;
            }
            char c = s.charAt(ibs + 1);

            switch (c) {
            case 't':
                sb.append('\t');
                break;

            case 'n':
                sb.append('\n');
                break;

            case 'r':
                sb.append('\r');
                break;

            case 'f':
                sb.append('\f');
                break;

            default:
                sb.append(c);
                break; // case '\\' and others.
            }
            from = ibs + 2;
            ibs = s.indexOf('\\', from);
            if (ibs == -1) {
                if (from < s.length()) {
                    sb.append(s.substring(from, s.length()));
                }
                break;
            }
        }           
        return sb.toString();
    }
}
