
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext;

import org.ejen.util.XSLUtil;
import java.util.Hashtable ;
import java.util.Enumeration ;
import org.w3c.dom.Document ;
import org.apache.xml.utils.WrappedRuntimeException;
import org.apache.xalan.extensions.XSLProcessorContext;
import org.apache.xalan.extensions.ExpressionContext;
import org.apache.xalan.templates.ElemExtensionCall;
import org.apache.xpath.objects.XObject;
import org.apache.xpath.objects.XString;
import org.apache.xpath.objects.XNodeSet;
import org.apache.xpath.objects.XRTreeFrag;
import org.apache.xpath.NodeSet;

/**
 * Gobal variables utility (instanciable).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:gvs="org.ejen.ext.GlobalVariables"
 *                  extension-element-prefixes="gvs ..."
 *                  exclude-result-prefixes="gvs ..."</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="xml" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;xls:variable name="vars1" select="gvs:{@link #GlobalVariables() new}()"/&gt;
 *      &lt;xls:variable name="vars2" select="gvs:{@link #GlobalVariables(GlobalVariables) new}($vars1)"/&gt;
 *      &lt;gvs:{@link #clear(XSLProcessorContext,ElemExtensionCall) clear}/&gt;
 *      &lt;vars1-count&gt;
 *        &lt;xsl:value-of select="gvs:{@link #size(ExpressionContext) size}([$vars1])"/&gt;
 *      &lt;/vars1-count&gt;
 *      &lt;gvs:{@link #remove(XSLProcessorContext,ElemExtensionCall) remove} name="myVar" [instance="$vars1"]/&gt;
 *      &lt;gvs:{@link #put(XSLProcessorContext,ElemExtensionCall) put} name="primary-keys"
 *              [select="column[@primary-key='true']"]
 *              [instance="$gvs2"]/&gt;
 *      <i>or</i>
 *      &lt;gvs:{@link #put(XSLProcessorContext,ElemExtensionCall) put} name="myVar"
 *              [attribute="Hello {people[$i]/@name} !"]
 *              [instance="$gvs2"]/&gt;
 *      &lt;xsl:value-of select="gvs:{@link #get(ExpressionContext,String) get}([$vars2,] 'myVar')"/&gt;
 *      &lt;xsl:for-each select="gvs:{@link #elements(ExpressionContext) elements}([$vars2])"&gt;
 *        ...
 *      &lt;/xsl:for-each&gt;
 *      &lt;xsl:for-each select="gvs:{@link #keys(ExpressionContext) keys}([$vars2])"&gt;
 *        ...
 *      &lt;/xsl:for-each&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class GlobalVariables {

    /** Table of variables */
    private Hashtable  _vars = null;

    /**
     * Constructs a new GlobalVariables object. If this constructor is
     * not explicitly used, the Xalan extension mechanism constructs a
     * default instance.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="gvs1" select="gvs:new()"/&gt;
     * </pre></td></tr></table>
     */ 
    public GlobalVariables() {
        _vars = new Hashtable ();
    }

    /**
     * Constructs a new GlobalVariables object with the same mappings as the
     * given GlobalVariables.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="gvs2" select="gvs:new($gvs1)"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory]</b> GlobalVariables instance.
     * </dl></dd>
     * <p>
     * @param gvs a GlobalVariables object to be copied in this GlobalVariables.
     */ 
    public GlobalVariables(GlobalVariables gvs) {
        _vars = new Hashtable (gvs._vars);
    }

    /**
     * Throws away all variables.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;gvs:clear [instance="$gvs2"]/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>instance <b>[Optional]</b> GlobalVariables instance (if not set,
     *       default instance is used).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @throws org.apache.xml.utils.WrappedRuntimeException if the instance attribute
     *         is not an instance of GlobalVariables.
     */
    public void clear(XSLProcessorContext context, ElemExtensionCall elem) {
        Object  o = XSLUtil.getOAttribute(context, elem, "instance",
                GlobalVariables.class, false, false);

        if (o == null) {
            _vars.clear();
        } else {
            ((GlobalVariables) o)._vars.clear();
        }
    }

    /**
     * Returns variables count.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="gvs:size([$gvs2])"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Optional]</b> GlobalVariables instance (if not set, default instance
     *          is used).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return variables count in this <code>GlobalVariables</code>.
     */
    public int size(ExpressionContext context) {
        return _vars.size();
    }

    /**
     * Removes a variable.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;gvs:remove name="myVar"[ instance="$gvs2"]/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>name <b>[Mandatory/AVT]</b> - name of the variable to be removed.
     *   <dd>instance <b>[Optional]</b> - GlobalVariables instance.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @throws org.apache.xml.utils.WrappedRuntimeException
     *         if the 'name' attribute is missing or if the instance attribute
     *         is not an instance of GlobalVariables.
     */
    public void remove(XSLProcessorContext context, ElemExtensionCall elem) {
        String  name = XSLUtil.getAttribute(context, elem, "name", true);
        Object  o = XSLUtil.getOAttribute(context, elem, "instance",
                GlobalVariables.class, false, false);

        if (o == null) {
            _vars.remove(name);
        } else {
            ((GlobalVariables) o)._vars.remove(name);
        }
    }

    /**
     * Creates (or updates the value of) a variable.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;gvs:put name="myVar"
     *          [select="column[@primary-key='true']"]
     *          [instance="$gvs2"]/&gt;
     * <i>or</i>
     *  &lt;gvs:put name="myVar"
     *          [attribute="Hello {people[$i]/@name} !"]
     *          [instance="$gvs2"]/&gt;
     * </pre></td></tr></table>
     * <p>
     * <i>'select' and 'attribute' cannot be used at the same time. If none of those
     * attributes is used, the default value "(no value)" is bound to the variable
     * name.</i>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>name <b>[Mandatory/AVT]</b> - name of the variable to be removed.
     *   <dd>select <b>[Optional/AVT]</b> - XPath expression (nodes set, XSL variable...).
     *   <dd>attribute <b>[Optional/AVT]</b> - a String.
     *   <dd>instance <b>[Optional]</b> - GlobalVariables instance (if not set, default instance
     *                                    is used).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @throws org.apache.xml.utils.WrappedRuntimeException with a XSL Exception
     *         if the 'name' attribute.
     */
    public void put(XSLProcessorContext context, ElemExtensionCall elem) {
        XObject xo = null;
        String  name = XSLUtil.getAttribute(context, elem, "name", true);
        String  select = XSLUtil.getAttribute(context, elem, "select", false);
        Object  o = XSLUtil.getOAttribute(context, elem, "instance",
                GlobalVariables.class, false, false);

        if (select != null) {
            xo = XSLUtil.evaluate(context, elem, select);
            if (xo == null) {
                throw new WrappedRuntimeException(new IllegalArgumentException ("bad \"select\" attribute ?!"));
            }
        } else {
            String  s = XSLUtil.getAttribute(context, elem, "attribute", false);

            if (s != null) {
                xo = new XString(s);
            }
        }
        if (xo == null) {
            xo = new XString("(no value)");
        }
        if (o == null) {
            _vars.put(name, xo);
        } else {
            ((GlobalVariables) o)._vars.put(name, xo);
        }
    }

    /**
     * Gets the value of a variable.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="gvs:get([$gvs2, ]'myVar')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Optional]</b> GlobalVariables instance (if not set, default instance
     *          is used).
     *   <dd><b>[Mandatory/AVT]</b> name of the variable.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param name the name of the variable.
     * @return the value of the variable or <code>null</code> if there was
     *         no such variable.
     */
    public XObject get(ExpressionContext context, String  name) {
        return (XObject) (_vars.get(XSLUtil.evaluate(context, name)));
    }

    /**
     * Gets the value of a variable.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;gvs:get name="myVar"[ instance="$gvs2"]/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>name <b>[Mandatory/AVT]</b> - name of the variable to be removed.
     *   <dd>instance <b>[Optional]</b> - GlobalVariables instance (if not set, default instance
     *                                    is used).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @return the value of the variable or <code>null</code> if there was
     *         no such variable.
     * @throws org.apache.xml.utils.WrappedRuntimeException with a XSL Exception
     *         if the 'name' attribute is missing.
     */
    public XObject get(XSLProcessorContext context, ElemExtensionCall elem) {
        Object  o = XSLUtil.getOAttribute(context, elem, "instance",
                GlobalVariables.class, false, false);
        String  name = XSLUtil.getAttribute(context, elem, "name", true);

        if (o == null) {
            return (XObject) (_vars.get(name));
        } else {
            return (XObject) (((GlobalVariables) o)._vars.get(name));
        }
    }

    /**
     * Returns a NodeSet that contains all variable values in this GlobalVariables.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:for-each select="gvs:elements([$gvs2])"&gt;
     *    ...
     *  &lt;/xsl:for-each&gt;
     * </pre></td></tr></table>
     * <p>
     * <i>Returned NodeSet is rather an union of the values than a collection
     * of the values: if the value of a variable is already a XNodeSet, returned
     * NodeSet will contain all Nodes of this XNodeSet (and not a Node that
     * contains the XNodeSet); same with XRTreeFrag values; if the
     * value is a XString (or XBoolean, XNumber...), then returned NodeSet
     * will contain a CDATA Node with a String representation of the value.</i>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Optional]</b> GlobalVariables instance (if not set, default instance
     *          is used).
     *   <dd><b>[Mandatory/AVT]</b> name of the variable.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return a NodeSet (may be empty) that contains all variable values in this
     *         GlobalVariables.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM Exception...
     */
    public NodeSet elements(ExpressionContext context) {
        Document  doc = XSLUtil.getContextDocument(context);

        try {
            NodeSet ns = new NodeSet();

            for (Enumeration  e = _vars.elements(); e.hasMoreElements();) {
                XObject xo = (XObject) (e.nextElement());

                if (xo instanceof XNodeSet) {
                    ns.addNodes(xo.nodeset());
                } else if (xo instanceof XRTreeFrag) {
                    ns.addNodes(((XRTreeFrag) xo).convertToNodeset());
                } else {
                    ns.addElement(doc.createCDATASection(xo.toString()));
                }
            }
            return ns;
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Returns a NodeSet that contains all variable names in this GlobalVariables.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:for-each select="gvs:keys([$gvs2])"&gt;
     *    ...
     *  &lt;/xsl:for-each&gt;
     * </pre></td></tr></table>
     * <p>
     * <i>Returned NodeSet is a collection of CDATA Nodes, each of them containing
     * a variable name.</i>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Optional]</b> GlobalVariables instance (if not set, default instance
     *          is used).
     *   <dd><b>[Mandatory/AVT]</b> name of the variable.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return a NodeSet (may be empty) that contains all variable names in this
     *         GlobalVariables.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM Exception...
     */
    public NodeSet keys(ExpressionContext context) {
        Document  doc = XSLUtil.getContextDocument(context);

        try {
            NodeSet ns = new NodeSet();

            for (Enumeration  e = _vars.keys(); e.hasMoreElements();) {
                ns.addElement(doc.createCDATASection((String ) (e.nextElement())));
            }
            return ns;
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }
}
