
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen;

import java.util.Properties ;
import java.io.File ;
import java.io.OutputStream ;
import java.io.FileOutputStream ;
import javax.xml.transform.Transformer ;
import javax.xml.transform.dom.DOMSource ;
import javax.xml.transform.stream.StreamResult ;
import javax.xml.transform.OutputKeys ;
import org.apache.xalan.processor.TransformerFactoryImpl;
import org.apache.xml.serializer.OutputPropertiesFactory;

/**
 * Save node class.
 * <p>
 * A Save node saves the current in memory DOM tree to file.
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (ant build file)</th></tr>
 * <tr><td class="usage"><pre><code>
 *  &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 *
 *  &lt;project name="generate" default="build"&gt;
 *
 *    &lt;taskdef name="ejen" classname="org.ejen.EjenTask"/&gt;
 *
 *    &lt;target name="build"&gt;
 *      &lt;{@link org.ejen.EjenTask ejen} ...&gt;
 *        ...
 *        <b>&lt;save {@link #setFile(String) file}="saved.xml"
 *             [{@link #setEncoding(String) encoding}="iso-8859-1"]
 *             [{@link #setIndent(String) indent}="(yes|no)"]
 *             [{@link #setAmount(String) amount}="2"]
 *        /&gt;</b>
 *        ...
 *      &lt;/ejen&gt;
 *    &lt;/target&gt;
 *
 *  &lt;/project&gt;
 * </code></pre></td></tr></table>
 * <p>
 * <b>Windows users</b>: an understandable (but stupid) problem with Windows platforms comes
 * with carriage return conversion on saving. If the DOM tree to be saved contains a Node
 * with the <code>"\r\n"</code> (<code>0D0A</code>) String (alone or not), the saved file
 * will contain the <code>"\r\r\n"</code> (<code>0D0D0A</code>) String instead. 
 * <p>
 * <b>Parent nodes</b>:
 * <ul>
 *   <li>{@link org.ejen.EjenTask ejen}
 * </ul>
 * @author F. Wolff
 * @version 1.0
 */
public class EjenSaveNode extends EjenChildNode {
    protected String  _file = null;
    protected String  _encoding = "iso-8859-1";
    protected String  _indent = "yes";
    protected String  _amount = "2";

    /**
     * Returns the name of this EjenSaveNode (always "save").
     * @return the name of this EjenSaveNode.
     */
    public String  nodeName() {
        return "save";
    }

    /**
     * Returns all non null attributes of this EjenSaveNode.
     * @return non null attributes of this EjenSaveNode.
     */
    public Properties  getAttributes() {
        Properties  attrs = super.getAttributes();

        if (_file != null) {
            attrs.setProperty("file", _file);
        }
        if (_encoding != null) {
            attrs.setProperty("encoding", _encoding);
        }
        if (_indent != null) {
            attrs.setProperty("indent", _indent);
        }
        if (_amount != null) {
            attrs.setProperty("amount", _amount);
        }
        return attrs;
    }

    /**
     * <b>[mandatory/AVT]</b> - sets the file attribute.
     * @param file name of the XML file to created.
     */
    public void setFile(String  file) {
        _file = file;
    }

    /**
     * <b>[optional/AVT]</b> - sets the encoding attribute.
     * @param encoding encoding String (default is "iso-8859-1").
     */
    public void setEncoding(String  encoding) {
        _encoding = encoding;
    }

    /**
     * <b>[optional/AVT]</b> - sets the indent attribute.
     * @param indent "yes" or "no" (default is "yes").
     */
    public void setIndent(String  indent) {
        _indent = indent;
    }

    /**
     * <b>[optional/AVT]</b> - sets the (indent) amount attribute.
     * @param indent positive integer (default is "2").
     */
    public void setAmount(String  amount) {
        _amount = amount;
    }

    /**
     * Checks this EjenSaveNode for mandatory attributes.
     * @throws org.ejen.EjenException if file attribute is not set.
     */
    public void check() {
        super.check();
        if (_file == null) {
            throw new EjenException(this, "No 'file' attribute");
        }
    }
    
    /**
     * Executes this EjenSaveNode.
     * @throws org.ejen.EjenException if something goes wrong...
     */
    public void process() {
        super.process();
        TransformerFactoryImpl tfi = null;
        DOMSource  src = null;

        try {
            tfi = (TransformerFactoryImpl) (getFromGlobalContext(CTX_TRANSFORMER_FACTORY_IMPL));
            src = (DOMSource ) (getFromGlobalContext(CTX_DOM_SOURCE));
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }
        if (tfi == null) {
            throw new EjenException(this,
                    "no '" + CTX_TRANSFORMER_FACTORY_IMPL
                    + "' in global context");
        }
        if (src == null) {
            throw new EjenException(this,
                    "no '" + CTX_DOM_SOURCE + "' in global context");
        }
        OutputStream  outputs = null;

        try {
            File  f = new File (evaluateAVT(_file));
            File  pf = f.getParentFile();

            if (pf != null) {
                pf.mkdirs();
            }
            outputs = new FileOutputStream (f.getPath());
            Transformer  serializer = tfi.newTransformer();

            serializer.setOutputProperty(OutputKeys.INDENT, evaluateAVT(_indent));
            serializer.setOutputProperty(OutputKeys.METHOD, "xml");
            serializer.setOutputProperty(OutputKeys.ENCODING,
                    evaluateAVT(_encoding));
            serializer.setOutputProperty( OutputPropertiesFactory.S_KEY_INDENT_AMOUNT,
                                          evaluateAVT(_amount));
            serializer.transform(src, new StreamResult (outputs));
        } catch (Exception  e) {
            throw new EjenException(this, _file, e);
        }
        finally {
            if (outputs != null) {
                try {
                    outputs.close();
                } catch (Exception  e) {}
                finally {
                    outputs = null;
                }
            }
        }
    }
}
