
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen;

import java.util.Properties ;
import javax.xml.transform.dom.DOMSource ;
import org.apache.xpath.XPathAPI;
import org.apache.xalan.transformer.TransformerImpl;

/**
 * Param node class.
 * <p>
 * A param node allows to pass a parameter to a stylesheet (filter, template,
 * import or include).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (ant build file)</th></tr>
 * <tr><td class="usage"><pre><code>
 *  &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 *
 *  &lt;project name="generate" default="build"&gt;
 *
 *    &lt;taskdef name="ejen" classname="org.ejen.EjenTask"/&gt;
 *
 *    &lt;target name="build"&gt;
 *      &lt;{@link org.ejen.EjenTask ejen} ...&gt;
 *        ...
 *        &lt;{@link org.ejen.EjenFilterNode filter} ...&gt;
 *          ...
 *          <b>&lt;param {@link #setName(String) name}="param1"
 *                 <i>either</i>
 *                 {@link #setSelect(String) select}="/ejen/entity-bean"
 *                 <i>or</i>
 *                 {@link #setLiteral(String) literal}="value"
 *          /&gt;</b>
 *          ...
 *        &lt;/filter&gt;
 *        ...
 *      &lt;/ejen&gt;
 *    &lt;/target&gt;
 *
 *  &lt;/project&gt;
 * </code></pre></td></tr></table>
 * <p>
 * <b>Parent nodes</b>:
 * <ul>
 *   <li>{@link org.ejen.EjenFilterNode filter}
 *   <li>{@link org.ejen.EjenTemplateNode template}
 *   <li>{@link org.ejen.EjenImportNode import}
 *   <li>{@link org.ejen.EjenIncludeNode include}
 * </ul>
 * @author F. Wolff
 * @version 1.0
 */
public class EjenParamNode extends EjenChildNode {
    protected String  _name = null;
    protected String  _select = null;
    protected String  _literal = null;

    /**
     * Returns the name of this EjenParamNode (always "param").
     * @return the name of this EjenParamNode.
     */
    public String  nodeName() {
        return "param";
    }

    /**
     * Returns all non null attributes of this EjenParamNode.
     * @return non null attributes of this EjenParamNode.
     */
    public Properties  getAttributes() {
        Properties  attrs = super.getAttributes();

        if (_name != null) {
            attrs.setProperty("name", _name);
        }
        if (_select != null) {
            attrs.setProperty("select", _select);
        }
        if (_literal != null) {
            attrs.setProperty("literal", _literal);
        }
        return attrs;
    }

    /**
     * <b>[mandatory/AVT]</b> - sets the name attribute.
     * @param name name of the parameter to be passed.
     */
    public void setName(String  name) {
        _name = name;
    }

    /**
     * <b>[mandatory if literal is not set/AVT]</b> - sets the select attribute.
     * @param select an XPATH select expression that gives the value of the parameter
     *        to be passed, relative to the current in memory DOM tree.
     */
    public void setSelect(String  select) {
        _select = select;
    }

    /**
     * <b>[mandatory if select is not set]</b> - sets the literal attribute.
     * @param literal a String that will be passed as the parameter value
     *        without interpretation.
     */
    public void setLiteral(String  literal) {
        _literal = literal;
    }

    /**
     * Checks this EjenParamNode for mandatory attributes.
     * @throws org.ejen.EjenException if name attribute is not set or
     *         if both select and literal are set or not set together.
     */
    public void check() {
        super.check();
        if (_name == null) {
            throw new EjenException(this, "No 'name' attribute");
        }
        if (_select == null && _literal == null) {
            throw new EjenException(this,
                    "Neither 'select' nor 'literal' attribute");
        }
        if (_select != null && _literal != null) {
            throw new EjenException(this,
                    "Cannot choose between 'select' and 'literal' attributes");
        }
    }
    
    /**
     * Executes this EjenParamNode.
     * @throws org.ejen.EjenException if something goes wrong...
     */
    public void process() {
        super.process();
        try {
            TransformerImpl ti = (TransformerImpl) (getFromContext(CTX_TRANSFORMER_IMPL));

            if (ti == null) {
                throw new EjenException(this,
                        "no '" + CTX_TRANSFORMER_IMPL + "' in context");
            }
            String  name = evaluateAVT(_name);

            if (_select != null) {
                DOMSource  src = (DOMSource ) (getFromGlobalContext(CTX_DOM_SOURCE));

                if (src == null) {
                    throw new EjenException(this,
                            "no '" + CTX_DOM_SOURCE + "' in global context");
                }
                ti.setParameter(name,
                        XPathAPI.selectNodeList(src.getNode(),
                        evaluateAVT(ti, _select)));
            } else {
                ti.setParameter(name, _literal);
            }
        } catch (EjenException e) {
            throw e;
        } catch (Exception  e) {
            throw new EjenException(this, "bad param", e);
        }
    }
}
