
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.util;

import org.ejen.EjenConstants;
import java.io.StringReader ;
import java.lang.reflect.Constructor ;
import javax.xml.transform.TransformerException ;
import javax.xml.transform.TransformerFactory ;
import javax.xml.transform.ErrorListener ;
import javax.xml.transform.stream.StreamSource ;
import org.w3c.dom.Node ;
import org.w3c.dom.NodeList ;
import org.w3c.dom.NamedNodeMap ;
import org.w3c.dom.Document ;
import org.w3c.dom.traversal.NodeIterator;
import org.apache.xml.utils.PrefixResolverDefault;
import org.apache.xml.utils.WrappedRuntimeException;
import org.apache.xalan.extensions.XSLProcessorContext;
import org.apache.xalan.extensions.ExpressionContext;
import org.apache.xalan.templates.ElemExtensionCall;
import org.apache.xalan.templates.AVT;
import org.apache.xalan.processor.StylesheetHandler;
import org.apache.xalan.processor.TransformerFactoryImpl;
import org.apache.xalan.transformer.TransformerImpl;
import org.apache.xpath.NodeSet;
import org.apache.xpath.XPathContext;
import org.apache.xpath.XPathContext.XPathExpressionContext;
import org.apache.xpath.XPath;
import org.apache.xpath.objects.XObject;

/**
 * XSL utility (static methods used in java code).
 * @author F. Wolff
 * @version 1.0
 */
public class XSLUtil {

    /**
     * Prevents instanciation.
     */
    protected XSLUtil() {}

    /**
     * Returns a new TransformerImpl using the
     * {@link org.ejen.EjenConstants#DEFAULT_XSL_DATA}
     * String as source.
     * @param tfi the TransformerFactoryImpl to use to build the TransformerImpl.
     * @return the new TransformerImpl.
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static TransformerImpl getDefaultTransformer(TransformerFactoryImpl tfi) {
        try {
            return (TransformerImpl) (tfi.newTransformer(new StreamSource (new StringReader (EjenConstants.DEFAULT_XSL_DATA))));
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Returns an interpreted value (AVT) of a Node attribute whose name is 'name'.
     * @param context the XSLProcessorContext to be used.
     * @param elem the ElemExtensionCall to be used.
     * @param name name of the attribute.
     * @param throwsIfNull if true, an IllegalArgumentException will be thrown
     *        if no 'name' attribute can be found, otherwise null will returned.
     * @return the interpreted value (AVT) of the Node attribute.
     * @throws java.lang.IllegalArgumentException ...
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static String  getAttribute(XSLProcessorContext context,
            ElemExtensionCall elem,
            String  name,
            boolean throwsIfNull) {
        try {
            String  value = elem.getAttribute(name, context.getContextNode(),
                    context.getTransformer());

            if (value == null && throwsIfNull) {
                throw new IllegalArgumentException ("no \"" + name
                        + "\" attribute");
            }
            return value;
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Returns an XObject value (AVT or not) of a Node attribute whose name is 'name'.
     * @param context the XSLProcessorContext to be used.
     * @param elem the ElemExtensionCall to be used.
     * @param name name of the attribute.
     * @param isAVT should the value be interpreted as an AVT?
     * @param throwsIfNull if true, an IllegalArgumentException will be thrown
     *        if no 'name' attribute can be found, otherwise null will returned.
     * @return the interpreted value (if AVT) of the Node attribute.
     * @throws java.lang.IllegalArgumentException ...
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static XObject getXOAttribute(XSLProcessorContext context,
            ElemExtensionCall elem,
            String  name,
            boolean isAVT,
            boolean throwsIfNull) {
        String  value = null;

        try {
            if (isAVT) {
                value = elem.getAttribute(name, context.getContextNode(),
                        context.getTransformer());
            } else {
                value = elem.getAttribute(name);
            }
            if (value == null && throwsIfNull) {
                throw new IllegalArgumentException ("no \"" + name
                        + "\" attribute");
            }
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        return (value == null) ? null : evaluate(context, elem, value);
    }

    /**
     * Returns an Object value (AVT or not) of a Node attribute whose name is 'name'.
     * @param context the XSLProcessorContext to be used.
     * @param elem the ElemExtensionCall to be used.
     * @param name name of the attribute.
     * @param clazz Class of the Object to return.
     * @param isAVT should the value be interpreted as an AVT?
     * @param throwsIfNull if true, an IllegalArgumentException will be thrown
     *        if no 'name' attribute can be found, otherwise null will returned.
     * @return the interpreted value (if AVT) of the Node attribute.
     * @throws java.lang.ClassCastException if the Object is not an instance of
     *         clazz.
     * @throws java.lang.IllegalArgumentException ...
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static Object  getOAttribute(XSLProcessorContext context,
            ElemExtensionCall elem,
            String  name,
            Class  clazz,
            boolean isAVT,
            boolean throwsIfNull) {
        XObject xo = getXOAttribute(context, elem, name, isAVT, throwsIfNull);

        if (xo == null) {
            return null;
        }
        if (xo.object() == null
                || clazz.isAssignableFrom(xo.object().getClass())) {
            return xo.object();
        }
        throw new WrappedRuntimeException(new ClassCastException ("Class of \""
                + name + "\" attribute should be " + clazz));
    }

    /**
     * Returns an interpreted value (AVT) of a Node attribute whose name is "avt".
     * @param context the XSLProcessorContext to be used.
     * @param elem the ElemExtensionCall to be used.
     * @return the evaluated result.
     * @throws java.lang.IllegalArgumentException bad context.
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static String  evaluate(XSLProcessorContext context, ElemExtensionCall elem) {
        return getAttribute(context, elem, "avt", true);
    }

    /**
     * Returns an interpreted value (AVT) of a Node attribute whose name is equals
     * to the avt parameter.
     * @param context the ExpressionContext to be used.
     * @param avt the name of the attribute.
     * @return the evaluated result.
     * @throws java.lang.IllegalArgumentException bad context.
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static String  evaluate(ExpressionContext context, String  avt) {
        try {
            if (!(context instanceof XPathExpressionContext)) {
                throw new IllegalArgumentException ("bad context: " + context);
            }
            XPathExpressionContext xpec = (XPathExpressionContext) context;
            StylesheetHandler sh = (StylesheetHandler) (
                    ((TransformerFactoryImpl) (TransformerFactory.newInstance())).newTemplatesHandler()
                    );

            return evaluateAttribute(sh, xpec.getXPathContext(),
                    xpec.getContextNode(), avt);
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Returns the interpretion (AVT) of the expression parameter.
     * @param context the XSLProcessorContext to be used.
     * @param elem the ElemExtensionCall to be used.
     * @param expression the expression to be evaluated.
     * @return the evaluated result.
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static XObject evaluate(XSLProcessorContext context,
            ElemExtensionCall elem,
            String  expression) {
        try {
            XPathContext xpc = context.getTransformer().getXPathContext();
            XPath xp = new XPath(expression, elem, xpc.getNamespaceContext(),
                    XPath.SELECT);

            return xp.execute(xpc, context.getContextNode(), elem);
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Returns an XObject resulting from the evaluation of the str parameter.
     * @param contextNode the Node to be used as context.
     * @param str the expression to be evaluated.
     * @return the evaluated result.
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static XObject evaluate(Node  contextNode, String  str) {
        return evaluate(contextNode, str, contextNode, null);
    }

    /**
     * Returns an XObject resulting from the evaluation of the str parameter.
     * @param contextNode the Node to be used as context.
     * @param str the expression to be evaluated.
     * @param el the ErrorListener to be used.
     * @return the evaluated result.
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static XObject evaluate(Node  contextNode, String  str, ErrorListener  el) {
        return evaluate(contextNode, str, contextNode, el);
    }

    /**
     * Returns an XObject resulting from the evaluation of the str parameter.
     * @param contextNode the Node to be used as context.
     * @param str the expression to be evaluated.
     * @param namespaceNode the namespace Node to be used.
     * @param el the ErrorListener to be used.
     * @return the evaluated result.
     * @throws org.apache.xml.utils.WrappedRuntimeException ...
     */
    public static XObject evaluate(Node  contextNode, String  str, Node  namespaceNode, ErrorListener  el) {
        try {
            XPathContext xpc = new XPathContext();
            PrefixResolverDefault pr = new PrefixResolverDefault((namespaceNode.getNodeType()
                    == Node.DOCUMENT_NODE)
                    ? ((Document ) namespaceNode).getDocumentElement()
                    : namespaceNode);
            XPath xp = new XPath(str, null, pr, XPath.SELECT, el);
            int iContextNode = xpc.getDTMHandleFromNode(contextNode);

            return xp.execute(xpc, iContextNode, pr);
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }
    
    // DT 11.08.2003 BEGIN
 // Construct AVT, depending on java release
 private static AVT constructAVT(StylesheetHandler handler, 
            String  uri, 
            String  name, 
            String  rawName, 
            String  stringedValue) throws Exception  {
        Class [] parameterType = new Class [6];
        Constructor  avtConstr = null;
        AVT avt = null;

        try {
            parameterType[0] = Class.forName("org.apache.xalan.processor.StylesheetHandler");
            parameterType[1] = Class.forName("java.lang.String");
            parameterType[2] = Class.forName("java.lang.String");
            parameterType[3] = Class.forName("java.lang.String");
            parameterType[4] = Class.forName("java.lang.String");
            parameterType[5] = Class.forName("org.apache.xalan.templates.ElemTemplateElement");
            avtConstr = Class.forName("org.apache.xalan.templates.AVT").getConstructor(parameterType);
            
            Object [] parameter = new Object [6];

            parameter[0] = handler;
            parameter[1] = uri;
            parameter[2] = name;
            parameter[3] = rawName;
            parameter[4] = stringedValue;
            parameter[5] = null;
            avt = (AVT) avtConstr.newInstance(parameter);
        } catch (Exception  e) {
            parameterType = new Class [5];
            parameterType[0] = Class.forName("org.apache.xalan.processor.StylesheetHandler");
            parameterType[1] = Class.forName("java.lang.String");
            parameterType[2] = Class.forName("java.lang.String");
            parameterType[3] = Class.forName("java.lang.String");
            parameterType[4] = Class.forName("java.lang.String");
            avtConstr = Class.forName("org.apache.xalan.templates.AVT").getConstructor(parameterType);
            
            Object [] parameter = new Object [5];

            parameter[0] = handler;
            parameter[1] = uri;
            parameter[2] = name;
            parameter[3] = rawName;
            parameter[4] = stringedValue;
            avt = (AVT) avtConstr.newInstance(parameter);
        }
        
        return avt;
    }

    // DT 11.08.2003 END
 
    /**
     * Returns a String resulting from the evaluation of the expr parameter (AVT).
     * @param sh the StylesheetHandler to be used.
     * @param xpc the XPathContext to be used.
     * @param node the Node to be used as context.
     * @param expr the expression to be evaluated.
     * @param namespaceNode the namespace Node to be used.
     * @param el the ErrorListener to be used.
     * @return the evaluated result.
     * @throws javax.xml.transform.TransformerException ...
     */
    public static String  evaluateAttribute(StylesheetHandler sh,
            XPathContext xpc,
            Node  node,
            String  expr)
        throws TransformerException  {
        String  uri = node.getNamespaceURI();

        if (uri == null) {
            uri = "";
        }
            
        String  name = node.getNodeName();

        if (name == null) {
            name = "";
        }
        String  prefix = node.getPrefix();
        String  rawName = ((prefix != null) ? prefix + ":" : "") + name;
        // DT 11.08.2003 BEGIN
 // AVT avt = new AVT(sh, uri, name, rawName,  expr);
 AVT avt = null;

        try {
            avt = constructAVT(sh, uri, name, rawName, expr);
        } catch (Exception  e) {
            e.printStackTrace();
            throw new TransformerException ("AVT constructor not found");
        }
        // DT 11.08.2003 END
 PrefixResolverDefault pr = new PrefixResolverDefault((node.getNodeType()
                == Node.DOCUMENT_NODE)
                ? ((Document ) node).getDocumentElement()
                : node);

        // System.out.println("Result " + avt.evaluate(xpc, xpc.getDTMHandleFromNode(node), pr).toString());
 return avt.evaluate(xpc, xpc.getDTMHandleFromNode(node), pr).toString();
    }

    /**
     * Checks ni1 and ni2 for strict equality (same names, same attributes,
     * same child nodes...).
     * @param ni1 the first NodeIterator.
     * @param ni2 the second NodeIterator.
     * @return true if ni1 equals ni2, false otherwise.
     */
    public static boolean equals(NodeIterator ni1, NodeIterator ni2) {
        NodeSet ns1 = new NodeSet(ni1);
        NodeSet ns2 = new NodeSet(ni2);

        if (ns1.getLength() != ns2.getLength()) {
            return false;
        }
        for (int i = 0; i < ns1.getLength(); i++) {
            if (!equals(ns1.elementAt(i), ns2.elementAt(i))) {
                return false;
            }
        }
        return true;
    }

    /**
     * Checks n1 and n2 for strict equality (same names, same attributes,
     * same child nodes...).
     * @param n1 the first Node.
     * @param n2 the second Node.
     * @return true if n1 equals n2, false otherwise.
     */
    public static boolean equals(Node  n1, Node  n2) {
        if (n1 == null && n2 == null) {
            return true;
        }
        if (n1 == null || n2 == null) {
            return false;
        }
        if (n1.getNodeType() != n2.getNodeType()) {
            return false;
        }
        if (!n1.getNodeName().equals(n2.getNodeName())) {
            return false;
        }
        String  v1 = n1.getNodeValue();
        String  v2 = n2.getNodeValue();

        if ((v1 == null && v2 != null) || (v1 != null && !v1.equals(v2))) {
            return false;
        }
        NamedNodeMap  nnm1 = n1.getAttributes();
        NamedNodeMap  nnm2 = n2.getAttributes();

        if ((nnm1 == null || nnm2 == null) && nnm1 != nnm2) {
            return false;
        }
        if (nnm1 != null) {
            int nnm1Length = nnm1.getLength();

            if (nnm1Length != nnm2.getLength()) {
                return false;
            }
            for (int i = 0; i < nnm1Length; i++) {
                if (!equals(nnm1.item(i), nnm2.item(i))) {
                    return false;
                }
            }
        }
        
        NodeList  nl1 = n1.getChildNodes();
        NodeList  nl2 = n2.getChildNodes();
        int nl1Length = nl1.getLength();

        if (nl1Length != nl2.getLength()) {
            return false;
        }
        for (int i = 0; i < nl1Length; i++) {
            if (!equals(nl1.item(i), nl2.item(i))) {
                return false;
            }
        }
        
        return true;
    }

    /**
     * Returns the Document used in the context parameter.
     * @param context an ExpressionContext.
     * @return the Document.
     * @throws org.apache.xml.utils.WrappedRuntimeException failed...
     */
    public static Document  getContextDocument(ExpressionContext context) {
        try {
            return context.getContextNode().getOwnerDocument();
        } catch (Exception  e) {
            throw new WrappedRuntimeException("Failed to get Document from ExpressionContext",
                    e);
        }
    }
}
