
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext;

import org.ejen.util.XSLUtil;
import org.ejen.util.DOMUtil;
import java.util.Enumeration ;
import java.io.BufferedInputStream ;
import java.io.FileInputStream ;
import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.apache.xpath.NodeSet;
import org.apache.xml.utils.WrappedRuntimeException;
import org.apache.xalan.extensions.ExpressionContext;

/**
 * Properties utility (static methods).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:pro="org.ejen.ext.Properties"
 *                  exclude-result-prefixes="pro ..."</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="xml" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;xsl:copy-of select="pro:{@link #getSystemProperties(ExpressionContext) getSystemProperties}()"/&gt;
 *      &lt;xsl:copy-of select="pro:{@link #load(ExpressionContext,String) load}($file-name)"/&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 * @see java.util.Properties
 */
public class Properties {

    /**
     * Protected constructor (prevents instanciation).
     */
    protected Properties() {}

    /**
     * Returns a NodeSet that contains system properties.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="pro:getSystemProperties()"/&gt;
     * </pre></td></tr></table>
     * <p>
     * Returned NodeSet contains a set of property Nodes with the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;property&gt;
     *    &lt;name&gt;&lt;![CDATA[java.runtime.name]]&gt;&lt;/name&gt;
     *    &lt;value&gt;&lt;![CDATA[Java(TM) 2 Runtime Environment, Standard Edition]]&gt;&lt;/value&gt;
     *  &lt;/property&gt;
     * </pre></td></tr></table>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return system properties.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error or SecurityException.
     */
    public static NodeSet getSystemProperties(ExpressionContext context) {
        Document  doc = XSLUtil.getContextDocument(context);

        try {
            java.util.Properties  props = System.getProperties();
            NodeSet ns = new NodeSet();

            for (Enumeration  e = props.propertyNames(); e.hasMoreElements();) {
                String  name = (String ) (e.nextElement());
                Element  parent = doc.createElement("property");

                DOMUtil.createCDATANode(doc, parent, "name", name);
                DOMUtil.createCDATANode(doc, parent, "value",
                        props.getProperty(name, ""));
                ns.addElement(parent);
            }
            return ns;
        } catch (WrappedRuntimeException e) {
            throw e;
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Loads a properties file and returns it as a NodeSet.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="pro:load($file-name)"/&gt;
     * </pre></td></tr></table>
     * <p>
     * Returned NodeSet contains a set of property Nodes with the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;property&gt;
     *    &lt;name&gt;&lt;![CDATA[path]]&gt;&lt;/name&gt;
     *    &lt;value&gt;&lt;![CDATA[../../ejen]]&gt;&lt;/value&gt;
     *  &lt;/property&gt;
     * </pre></td></tr></table>
     * <p>
     * See {@link java.util.Properties#load(java.io.InputStream)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the file.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param fileName name of the file.
     * @return system properties.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error or SecurityException.
     */
    public static NodeSet load(ExpressionContext context, String  fileName) {
        fileName = XSLUtil.evaluate(context, fileName);
        Document  doc = XSLUtil.getContextDocument(context);
        BufferedInputStream  bis = null;

        try {
            bis = new BufferedInputStream (new FileInputStream (fileName));
            java.util.Properties  props = new java.util.Properties ();

            props.load(bis);
            NodeSet ns = new NodeSet();

            for (Enumeration  e = props.propertyNames(); e.hasMoreElements();) {
                String  name = (String ) (e.nextElement());
                Element  parent = doc.createElement("property");

                DOMUtil.createCDATANode(doc, parent, "name", name);
                DOMUtil.createCDATANode(doc, parent, "value",
                        props.getProperty(name, ""));
                ns.addElement(parent);
            }
            return ns;
        } catch (WrappedRuntimeException e) {
            throw e;
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        finally {
            if (bis != null) {
                try {
                    bis.close();
                } catch (Exception  e) {}
                finally {
                    bis = null;
                }
            }
        }
    }
}
