
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen;

import org.apache.tools.ant.Task;
import org.apache.tools.ant.BuildException;

/**
 * Ejen ant task class (wrapes an {@link org.ejen.EjenRootNode EjenRootNode}).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (ant build file)</th></tr>
 * <tr><td class="usage"><pre><code>
 *  &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 *
 *  &lt;project name="generate" default="build"&gt;
 *
 *    <b>&lt;taskdef name="ejen" classname="org.ejen.EjenTask"/&gt;</b>
 *
 *    &lt;target name="build"&gt;
 *      <b>&lt;ejen [{@link #setStacktrace(boolean) stacktrace}="(true|false)"]&gt;</b>
 *        ...
 *        &lt;{@link org.ejen.EjenSourceNode source} .../&gt;
 *        &lt;{@link org.ejen.EjenMergeNode merge} .../&gt;
 *        &lt;{@link org.ejen.EjenSaveNode save} .../&gt;
 *        &lt;{@link org.ejen.EjenFilterNode filter} .../&gt;
 *        &lt;{@link org.ejen.EjenTemplateNode template} .../&gt;
 *        ...
 *      <b>&lt;/ejen&gt;</b>
 *    &lt;/target&gt;
 *
 *  &lt;/project&gt;
 * </code></pre></td></tr></table>
 * <p>
 * <b>Parent nodes</b>:
 * <ul>
 *   <li>Ant <code>target</code> node.
 * </ul>
 * <p>
 * @author F. Wolff
 * @version 1.0
 */
public class EjenTask extends Task {
    private EjenRootNode _ern = null;
    private boolean _stacktrace = false;

    /**
     * Constructor (creates a new EjenRootNode and register an
     * {@link org.ejen.EjenListener EjenListener} for message/error
     * reporting.
     */
    public EjenTask() {
        _ern = new EjenRootNode();
        _ern.setListener(new EjenListener() {
            public void stateChanged(EjenEvent ev) {
                log("(" + ev.getMessage() + ") " + ev.getSource().toString(),
                        ev.getLevel());
            }

            public void nodeMessageSent(EjenEvent ev) {
                log(ev.getMessage(), ev.getLevel());
            }

            public void xslMessageSent(EjenEvent ev) {
                log(ev.getMessage(), ev.getLevel());
            }
        });
    }

    /**
     * <b>[optional]</b> - sets the stacktrace option for error reporting (print
     * or not a stack trace). Default is <code>false</code>.
     * @param stacktrace a <code>String</code> equals to "true" or "false",
     *        automatically converted to a <code>boolean</code> by Ant.
     */
    public void setStacktrace(boolean stacktrace) {
        _stacktrace = stacktrace;
    }

    /**
     * Creates a new filter node.
     * @return the new filter node.
     */
    public EjenFilterNode createFilter() {
        EjenFilterNode f = new EjenFilterNode();

        _ern.appendChildNode(f);
        return f;
    }

    /**
     * Creates a new template node.
     * @return the new template node.
     */
    public EjenTemplateNode createTemplate() {
        EjenTemplateNode t = new EjenTemplateNode();

        _ern.appendChildNode(t);
        return t;
    }

    /**
     * Creates a new source node.
     * @return the new source node.
     */
    public EjenSourceNode createSource() {
        EjenSourceNode s = new EjenSourceNode();

        _ern.appendChildNode(s);
        return s;
    }

    /**
     * Creates a new save node.
     * @return the new save node.
     */
    public EjenSaveNode createSave() {
        EjenSaveNode s = new EjenSaveNode();

        _ern.appendChildNode(s);
        return s;
    }

    /**
     * Creates a new merge node.
     * @return the new merge node.
     */
    public EjenMergeNode createMerge() {
        EjenMergeNode s = new EjenMergeNode();

        _ern.appendChildNode(s);
        return s;
    }

    /**
     * Executes the generation, according to child nodes.
     * @throws org.apache.tools.ant.BuildException if goes wrong.
     */
    public void execute() throws BuildException {
        try {
            _ern.check();
            _ern.beforeProcess();
            _ern.process();
            _ern.afterProcess();
            _ern.idle();
        } catch (Exception  e) {
            throw new BuildException("\n\n"
                    + EjenErrors.toString(null, null, e, _stacktrace));
        }
    }
}
