
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen;

import java.lang.reflect.InvocationTargetException ;
import java.util.Properties ;
import java.util.Vector ;
import java.util.Enumeration ;
import java.io.PrintWriter ;
import java.io.StringWriter ;
import javax.xml.transform.TransformerException ;
import javax.xml.transform.SourceLocator ;
import org.apache.xml.utils.WrappedRuntimeException;
import org.xml.sax.SAXException ;
import org.xml.sax.SAXParseException ;

/**
 * Ejen errors class utility.
 * @author F. Wolff
 * @version 1.0
 */
public class EjenErrors {
    public static final String  EJEN_INFORMATION = "Ejen information";
    public static final String  STACK_TRACE = "stack-trace";
    public static final String  ID_FILE = "file";
    public static final String  ID_NOTE = "note";
    public static final String  ID_MESSAGE = "message";
    public static final String  ID_PUBLIC_ID = "public-id";
    public static final String  ID_SYSTEM_ID = "system-id";
    public static final String  ID_LINE = "line";
    public static final String  ID_COLUMN = "column";
    public static final String  LINE_SEPARATOR = System.getProperty("line.separator",
            "\n");
    private EjenErrors() {}

    /**
     * Returns an EjenError array (no stack trace).
     * See {@link #get(String file, String note, Throwable t, boolean printStackTrace)}.
     * @param file name of the file where the error occured.
     * @param note additional message.
     * @param t the exception to be analysed in order to retrieve all
     *        embedded exceptions information.
     * @return the EjenError array.
     */
    public static EjenError[] get(String  file, String  note, Throwable  t) {
        return get(file, note, t, false);
    }

    /**
     * Returns an EjenError array.
     * <p>
     * If the file or the note argument is not null, the first EjenError will be
     * an "Ejen information" with the properties "file" or "note" filled in (at least
     * one those properties, depending on nullity).
     * <p>
     * If the t argument is not null, the following EjenError(s) will have the name
     * of the exception as name and (if not null) the property "message" set to
     * the exception message. While there is an embedded exception in the
     * current exception, a new EjenError is appended to the array (with specific
     * information from the embedded exception) and the embedded exception becomes
     * the current exception. If one of those exception is a SAXParseException or
     * a TransformerException, the properties "column", "line", "public-id" and
     * "system-id" may be included as well (depending on nullity).
     * <p>
     * If printStackTrace is true, an EjenError is finally appended to the array with
     * name set to the last embedded exception name and a "stack-trace" property
     * is added (whose value is the stack trace). 
     * @param file name of the file where the error occured.
     * @param note additional message.
     * @param t the exception to be analysed in order to retrieve all
     *        embedded exceptions information.
     * @param printStackTrace should stack trace be provided or not.
     * @return the EjenError array.
     */
    public static EjenError[] get(String  file, String  note, Throwable  t, boolean printStackTrace) {
        
        Vector  errors = new Vector ();
        Throwable  lastNonNullThrowable = null;

        if (file != null || note != null) {
            EjenError ee = new EjenError(EJEN_INFORMATION);

            if (file != null) {
                ee.putMessage(ID_FILE, file);
            }
            if (note != null) {
                ee.putMessage(ID_NOTE, note);
            }
            errors.add(ee);
        }
        int line, column;
        String  publicId, systemId, eMsg;

        while (t != null) {
            lastNonNullThrowable = t;
            EjenError ee = new EjenError(t.getClass().getName());

            if (t.getMessage() != null) {
                ee.putMessage(ID_MESSAGE, t.getMessage());
            }
            line = column = -1;
            publicId = systemId = null;
            if (t instanceof EjenException) {
                t = ((EjenException) t).getEmbeddedThrowable();
            } else if (t instanceof WrappedRuntimeException) {
                t = ((WrappedRuntimeException) t).getException();
            } else if (t instanceof SAXException ) {
                SAXException  se = (SAXException ) t;

                if (se instanceof SAXParseException ) {
                    SAXParseException  spe = (SAXParseException ) se;

                    column = spe.getColumnNumber();
                    line = spe.getLineNumber();
                    publicId = spe.getPublicId();
                    systemId = spe.getSystemId();
                }
                t = se.getException();
            } else if (t instanceof TransformerException ) {
                TransformerException  te = (TransformerException ) t;
                SourceLocator  sl = te.getLocator();

                if (sl != null) {
                    column = sl.getColumnNumber();
                    line = sl.getLineNumber();
                    publicId = sl.getPublicId();
                    systemId = sl.getSystemId();
                }
                t = ((TransformerException ) t).getCause();
            } else if (t instanceof InvocationTargetException ) {
                t = ((InvocationTargetException ) t).getTargetException();
            } else {
                t = null;
            }
            if (publicId != null) {
                ee.putMessage(ID_PUBLIC_ID, publicId);
            }
            if (systemId != null) {
                ee.putMessage(ID_SYSTEM_ID, systemId);
            }
            if (line != -1) {
                ee.putMessage(ID_LINE, Integer.toString(line));
            }
            if (column != -1) {
                ee.putMessage(ID_COLUMN, Integer.toString(column));
            }
            errors.add(ee);
        }
        if (printStackTrace && lastNonNullThrowable != null) {
            PrintWriter  pw = null;
            EjenError ee = new EjenError(lastNonNullThrowable.getClass().getName());

            try {
                StringWriter  sw = new StringWriter ();

                pw = new PrintWriter (sw);
                lastNonNullThrowable.printStackTrace(pw);
                ee.putMessage(STACK_TRACE, sw.toString());
            } catch (Exception  e) {
                ee.putMessage(STACK_TRACE, "(failed)");
            }
            finally {
                errors.add(ee);
                pw.close();
            }
        }
        return (EjenError[]) errors.toArray(new EjenError[0]);
    }

    /**
     * Returns a String representation of the errors argument.
     * <p>
     * If the errors argument is null, "errors.null" is returned.
     * <p>
     * Otherwise, for each EjenError in the array, a String with the
     * following structure is appended:<pre><code>[&lt;name of the EjenError&gt;] {
     *  &lt;name of property&gt;: &lt;value of property&gt;
     *  &lt;name of property&gt;: &lt;value of property&gt;
     *  ...
     *}<code></pre>
     * @param errors an EjenError array.
     * @return the String representation.
     */
    public static String  toString(EjenError[] errors) {
        if (errors == null) {
            return "errors.null";
        }
        
        StringBuffer  sb = new StringBuffer ();

        for (int i = 0; i < errors.length; i++) {
            if (i > 0) {
                sb.append(LINE_SEPARATOR);
            }
            sb.append('[').append(errors[i].getName()).append("] {").append(LINE_SEPARATOR);
            Properties  msgs = errors[i].getMessages();

            for (Enumeration  e = msgs.propertyNames(); e.hasMoreElements();) {
                String  name = (String ) (e.nextElement());

                sb.append("  ").append(name).append(": ").append(msgs.getProperty(name)).append(LINE_SEPARATOR);
            }
            sb.append("}");
        }
        return sb.toString();
    }

    /**
     * Identical to <code>toString(get(file, note, t, false))</code>.
     * @param file name of the file where the error occured.
     * @param note additional message.
     * @param t the exception to be analysed in order to retrieve all
     *        embedded exceptions information.
     * @return the String representation.
     */
    public static String  toString(String  file, String  note, Throwable  t) {
        return toString(get(file, note, t, false));
    }

    /**
     * Identical to <code>toString(get(file, note, t, printStackTrace))</code>.
     * @param file name of the file where the error occured.
     * @param note additional message.
     * @param t the exception to be analysed in order to retrieve all
     *        embedded exceptions information.
     * @param printStackTrace should stack trace be provided or not.
     * @return the String representation.
     */
    public static String  toString(String  file, String  note, Throwable  t, boolean printStackTrace) {
        return toString(get(file, note, t, printStackTrace));
    }
    
    /**
     * Class that represents one error.
     * @author F. Wolff.
     * @version 1.0.
     */
    public static class EjenError {
        private String  _name;
        private Properties  _messages;

        /**
         * Constructs an EjenError whose name is 'name'.
         * @param name name of this EjenError.
         */
        public EjenError(String  name) {
            _name = name;
            _messages = new Properties ();
        }

        /**
         * Returns the name of this EjenError.
         * @return the name of this EjenError.
         */
        public String  getName() {
            return _name;
        }

        /**
         * Returns the messages of this EjenError as Properties.
         * @return the messages.
         */
        public Properties  getMessages() {
            return _messages;
        }

        /**
         * Adds a message into this EjenError with a name set to id.
         * @param id name of the new message.
         * @param the new message.
         */
        public void putMessage(String  id, String  message) {
            _messages.setProperty(id, message);
        }
    }
}
