
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext.db;

import org.ejen.util.XSLUtil;
import org.ejen.util.DOMUtil;
import java.util.Hashtable ;
import java.util.Enumeration ;
import java.util.Properties ;
import java.sql.Connection ;
import java.sql.DriverManager ;
import org.w3c.dom.Node ;
import org.w3c.dom.traversal.NodeIterator;
import org.apache.xalan.extensions.XSLProcessorContext;
import org.apache.xalan.extensions.ExpressionContext;
import org.apache.xalan.templates.ElemExtensionCall;
import org.apache.xml.utils.WrappedRuntimeException;
import org.apache.xpath.objects.XObject;
import org.apache.xpath.objects.XNodeSet;

/**
 * JDBC connections utility (static methods).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:mta="org.ejen.ext.db.BasicMetaDataConnection"
 *                  extension-element-prefixes="mta ..."
 *                  exclude-result-prefixes="mta ..."</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="xml" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;mta:{@link #open(XSLProcessorContext,ElemExtensionCall) open} name="connection1" select="connections/connection[1]"/&gt;
 *      &lt;xsl:if test="not(mta:{@link #open(ExpressionContext,String,NodeIterator) open}('connection3',connections/connection[3]))"&gt;
 *        ...
 *      &lt;/xsl:if&gt;
 *      &lt;mta:{@link #setActive(XSLProcessorContext,ElemExtensionCall) setActive} name="connection1"/&gt;
 *      &lt;mta:{@link #close(XSLProcessorContext,ElemExtensionCall) close} name="connection1"/&gt;
 *      &lt;xsl:if test="not(mta:{@link #close(ExpressionContext,String) close}('connection3'))"&gt;
 *        ...
 *      &lt;/xsl:if&gt;
 *      &lt;mta:{@link #closeAll(XSLProcessorContext,ElemExtensionCall) closeAll}/&gt;
 *      &lt;xsl:if test="not(mta:{@link #closeAll(ExpressionContext) closeAll}())"&gt;
 *        ...
 *      &lt;/xsl:if&gt;
 *
 *      &lt;xsl:copy-of select="mta:{@link TableMetaDataNodeBuilder#getTableMetaData(ExpressionContext,String) getTableMetaData}('ADDRESS')"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link TableMetaDataNodeBuilder#getTableMetaData(ExpressionContext,String,String,String) getTableMetaData}('CAT','SCH','ADDRESS')"/&gt;
 *
 *      &lt;xsl:variable name="errors" select="mta:{@link MetaDataNodeBuilder#getErrors(ExpressionContext) getErrors()}"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getDatabaseInformation(ExpressionContext) getDatabaseInformation}()/@*"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getResultSetMetaData(ExpressionContext,String) getResultSetMetaData}('ADDRESS')"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getResultSetMetaData(ExpressionContext,String,String) getResultSetMetaData}('ADDRESS','COL1,COL2')"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getPrimaryKeys(ExpressionContext,String) getPrimaryKeys}('ADDRESS')"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getPrimaryKeys(ExpressionContext,String,String,String) getPrimaryKeys}('CAT','SCH','ADDRESS')"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getImportedKeys(ExpressionContext,String) getImportedKeys}('ADDRESS')"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getImportedKeys(ExpressionContext,String,String,String) getImportedKeys}('CAT','SCH','ADDRESS')"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getExportedKeys(ExpressionContext,String) getExportedKeys}('ADDRESS')"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getExportedKeys(ExpressionContext,String,String,String) getExportedKeys}('CAT','SCH','ADDRESS')"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getIndexInfo(ExpressionContext,String) getIndexInfo}('ADDRESS')"/&gt;
 *      &lt;xsl:copy-of select="mta:{@link MetaDataNodeBuilder#getIndexInfo(ExpressionContext,String,String,String) getIndexInfo}('CAT','SCH','ADDRESS')"/&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class BasicMetaDataConnection extends TableMetaDataNodeBuilder {
    public static final String  S_CI_CONNECTION_NODE_NAME = "connection";
    public static final String  S_CI_DRIVER = "driver";
    public static final String  S_CI_URL = "url";

    /** Connections table */
    protected static Hashtable  _conns = new Hashtable ();

    /**
     * Prevents instanciation.
     */
    protected BasicMetaDataConnection() {}

    /**
     * Opens a JDBC connection and puts it in the connections table. If a
     * connection is already registered with the same name, closes it before.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;mta:open name="connection1" select="connections/connection[1]"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The selected connection <code>Node</code> must have the following format:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;connection url="jdbc:hsqldb:hsql://localhost:1476"
     *              driver="org.hsqldb.jdbcDriver"&gt;
     *    &lt;property name="user" value="sa"/&gt;
     *    &lt;property name="password" value=""/&gt;
     *    [...any number of other JDBC properties]
     *  &lt;/connection&gt;
     * </pre></td></tr></table>
     * <p>
     * The <code>&lt;property...&gt;</code> nodes are all passed to the
     * {@link java.sql.DriverManager#getConnection(String,Properties)} method.
     * <p>
     * Connections errors (SQLException) are not thrown. Instead, an errors
     * <code>NodeSet</code> is built that can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>name <b>[Mantatory/AVT]</b> name of the connection to be opened.
     *   <dd>name <b>[Mantatory]</b> connection Node to use.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @return <code>true</code> if connection succeed, <code>false</code> otherwise.
     * @throws java.lang.RuntimeException
     *         if the 'name' or 'select' attribute is missing or incorrect.
     * @throws org.apache.xml.utils.WrappedRuntimeException
     *         if the 'select' result is not a <code>NodeIterator</code>.
     */
    public static void open(XSLProcessorContext context, ElemExtensionCall elem) {
        String  name = XSLUtil.getAttribute(context, elem, "name", true);
        String  select = XSLUtil.getAttribute(context, elem, "select", true);
        XObject xo = XSLUtil.evaluate(context, elem, select);

        if (xo == null || !(xo instanceof XNodeSet)) {
            throw new RuntimeException ("illegal 'select' attribute in 'connection' node");
        }
        NodeIterator ni = null;

        try {
            ni = xo.nodeset();
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        open(null, name, ni);
    }

    /**
     * Opens a JDBC connection and puts it in the connections table. If a
     * connection is already registered with the same name, closes it before.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:if test="not(mta:open('connection3',connections/connection[3]))"&gt;
     *    ...
     *  &lt;/xsl:if&gt;
     * </pre></td></tr></table>
     * <p>
     * See {@link #open(XSLProcessorContext,ElemExtensionCall)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the connection to be opened.
     *   <dd><b>[Mandatory]</b> connection Node to use.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param name name of the connection.
     * @param connectionNi connection <code>Node</code>.
     * @return <code>true</code> if connection succeed, <code>false</code>
     *         otherwise.
     * @throws java.lang.RuntimeException
     *         if the connection <code>Node</code> incorrect.
     */
    public static boolean open(ExpressionContext context, String  name, NodeIterator connectionNi) {
        name = XSLUtil.evaluate(context, name);
        if (!close(null, name)) {
            return false;
        }
        try {
            boolean badNode = false;
            Properties  props = null;
            String  driver = null;
            String  url = null;
            Node  elt = connectionNi.nextNode();

            if (elt == null
                    || !S_CI_CONNECTION_NODE_NAME.equals(elt.getNodeName())) {
                badNode = true;
            } else {
                driver = DOMUtil.getAttribute(elt, S_CI_DRIVER);
                url = DOMUtil.getAttribute(elt, S_CI_URL);
                if (driver == null || url == null) {
                    badNode = true;
                } else {
                    props = DOMUtil.getChildProperties(elt);
                }
            }
            if (badNode) {
                throw new RuntimeException ("Illegal jdbc 'connection' node");
            }
            Class.forName(driver);
            _activeConn = DriverManager.getConnection(url, props);
            _conns.put(name, _activeConn);
        } catch (Exception  e) {
            appendErrorNode(e);
        }
        return _errors == null;
    }

    /**
     * Closes a JDBC connection and removes it from the connections table.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;mta:close name="connection1"/&gt;
     * </pre></td></tr></table>
     * <p>
     * Connections errors (SQLException) are not thrown. Instead, an errors
     * <code>NodeSet</code> is built that can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>name <b>[Mantatory/AVT]</b> name of the connection to be closed.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @return <code>true</code> if closing connection succeed, <code>false</code>
     *         otherwise.
     * @throws java.lang.RuntimeException if the 'name' attribute is missing or
     *         does not match a registered connection.
     */
    public static void close(XSLProcessorContext context, ElemExtensionCall elem) {
        close(null, XSLUtil.getAttribute(context, elem, "name", true));
    }

    /**
     * Closes a JDBC connection and removes it from the connections table.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:if test="not(mta:close('connection3'))"&gt;
     *    ...
     *  &lt;/xsl:if&gt;
     * </pre></td></tr></table>
     * <p>
     * Connections errors (SQLException) are not thrown. Instead, an errors
     * <code>NodeSet</code> is built that can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the connection to be closed.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param name connection name.
     * @return <code>true</code> if closing connection succeed, <code>false</code>
     *         otherwise.
     * @throws java.lang.RuntimeException
     *         if the 'name' parameter does not match a registered connection.
     */
    public static boolean close(ExpressionContext context, String  name) {
        if (context != null) {
            name = XSLUtil.evaluate(context, name);
        }
        _errors = null;
        if (context != null) {
            Connection  conn = (Connection ) (_conns.get(name));

            if (conn != null) {
                if (_activeConn == conn) {
                    _activeConn = null;
                }
                try {
                    conn.close();
                } catch (Exception  e) {
                    appendErrorNode(e);
                }
                finally {
                    _conns.remove(name);
                }
            } else {
                throw new RuntimeException ("No jdbc connection with this name: "
                        + name);
            }
        }
        return _errors == null;
    }

    /**
     * Sets the current active JDBC connection.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;mta:setActive name="connection1"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>name <b>[Mantatory/AVT]</b> name of the connection to be activated.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @throws java.lang.RuntimeException if the 'name' attribute is missing or
     *         does not match a registered connection.
     */
    public static void setActive(XSLProcessorContext context, ElemExtensionCall elem) {
        String  name = XSLUtil.getAttribute(context, elem, "name", true);
        Connection  conn = (Connection ) (_conns.get(name));

        if (conn != null) {
            _activeConn = conn;
        } else {
            throw new RuntimeException ("No jdbc connection with this name: "
                    + name);
        }
    }

    /**
     * Closes all JDBC connections.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;mta:closeAll/&gt;
     * </pre></td></tr></table>
     * <p>
     * Connections errors (SQLException) are not thrown. Instead, an errors
     * <code>NodeSet</code> is built that can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @return <code>true</code> if closing all connections succeed, <code>false</code>
     *         otherwise.
     */
    public static void closeAll(XSLProcessorContext context, ElemExtensionCall elem) {
        closeAll(null);
    }

    /**
     * Closes all JDBC connections.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:if test="not(mta:closeAll())"&gt;
     *    ...
     *  &lt;/xsl:if&gt;
     * </pre></td></tr></table>
     * <p>
     * Connections errors (SQLException) are not thrown. Instead, an errors
     * <code>NodeSet</code> is built that can be retreived by the
     * {@link MetaDataNodeBuilder#getErrors(ExpressionContext)} method.
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return <code>true</code> if closing all connections succeed, <code>false</code>
     *         otherwise.
     */
    public static boolean closeAll(ExpressionContext context) {
        _errors = null;
        for (Enumeration  e = _conns.elements(); e.hasMoreElements();) {
            Connection  conn = (Connection ) (e.nextElement());

            try {
                conn.close();
            } catch (Exception  f) {
                appendErrorNode(f);
            }
        }
        _conns.clear();
        return _errors == null;
    }
}
