
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext;

import org.ejen.util.XSLUtil;
import org.ejen.util.DOMUtil;
import java.io.File ;
import java.io.FileFilter ;
import java.io.IOException ;
import java.net.MalformedURLException ;
import org.w3c.dom.Document ;
import org.w3c.dom.Element ;
import org.apache.regexp.RE;
import org.apache.xpath.NodeSet;
import org.apache.xml.utils.WrappedRuntimeException;
import org.apache.xalan.extensions.ExpressionContext;

/**
 * File utility (static methods).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:fiu="org.ejen.ext.FileUtil"
 *                  exclude-result-prefixes="fiu ..."</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="xml" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;xsl:copy-of select="fiu:{@link #properties(ExpressionContext,String) properties}($file-name)"/&gt;
 *
 *      &lt;xsl:copy-of select="fiu:{@link #listFiles(ExpressionContext,String) listFiles}('../../src')"/&gt;
 *      &lt;xsl:copy-of select="fiu:{@link #listFiles(ExpressionContext,String,String) listFiles}('../../src','\.java$|\.jjt$')"/&gt;
 *      &lt;xsl:copy-of select="fiu:{@link #listFiles(ExpressionContext,String,String,boolean) listFiles}('../../src','\.java$|\.jjt$',true())"/&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class FileUtil {
    public static final String  S_TRUE = "true";
    public static final String  S_FALSE = "false";

    /**
     * Protected constructor (prevents instanciation).
     */
    protected FileUtil() {}

    /**
     * Returns a NodeSet that contains informations about the fileName file.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="fiu:properties('$file-name')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The returned NodeSet will have the following structure:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;raw&gt;&lt;![CDATA[../../ejb-1.1/bmp-jboss-hsqldb/out/test/src/org/ejb/test/EJB_DOUBLE_UNIQUE_PK.java]]&gt;&lt;/raw&gt;
     *  &lt;name&gt;&lt;![CDATA[EJB_DOUBLE_UNIQUE_PK.java]]&gt;&lt;/name&gt;
     *  &lt;name-no-ext&gt;&lt;![CDATA[EJB_DOUBLE_UNIQUE_PK]]&gt;&lt;/name-no-ext&gt;
     *  &lt;parent&gt;&lt;![CDATA[../../ejb-1.1/bmp-jboss-hsqldb/out/test/src/org/ejb/test]]&gt;&lt;/parent&gt;
     *  &lt;path&gt;&lt;![CDATA[../../ejb-1.1/bmp-jboss-hsqldb/out/test/src/org/ejb/test/EJB_DOUBLE_UNIQUE_PK.java]]&gt;&lt;/path&gt;
     *  &lt;is-absolute&gt;&lt;![CDATA[false]]&gt;&lt;/is-absolute&gt;
     *  &lt;absolute-path&gt;&lt;![CDATA[/devs/java/ejen-1.0-pre2/examples/simples/file/../../ejb-1.1/bmp-jboss-hsqldb/out/test/src/org/ejb/test/EJB_DOUBLE_UNIQUE_PK.java]]&gt;&lt;/absolute-path&gt;
     *  &lt;canonical-path&gt;&lt;![CDATA[/devs/java/ejen-1.0-pre2/examples/ejb-1.1/bmp-jboss-hsqldb/out/test/src/org/ejb/test/EJB_DOUBLE_UNIQUE_PK.java]]&gt;&lt;/canonical-path&gt;
     *  &lt;url&gt;&lt;![CDATA[file:/devs/java/ejen-1.0-pre2/examples/simples/file/../../ejb-1.1/bmp-jboss-hsqldb/out/test/src/org/ejb/test/EJB_DOUBLE_UNIQUE_PK.java]]&gt;&lt;/url&gt;
     *  &lt;can-read&gt;&lt;![CDATA[true]]&gt;&lt;/can-read&gt;
     *  &lt;can-write&gt;&lt;![CDATA[true]]&gt;&lt;/can-write&gt;
     *  &lt;exists&gt;&lt;![CDATA[true]]&gt;&lt;/exists&gt;
     *  &lt;is-directory&gt;&lt;![CDATA[false]]&gt;&lt;/is-directory&gt;
     *  &lt;is-hidden&gt;&lt;![CDATA[false]]&gt;&lt;/is-hidden&gt;
     *  &lt;last-modified&gt;&lt;![CDATA[1016763382000]]&gt;&lt;/last-modified&gt;
     *  &lt;length&gt;&lt;![CDATA[2380]]&gt;&lt;/length&gt;
     * </pre></td></tr></table>
     * <p>
     * "raw" node contains simply the fileName parameter value. Some of those Nodes may
     * not be available (MalformedURLException [url], IOException [canonical-path],
     * or SecurityException [can-read, can-write, ..., length]).
     * <p>
     * See {@link java.io.File}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the file.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param fileName name of the file.
     * @return file properties.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM error.
     */
    public static NodeSet properties(ExpressionContext context, String  fileName) {
        fileName = XSLUtil.evaluate(context, fileName);
        Document  doc = XSLUtil.getContextDocument(context);

        try {
            File  f = new File (fileName);
            NodeSet ns = new NodeSet();

            ns.addElement(DOMUtil.createCDATANode(doc, "raw", fileName));
            String  s = f.getName();

            ns.addElement(DOMUtil.createCDATANode(doc, "name", s));
            int i = s.indexOf('.');

            ns.addElement(DOMUtil.createCDATANode(doc, "name-no-ext",
                    ((i != -1) ? s.substring(0, i) : s)));
            s = f.getParent();
            ns.addElement(DOMUtil.createCDATANode(doc, "parent",
                    ((s != null) ? s : "")));
            ns.addElement(DOMUtil.createCDATANode(doc, "path", f.getPath()));
            ns.addElement(DOMUtil.createCDATANode(doc, "is-absolute",
                    (f.isAbsolute() ? S_TRUE : S_FALSE)));
            ns.addElement(DOMUtil.createCDATANode(doc, "absolute-path",
                    f.getAbsolutePath()));
            try {
                ns.addElement(DOMUtil.createCDATANode(doc, "canonical-path",
                        f.getCanonicalPath()));
            } catch (IOException  e) {}
            try {
                ns.addElement(DOMUtil.createCDATANode(doc, "url",
                        f.toURL().toString()));
            } catch (MalformedURLException  e) {}
            try {
                ns.addElement(DOMUtil.createCDATANode(doc, "can-read",
                        (f.canRead() ? S_TRUE : S_FALSE)));
            } catch (SecurityException  e) {}
            try {
                ns.addElement(DOMUtil.createCDATANode(doc, "can-write",
                        (f.canWrite() ? S_TRUE : S_FALSE)));
            } catch (SecurityException  e) {}
            try {
                ns.addElement(DOMUtil.createCDATANode(doc, "exists",
                        (f.exists() ? S_TRUE : S_FALSE)));
            } catch (SecurityException  e) {}
            try {
                ns.addElement(DOMUtil.createCDATANode(doc, "is-directory",
                        (f.isDirectory() ? S_TRUE : S_FALSE)));
            } catch (SecurityException  e) {}
            try {
                ns.addElement(DOMUtil.createCDATANode(doc, "is-hidden",
                        (f.isHidden() ? S_TRUE : S_FALSE)));
            } catch (SecurityException  e) {}
            try {
                ns.addElement(DOMUtil.createCDATANode(doc, "last-modified",
                        Long.toString(f.lastModified())));
            } catch (SecurityException  e) {}
            try {
                ns.addElement(DOMUtil.createCDATANode(doc, "length",
                        Long.toString(f.length())));
            } catch (SecurityException  e) {}
            
            return ns;
        } catch (WrappedRuntimeException e) {
            throw e;
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }
    
    /**
     * Returns a NodeSet that contains files list for a given directory.
     * <p>
     * This method does not filter anything but directories (only regular file
     * names are returned) and is not recursive (only file names in the given
     * directory are listed).
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="fiu:listFiles('../../src/org/ejen/ext')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The returned NodeSet will have the following structure:
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;file&gt;
     *    &lt;path&gt;&lt;![CDATA[/devs/java/ejen-1.0-pre3/src/org/ejen/ext]]&gt;&lt;/path&gt;
     *    &lt;sep&gt;&lt;![CDATA[/]]&gt;&lt;/sep&gt;
     *    &lt;name&gt;&lt;![CDATA[XMLInclude.java]]&gt;&lt;/name&gt;
     *  &lt;/file&gt;
     *  &lt;file&gt;
     *    &lt;path&gt;&lt;![CDATA[/devs/java/ejen-1.0-pre3/src/org/ejen/ext]]&gt;&lt;/path&gt;
     *    &lt;sep&gt;&lt;![CDATA[/]]&gt;&lt;/sep&gt;
     *    &lt;name&gt;&lt;![CDATA[FileUtil.java]]&gt;&lt;/name&gt;
     *  &lt;/file&gt;
     *  ...
     * </pre></td></tr></table>
     * <p>
     * "path" Nodes contain "canonical" paths (absolute, without symbolic
     * links, etc.). Because this method is not recursive, "path" Nodes will
     * be always equal to the canonical representation of the "path" argument.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> directory path.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param path directory path.
     * @return a NodeSet that contains "file" Nodes.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM or IO errors...
     */
    public static NodeSet listFiles(ExpressionContext context, String  path) {
        return listFiles(context, path, null, false);
    }
    
    /**
     * Returns a NodeSet that contains a filtered files list for a given directory.
     * <p>
     * This method is not recursive (only file names in the given
     * directory are listed). The "filter" argument must be a regular expression
     * (see {@link org.apache.regexp.RE}) and only refers to regular file names (neither
     * directory names nor full path file names).
     * <p>
     * Tips: "<code>\.java$</code>" means "accept file names that end with
     * '<code>.java</code>'". "<code>\.java$|\.jjt$</code>" means "accept file names
     * that end with '<code>.java</code>' or '<code>.jjt</code>'".
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="fiu:listFiles('../../src','\.java$|\.jjt$')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * See {@link #listFiles(ExpressionContext, String)}.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> directory path.
     *   <dd><b>[Mandatory/AVT]</b> file name filter.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param path directory path.
     * @param filter file name filter.
     * @return a NodeSet that contains "file" Nodes.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM or IO errors...
     */
    public static NodeSet listFiles(ExpressionContext context, String  path,
            String  filter) {
        return listFiles(context, path, filter, false);
    }
    
    /**
     * Returns a NodeSet that contains a filtered files list for a given directory
     * and (optionally) for its sub-directories.
     * <p>
     * This method is recursive if the "rec" argument is true. The "filter" argument must
     * be a regular expression (see {@link #listFiles(ExpressionContext, String,String)}).
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="fiu:listFiles('../../src','\.java$|\.jjt$',true())"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> directory path.
     *   <dd><b>[Mandatory/AVT]</b> file name filter.
     *   <dd><b>[Mandatory]</b> true() or false(): list recursively or not.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param path directory path.
     * @param filter file name filter.
     * @param rec true (recursive) or false.
     * @return a NodeSet that contains "file" Nodes.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM or IO errors...
     */
    public static NodeSet listFiles(ExpressionContext context, String  path,
            String  filter, boolean rec) {
        path = XSLUtil.evaluate(context, path);
        final String  ffilter = (filter != null)
                ? XSLUtil.evaluate(context, filter)
                : null;
        Document  doc = XSLUtil.getContextDocument(context);

        try {
            File  f = new File (path);

            if (!f.isDirectory()) {
                throw new IllegalArgumentException (path + " is not a directory.");
            }
            FileFilter  ff = null;

            if (ffilter != null) {
                ff = new FileFilter () {
                    RE _re = new RE(ffilter);
                    public boolean accept(File  f) {
                        return f.isDirectory() || _re.match(f.getName());
                    }

                    public String  getDescription() {
                        return ffilter;
                    }
                };
            }
            File [] files = f.listFiles(ff);
            NodeSet ns = new NodeSet();

            appendFiles(doc, ns, files, ff, rec);
            return ns;
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }
    
    /**
     * Creates "file" Nodes and adds them to a NodeSet.
     * <p>
     * @param doc Document to be used for Node creation.
     * @param ns NodeSet (where Nodes have to be added).
     * @param files files list.
     * @param ff FileFilter to use.
     * @param rec true (recursive) or false.
     * @throws org.apache.xml.utils.WrappedRuntimeException DOM or IO errors...
     */
    protected static void appendFiles(Document  doc, NodeSet ns, File [] files,
            FileFilter  ff, boolean rec) throws Exception  {
        for (int i = 0; i < files.length; i++) {
            File  f = files[i];

            if (!f.isDirectory()) {
                Element  file = doc.createElement("file");
                Element  elt = doc.createElement("path");
                String  path = f.getCanonicalFile().getParent();

                elt.appendChild(doc.createCDATASection((path != null)
                        ? path
                        : ""));
                file.appendChild(elt);
                elt = doc.createElement("sep");
                elt.appendChild(doc.createCDATASection(f.separator));
                file.appendChild(elt);
                elt = doc.createElement("name");
                elt.appendChild(doc.createCDATASection(f.getName()));
                file.appendChild(elt);
                ns.addElement(file);
            } else if (rec) {
                appendFiles(doc, ns, f.listFiles(ff), ff, rec);
            }
        }
    }
}
