
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext;

import org.ejen.util.XSLUtil;
import org.apache.xalan.extensions.XSLProcessorContext;
import org.apache.xalan.extensions.ExpressionContext;
import org.apache.xalan.templates.ElemExtensionCall;
import org.apache.xml.utils.WrappedRuntimeException;

/**
 * Counter utility (instanciable).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:cnt="org.ejen.ext.Counter"
 *                  extension-element-prefixes="cnt ..."
 *                  exclude-result-prefixes="cnt ..."</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="xml" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;xsl:variable name="cnt1" select="cnt:{@link #Counter() new}()"/&gt;
 *      &lt;xsl:variable name="cnt2" select="cnt:{@link #Counter(int) new}(-5)"/&gt;
 *      &lt;xsl:variable name="cnt3" select="cnt:{@link #Counter(Counter) new}([$cnt2])"/&gt;
 *      &lt;cnt:{@link #reset(XSLProcessorContext,ElemExtensionCall) reset} [value="0"] [instance="$cnt2"]/&gt;
 *      &lt;cnt:{@link #incr(XSLProcessorContext,ElemExtensionCall) incr} [step="-2"] [instance="$cnt2"]/&gt;
 *      &lt;cnt:{@link #read(XSLProcessorContext,ElemExtensionCall) read} [instance="$cnt2"]/&gt;
 *      &lt;xsl:param name="i" select="cnt:{@link #read(ExpressionContext) read}([$cnt2])"/&gt;
 *      &lt;cnt:{@link #incrAndRead(XSLProcessorContext,ElemExtensionCall) incrAndRead} [step="-2"] [instance="$cnt2"]/&gt;
 *      &lt;xsl:param name="i" select="cnt:{@link #incrAndRead(ExpressionContext) incrAndRead}([$cnt2])"/&gt;
 *      &lt;xsl:param name="i" select="cnt:{@link #incrAndRead(ExpressionContext,int) incrAndRead}([$cnt2,] -2)"/&gt;
 *      &lt;cnt:{@link #readAndIncr(XSLProcessorContext,ElemExtensionCall) readAndIncr} [step="-2"] [instance="$cnt2"]/&gt;
 *      &lt;xsl:param name="i" select="cnt:{@link #readAndIncr(ExpressionContext) readAndIncr}([$cnt2])"/&gt;
 *      &lt;xsl:param name="i" select="cnt:{@link #readAndIncr(ExpressionContext,int) readAndIncr}([$cnt2,] -2)"/&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class Counter {

    /** Counter default initial value: 1 */
    private static final int DEFAULT_INITIAL_VALUE = 1;

    /** Current value of this Counter */
    private int _value;

    /**
     * Constructs a new Counter object with initial value set to 1. If this
     * constructor is not explicitly used, the Xalan extension mechanism constructs
     * a default instance.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="cnt1" select="cnt:new()"/&gt;
     * </pre></td></tr></table>
     */ 
    public Counter() {
        _value = DEFAULT_INITIAL_VALUE;
    }

    /**
     * Constructs a new Counter object whose initial value is the value parameter.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="cnt2" select="cnt:new(0)"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory]</b> initial value of this Counter.
     * </dl></dd>
     * <p>
     * @param value initial value of this Counter.
     */ 
    public Counter(int value) {
        _value = value;
    }

    /**
     * Constructs a new Counter object whose initial value is the current
     * value of the cnt parameter.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:variable name="cnt3" select="cnt:new($cnt1)"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory]</b> Counter instance.
     * </dl></dd>
     * <p>
     * @param cnt a Counter object.
     */ 
    public Counter(Counter cnt) {
        _value = cnt._value;
    }

    /**
     * Resets the Counter value.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;cnt:reset [value="0"] [instance="$cnt2"]/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>instance <b>[Optional]</b> Counter instance (if not set,
     *       default instance is used).
     *   <dd>value <b>[Optional/AVT]</b> new Counter value (if not set, the counter
     *       will be reset to <code>DEFAULT_INITIAL_VALUE</code>).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @throws org.apache.xml.utils.WrappedRuntimeException with a XSL Exception
     *         or a java.lang.NumberFormatException.
     */
    public void reset(XSLProcessorContext context, ElemExtensionCall elem) {
        Object  o = XSLUtil.getOAttribute(context, elem, "instance",
                Counter.class, false, false);
        String  value = null;

        try {
            value = elem.getAttribute("value", context.getContextNode(),
                    context.getTransformer());
            int iValue = (value != null)
                    ? Integer.parseInt(value)
                    : DEFAULT_INITIAL_VALUE;

            if (o == null) {
                _value = iValue;
            } else {
                ((Counter) o)._value = iValue;
            }
        } catch (Exception  e) {
            throw new WrappedRuntimeException(Counter.class.getName()
                    + ".reset : exception while getting or parsing node attribute 'value' or 'instance'",
                    e);
        }
    }

    /**
     * Increments the value of this Counter by 'step'.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;cnt:incr [step="-2"] [instance="$cnt2"]/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>instance <b>[Optional]</b> Counter instance (if not set,
     *       default instance is used).
     *   <dd>step <b>[Optional/AVT]</b> step used for incrementation (if not set,
     *       the counter will be incremented by 1).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @throws org.apache.xml.utils.WrappedRuntimeException with a XSL Exception
     *         or a java.lang.NumberFormatException.
     */
    public void incr(XSLProcessorContext context, ElemExtensionCall elem) {
        Object  o = XSLUtil.getOAttribute(context, elem, "instance",
                Counter.class, false, false);
        String  step = null;

        try {
            step = elem.getAttribute("step", context.getContextNode(),
                    context.getTransformer());
            int iStep = (step != null) ? Integer.parseInt(step) : 1;

            if (o == null) {
                _value += iStep;
            } else {
                ((Counter) o)._value += iStep;
            }
        } catch (Exception  e) {
            throw new WrappedRuntimeException(Counter.class.getName()
                    + ".incr : exception while getting or parsing node attribute 'step'",
                    e);
        }
    }

    /**
     * Returns the current value of this Counter.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;cnt:read [instance="$cnt2"]/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>instance <b>[Optional]</b> Counter instance (if not set,
     *       default instance is used).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @return the current value of this Counter.
     */
    public int read(XSLProcessorContext context, ElemExtensionCall elem) { 
        Object  o = XSLUtil.getOAttribute(context, elem, "instance",
                Counter.class, false, false);

        if (o == null) {
            return _value;
        } else {
            return ((Counter) o)._value;
        }
    }

    /**
     * Returns the current value of this Counter.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:param name="i" select="cnt:read([$cnt2])"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Optional]</b> Counter instance (if not set, default instance
     *          is used).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return the current value of this Counter.
     */
    public int read(ExpressionContext context) {
        return _value;
    }

    /**
     * Increments the value of this Counter, then returns it.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;cnt:incrAndRead [step="-2"] [instance="$cnt2"]/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>instance <b>[Optional]</b> Counter instance (if not set,
     *       default instance is used).
     *   <dd>step <b>[Optional/AVT]</b> step used for incrementation (if not set,
     *       the counter will be incremented by 1).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @return the current value of this Counter (after incrementation).
     * @throws org.apache.xml.utils.WrappedRuntimeException with a XSL Exception
     *         or a java.lang.NumberFormatException.
     */
    public int incrAndRead(XSLProcessorContext context, ElemExtensionCall elem) {
        Object  o = XSLUtil.getOAttribute(context, elem, "instance",
                Counter.class, false, false);
        String  step = null;

        try {
            step = elem.getAttribute("step", context.getContextNode(),
                    context.getTransformer());
            int iStep = (step != null) ? Integer.parseInt(step) : 1;

            if (o == null) {
                _value += iStep;
            } else {
                ((Counter) o)._value += iStep;
            }
        } catch (Exception  e) {
            throw new WrappedRuntimeException(Counter.class.getName()
                    + ".incrAndRead : exception while getting or parsing node attribute 'step' or 'instance'",
                    e);
        }
        return _value;
    }

    /**
     * Increments the value of this Counter by 1, then returns it.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:param name="i" select="cnt:incrAndRead([$cnt2])"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Optional]</b> Counter instance (if not set, default instance
     *       is used).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return the current value of this Counter (after incrementation).
     */
    public int incrAndRead(ExpressionContext context) {
        return ++_value;
    }

    /**
     * Increments the value of this Counter by 'step', then returns it.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:param name="i" select="cnt:incrAndRead([$cnt2,] -2)"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Optional]</b> Counter instance (if not set, default instance
     *       is used).
     *   <dd><b>[Mandatory]</b> step used for incrementation (if not set,
     *       the counter will be incremented by 1).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param step incrementation step.
     * @return the current value of this Counter (after incrementation).
     */
    public int incrAndRead(ExpressionContext context, int step) {
        return (_value += step);
    }

    /**
     * Returns the value of this Counter, then increments it.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;cnt:readAndIncr [step="-2"] [instance="$cnt2"]/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT Attributes:</b>
     *   <dd>instance <b>[Optional]</b> Counter instance (if not set,
     *       default instance is used).
     *   <dd>step <b>[Optional/AVT]</b> step used for incrementation (if not set,
     *       the counter will be incremented by 1).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param elem automatically passed by the xalan extension mechanism.
     * @return the current value of this Counter (before incrementation).
     * @throws org.apache.xml.utils.WrappedRuntimeException with a XSL Exception
     *         or a java.lang.NumberFormatException.
     */
    public int readAndIncr(XSLProcessorContext context, ElemExtensionCall elem) {
        Object  o = XSLUtil.getOAttribute(context, elem, "instance",
                Counter.class, false, false);
        int oldValue = _value;
        String  step = null;

        try {
            step = elem.getAttribute("step", context.getContextNode(),
                    context.getTransformer());
            int iStep = (step != null) ? Integer.parseInt(step) : 1;

            if (o == null) {
                _value += iStep;
            } else {
                ((Counter) o)._value += iStep;
            }
        } catch (Exception  e) {
            throw new WrappedRuntimeException(Counter.class.getName()
                    + ".readAndIncr : exception while getting or parsing node attribute 'step' or 'instance'",
                    e);
        }
        return oldValue;
    }

    /**
     * Returns the value of this Counter, then increments it by 1.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:param name="i" select="cnt:readAndIncr([$cnt2])"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Optional]</b> Counter instance (if not set, default instance
     *       is used).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @return the current value of this Counter (before incrementation).
     */
    public int readAndIncr(ExpressionContext context) {
        return _value++;
    }

    /**
     * Returns the value of this Counter, then increments it by 'step'.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:param name="i" select="cnt:readAndIncr([$cnt2,] -2)"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Optional]</b> Counter instance (if not set, default instance
     *       is used).
     *   <dd><b>[Mandatory]</b> step used for incrementation (if not set,
     *       the counter will be incremented by 1).
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param step incrementation step.
     * @return the current value of this Counter (before incrementation).
     */
    public int readAndIncr(ExpressionContext context, int step) {
        int oldValue = _value;

        _value += step;
        return oldValue;
    }
}
