
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen;

import java.util.Vector ;
import java.util.Enumeration ;
import java.util.Properties ;
import java.io.*;
import javax.xml.transform.dom.DOMSource ;
import javax.xml.transform.stream.StreamSource ;
import org.apache.xalan.processor.TransformerFactoryImpl;
import org.apache.xalan.transformer.TransformerImpl;
import org.apache.xalan.templates.StylesheetRoot;
import javax.xml.parsers.DocumentBuilder ;
import javax.xml.parsers.DocumentBuilderFactory ;
 
import org.w3c.dom.*;
// For write operation
import javax.xml.transform.Transformer ;
import javax.xml.transform.TransformerFactory ;
import javax.xml.transform.stream.StreamResult ;

/**
 * Parent abstract class of all Ejen...Node classes that use a stylesheet
 * (filter, template, include and import).
 * @author F. Wolff
 * @version 1.0
 */
public abstract class EjenStylesheetNode extends EjenChildNode {
    private String  stylesheet = "";
    
    protected Vector  _childNodes = new Vector ();
    protected String  _file = null;

    /**
     * Returns all non null attributes of this EjenStylesheetNode.
     * @return non null attributes of this EjenStylesheetNode.
     */
    public Properties  getAttributes() {
        Properties  attrs = super.getAttributes();

        if (_file != null) {
            attrs.setProperty("file", _file);
        }
        return attrs;
    }

    /**
     * Returns all child nodes of this EjenStylesheetNode (include, import or
     * param).
     * @return all child nodes of this EjenStylesheetNode.
     */
    public Vector  getChildren() {
        Vector  children = super.getChildren();

        children.addAll(_childNodes);
        return children;
    }

    /**
     * <b>[mandatory/AVT]</b> - sets the file attribute.
     * @param file name of the XSL file stylesheet to be used as filter, template,
     *        include or import.
     */
    public void setFile(String  file) {
        this.transformInputFile(file);
        _file = file;
    }

    /**
     * Creates a new param node and appends it to the list of current child
     * nodes of this EjenStylesheetNode.
     * @return the new param node.
     */
    public EjenParamNode createParam() {
        EjenParamNode p = new EjenParamNode();

        _childNodes.addElement(p);
        return p;
    }

    /**
     * Creates a new include node and appends it to the list of current child
     * nodes of this EjenStylesheetNode.
     * @return the new include node.
     */
    public EjenIncludeNode createInclude() {
        EjenIncludeNode i = new EjenIncludeNode();

        _childNodes.addElement(i);
        return i;
    }

    /**
     * Creates a new import node and appends it to the list of current child
     * nodes of this EjenStylesheetNode.
     * @return the new import node.
     */
    public EjenImportNode createImport() {
        EjenImportNode i = new EjenImportNode();

        _childNodes.addElement(i);
        return i;
    }

    /**
     * Checks this EjenStylesheetNode for mandatory attributes.
     * @throws org.ejen.EjenException if file attribute is not set or if any
     *         call to the check() method of a child node fails.
     */
    public void check() {
        super.check();
        if (_file == null) {
            throw new EjenException(this, "no 'file' attribute");
        }
        for (Enumeration  e = _childNodes.elements(); e.hasMoreElements();) {
            ((EjenChildNode) e.nextElement()).check();
        }
    }

    /**
     * Prepares this EjenStylesheetNode execution. Creates a new Transformer with
     * the stylesheet whose name is provided by the file attribute and pushes a
     * new context onto the context stack.
     */
    public void beforeProcess() {
        super.beforeProcess();
        
        TransformerFactoryImpl tfi = null;

        try {
            tfi = (TransformerFactoryImpl) (getFromGlobalContext(CTX_TRANSFORMER_FACTORY_IMPL));
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }           
        if (tfi == null) {
            throw new EjenException(this,
                    "no '" + CTX_TRANSFORMER_FACTORY_IMPL
                    + "' in global context");
        }
        InputStream inputs = null;

        try {
            inputs = new FileInputStream(evaluateAVT(_file));
            TransformerImpl ti = (TransformerImpl) (tfi.newTransformer(new StreamSource (inputs)));
            EjenContext newContext = cloneContext();

            newContext.put(CTX_TRANSFORMER_IMPL, ti);
            newContext.put(CTX_STYLESHEET_ROOT, ti.getStylesheet());
            pushContext(newContext);
        } catch (EjenException e) {
            throw e;
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }
        finally {
            if (inputs != null) {
                try {
                    inputs.close();
                } catch (Exception  e) {}
                finally {
                    inputs = null;
                }
            }
        }
        
    }

    /**
     * Executes this EjenStylesheetNode (and all child nodes). This method is called
     * by the process() method in each implementation of the EjenStylesheetNode abstract
     * class.
     */
    public void process() {
        super.process();
        try {
            StylesheetRoot sr = (StylesheetRoot) (getFromContext(CTX_STYLESHEET_ROOT));

            if (sr == null) {
                throw new EjenException(this,
                        "no '" + CTX_STYLESHEET_ROOT + "' in context");
            }
            for (Enumeration  is = _childNodes.elements(); is.hasMoreElements();) {
                EjenChildNode ecn = (EjenChildNode) (is.nextElement());
                // sendMessageEvent("Processing " + ecn.toString());
 String  oldMessageIndent = _messageIndent;

                _messageIndent = LOG_INDENT_STR1 + _messageIndent;
                ecn.beforeProcess();
                ecn.process();
                ecn.afterProcess();
                ecn.idle();
                _messageIndent = oldMessageIndent;
            }
            sr.recompose();
        } catch (EjenException e) {
            throw e;
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }
    }

    /**
     * Pops the context that has been pushed by the {@link #beforeProcess()} method.
     */
    public void afterProcess() {
        super.afterProcess();
        
        try {
            popContext();
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }
    }
    
    private boolean transformInputFile(String  file) {
      
        boolean retVal = true;
        DocumentBuilderFactory  factory = DocumentBuilderFactory.newInstance();

        try {
            File datafile = new File(file);
            File output = new File(file);
            
            DocumentBuilder  builder = factory.newDocumentBuilder();
            // factory.setNamespaceAware(false);
            Document inputDocument = builder.parse(datafile);
            Document document = builder.parse(datafile);
            
            boolean hasImports = true;
            NodeList list = document.getElementsByTagName("xsl:import");
            String  importHrefValue = "";

            if (list.item(0) != null) {
                importHrefValue = ((Element) list.item(0)).getAttribute("href");
            } else {
                hasImports = false;
            }
            
            boolean hasIncludes = true;

            list = document.getElementsByTagName("xsl:include");
            String  includeHrefValue = "";

            if (list.item(0) != null) {
                includeHrefValue = ((Element) list.item(0)).getAttribute("href");
            } else {
                hasIncludes = false;
            }
            
            String  hrefValue = "";

            if (hasImports) {
                hrefValue = importHrefValue;
            } else if (hasIncludes) {
                hrefValue = includeHrefValue;
            }
            
            // if path is OK,leave it unchanged.
 if ((hasImports || hasIncludes)
                    && !this.checkFilePath(file, hrefValue)) {
                // Use a Transformer for output
 TransformerFactory  tFactory = TransformerFactory.newInstance();
                StreamSource  stylesource = new StreamSource (new StringReader(stylesheet));
                Transformer  transformer = tFactory.newTransformer(stylesource);
   
                DOMSource  source = new DOMSource (inputDocument);
                StreamResult  result = new StreamResult (new FileOutputStream(output));

                transformer.transform(source, result);
            }
        } catch (Exception  e) {
            e.printStackTrace();
            System.out.println("exception in EjenStylesheetNode.transformInputFile() : "
                    + e.getMessage());
            retVal = false;
        }
        return retVal;
    }
    
    private boolean checkFilePath(String  file, String  hrefValue) {
        boolean retVal = false;
        String  relPath = file;
        String  hrefPath = "";
        // getting main template's path ( path is absolute,so at least one '/' must be in it )
 int indexOfBackslash = file.lastIndexOf("/");

        relPath = relPath.substring(0, indexOfBackslash + 1);
        // getting path in href
        indexOfBackslash = hrefValue.lastIndexOf("/");
        if (indexOfBackslash != -1) {
            hrefPath = hrefValue.substring(0, indexOfBackslash + 1);
        }
        // comparing href attribute value and relPath
 if (comparePaths(relPath, hrefPath)) {
            retVal = true;
        } else {
            int startPosition = 0;

            if (indexOfBackslash == -1) {
                startPosition = 1;
            } else {
                startPosition = hrefPath.length() + 1;
            }
            composeStylesheet(relPath, startPosition);
        }
        return retVal; 
    }
    
    private boolean comparePaths(String  relPath, String  hrefPath) {
        boolean retVal = false;

        if (System.getProperty("path.separator").equals(";")) {
            retVal = relPath.equalsIgnoreCase(hrefPath);
        } else {
            retVal = relPath.equals(hrefPath);
        }
        return retVal;
    }
    
    private void composeStylesheet(String  relPath, int startPosition) {
      
        String  stylesheet = "<?xml version=\"1.0\" encoding=\"UTF-8\"?>"
                + "<xsl:stylesheet version=\"1.0\" xmlns:xsl=\"http://www.w3.org/1999/XSL/Transform\" xmlns:gvs=\"org.ejen.ext.GlobalVariables\">"
                + "<xsl:output method=\"xml\" indent=\"yes\" encoding=\"UTF-8\"/>"
                + "<xsl:template match=\"/\">"
                + "<xsl:element name=\"xsl:stylesheet\">"
                + "<xsl:attribute name=\"version\">1.0</xsl:attribute>"
                + "<xsl:attribute name=\"extension-element-prefixes\">gvs</xsl:attribute>"
                + "<xsl:attribute name=\"exclude-result-prefixes\">gvs</xsl:attribute>"
                + "<xsl:apply-templates/>" + "</xsl:element>"
                + "</xsl:template>" + "<xsl:template match=\"stylesheet/*\">"
                + "<xsl:choose>" + "<xsl:when test=\"name() = 'xsl:import'\">"
                + "<xsl:call-template name=\"import\"/>" + "</xsl:when>"
                + "<xsl:when test=\"name() = 'xsl:include'\">"
                + "<xsl:call-template name=\"include\"/>" + "</xsl:when>"
                + "<xsl:otherwise>" + "<xsl:copy-of select=\".\"/>"
                + "</xsl:otherwise>" + "</xsl:choose>" + "</xsl:template>"
                + "<xsl:template name=\"import\">"
                + "<xsl:element name=\"xsl:import\">"
                + "<xsl:attribute name=\"href\">" + relPath
                + "<xsl:value-of select=\"substring(@href," + startPosition
                + ")\"/></xsl:attribute>" + "</xsl:element>" + "</xsl:template>"
                + "<xsl:template name=\"include\">"
                + "<xsl:element name=\"xsl:include\">"
                + "<xsl:attribute name=\"href\">" + relPath
                + "<xsl:value-of select=\"substring(@href," + startPosition
                + ")\"/></xsl:attribute>" + "</xsl:element>" + "</xsl:template>"
                + "</xsl:stylesheet>";

        this.stylesheet = stylesheet;
      
    }
}
