
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext.parsers.java_1_2;

import org.ejen.util.XSLUtil;
import org.ejen.util.arl.ArlUtil;
import java.io.FileReader ;
import java.io.BufferedReader ;
import java.util.StringTokenizer ;
import java.util.Hashtable ;
import org.apache.xalan.extensions.ExpressionContext;
import org.apache.xml.utils.WrappedRuntimeException;

/**
 * Java source file compilation utility (static methods).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:jsx="org.ejen.ext.parsers.java_1_2.JavaSourceToXML"</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="xml" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;xsl:copy-of select="jsx:{@link #process(ExpressionContext,String) process}('{$name}.java')"/&gt;
 *      &lt;xsl:copy-of select="jsx:{@link #process(ExpressionContext,String,boolean) process}('{$name}.java',true)"/&gt;
 *      &lt;xsl:copy-of select="jsx:{@link #process(ExpressionContext,String,String,String) process}('{$name}.java','','')"/&gt;
 *      &lt;xsl:copy-of select="jsx:{@link #process(ExpressionContext,String,String,String,boolean) process}('{$name}.java','','',true)"/&gt;
 *      &lt;xsl:copy-of select="jsx:{@link #parseJavadoc(ExpressionContext,String) parseJavadoc}(tok/stok[@ki=10])"/&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class JavaSourceToXML implements JavaParserTreeConstants, JavaParserConstants {
    
    private static Hashtable  _nodesMapCache = new Hashtable ();
    private static Hashtable  _tokensMapCache = new Hashtable ();
    
    private final static int[] DEFAULT_NODES_MAP = new int[jjtNodeName.length];
    private final static int[] DEFAULT_TOKENS_MAP = new int[tokenImage.length];
    static {
        DEFAULT_NODES_MAP[JJTIMPORTDECLARATION] = ArlUtil.F_REMOVE;
        DEFAULT_NODES_MAP[JJTTYPEDECLARATION] = ArlUtil.F_CROSS;
        DEFAULT_NODES_MAP[JJTCLASSBODY] = ArlUtil.F_CROSS;
        DEFAULT_NODES_MAP[JJTCLASSBODYDECLARATION] = ArlUtil.F_CROSS;
        DEFAULT_NODES_MAP[JJTFIELDDECLARATION] = ArlUtil.F_REMOVE;
        DEFAULT_NODES_MAP[JJTFORMALPARAMETERS] = ArlUtil.F_CROSS;
        DEFAULT_NODES_MAP[JJTINITIALIZER] = ArlUtil.F_REMOVE;
        DEFAULT_NODES_MAP[JJTBLOCK] = ArlUtil.F_REMOVE;
        for (int i = 0; i <= SINGLE_LINE_COMMENT; i++) {
            DEFAULT_TOKENS_MAP[i] = ArlUtil.F_REMOVE;
        }
        DEFAULT_TOKENS_MAP[MULTI_LINE_COMMENT] = ArlUtil.F_REMOVE;
        DEFAULT_TOKENS_MAP[PACKAGE] = ArlUtil.F_REMOVE;
        DEFAULT_TOKENS_MAP[LPAREN] = ArlUtil.F_REMOVE;
        DEFAULT_TOKENS_MAP[RPAREN] = ArlUtil.F_REMOVE;
        DEFAULT_TOKENS_MAP[LBRACE] = ArlUtil.F_REMOVE;
        DEFAULT_TOKENS_MAP[RBRACE] = ArlUtil.F_REMOVE;
        DEFAULT_TOKENS_MAP[SEMICOLON] = ArlUtil.F_REMOVE;
        DEFAULT_TOKENS_MAP[COMMA] = ArlUtil.F_REMOVE;
        DEFAULT_TOKENS_MAP[THROWS] = ArlUtil.F_REMOVE;
        DEFAULT_TOKENS_MAP[IMPLEMENTS] = ArlUtil.F_REMOVE;
        DEFAULT_TOKENS_MAP[EXTENDS] = ArlUtil.F_REMOVE;
        _nodesMapCache.put("default", DEFAULT_NODES_MAP);
        _tokensMapCache.put("default", DEFAULT_TOKENS_MAP);
    }

    /**
     * Protected constructor (prevents instanciation).
     */
    protected JavaSourceToXML() {}

    /**
     * Returns a <code>Node</code> that represents a java source file.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="jsx:process($java-file)"/&gt;
     * </pre></td></tr></table>
     * <p>
     * Token positions are not included (see
     * {@link #process(ExpressionContext,String,boolean)}).
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the java source file.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param fileName name of the java source file.
     * @return a <code>NodeSet</code> that represents the java source file.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (file not found...).
     */
    public static org.apache.xpath.NodeSet process(ExpressionContext context, String  fileName) {
        return process(XSLUtil.getContextDocument(context),
                XSLUtil.evaluate(context, fileName), null, null, false);
    }

    /**
     * Returns a <code>Node</code> that represents a java source file.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="jsx:process($java-file)"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the java source file.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param fileName name of the java source file.
     * @param tokensPos if true, each "tok" or "stok" Node will include positions
     *        coordinates (see
     *        {@link Token#toNode(org.w3c.dom.Document,org.w3c.dom.Node,int[],boolean)
     *         Token.toNode(...)}).
     * @return a <code>NodeSet</code> that represents the java source file.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (file not found...).
     */
    public static org.apache.xpath.NodeSet process(ExpressionContext context,
            String  fileName,
            boolean tokensPos) {
        return process(XSLUtil.getContextDocument(context),
                XSLUtil.evaluate(context, fileName), null, null, tokensPos);
    }

    /**
     * Returns a <code>Node</code> that represents a java source file.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="jsx:process($java-file,'','')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * Token positions are not included (see
     * {@link #process(ExpressionContext,String,String,String,boolean)}).
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the java source file.
     *   <dd><b>[Mandatory/AVT]</b> {@link org.ejen.util.arl.ArlUtil arl} sequence
     *        that defines which nodes should be accepted/removed/crossed. If the
     *        this parameter is equals to "default", then default is used.
     *   <dd><b>[Mandatory/AVT]</b> {@link org.ejen.util.arl.ArlUtil arl} sequence
     *        that defines which tokens should be accepted/removed/crossed. If the
     *        this parameter is equals to "default", then default is used.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param fileName name of the java source file.
     * @param nodesArl {@link org.ejen.util.arl.ArlUtil arl} sequence that defines
     *        which nodes should be accepted/removed/crossed.
     * @param tokensArl {@link org.ejen.util.arl.ArlUtil arl} sequence that defines
     *        which tokens should be accepted/removed/crossed.
     * @return a <code>NodeSet</code> that represents the java source file.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (file not found...).
     */
    public static org.apache.xpath.NodeSet process(ExpressionContext context,
            String  fileName,
            String  nodesArl,
            String  tokensArl) {
        return process(XSLUtil.getContextDocument(context),
                XSLUtil.evaluate(context, fileName),
                XSLUtil.evaluate(context, nodesArl),
                XSLUtil.evaluate(context, tokensArl), false);
    }

    /**
     * Returns a <code>Node</code> that represents a java source file.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="jsx:process($java-file,'','')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the java source file.
     *   <dd><b>[Mandatory/AVT]</b> {@link org.ejen.util.arl.ArlUtil arl} sequence
     *        that defines which nodes should be accepted/removed/crossed.
     *   <dd><b>[Mandatory/AVT]</b> {@link org.ejen.util.arl.ArlUtil arl} sequence
     *        that defines which tokens should be accepted/removed/crossed.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param fileName name of the java source file.
     * @param nodesArl {@link org.ejen.util.arl.ArlUtil arl} sequence that defines
     *        which nodes should be accepted/removed/crossed. If the
     *        this parameter is equals to "default", then default is used.
     * @param tokensArl {@link org.ejen.util.arl.ArlUtil arl} sequence that defines
     *        which tokens should be accepted/removed/crossed. If the
     *        this parameter is equals to "default", then default is used.
     * @param tokensPos if true, each "tok" or "stok" Node will include positions
     *        coordinates (see
     *        {@link Token#toNode(org.w3c.dom.Document,org.w3c.dom.Node,int[],boolean)
     *         Token.toNode(...)}).
     * @return a <code>NodeSet</code> that represents the java source file.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (file not found...).
     */
    public static org.apache.xpath.NodeSet process(ExpressionContext context,
            String  fileName,
            String  nodesArl,
            String  tokensArl,
            boolean tokensPos) {
        return process(XSLUtil.getContextDocument(context),
                XSLUtil.evaluate(context, fileName),
                XSLUtil.evaluate(context, nodesArl),
                XSLUtil.evaluate(context, tokensArl), tokensPos);
    }
    
    /**
     * Returns a <code>Node</code> that represents a java source file.
     * <p>
     * @param fileName name of the java source file.
     * @param nodesArl {@link org.ejen.util.arl.ArlUtil arl} sequence that defines
     *        which nodes should be accepted/removed/crossed.
     * @param tokensArl {@link org.ejen.util.arl.ArlUtil arl} sequence that defines
     *        which tokens should be accepted/removed/crossed.
     * @return a <code>NodeSet</code> that represents the java source file.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (file not found...).
     */
    protected static org.apache.xpath.NodeSet process(org.w3c.dom.Document  doc,
            String  fileName,
            String  nodesArl,
            String  tokensArl,
            boolean tokensPos) {
        BufferedReader  br = null;

        try {
            br = new BufferedReader (new FileReader (fileName));
            if (JavaParser.token_source == null) {
                new JavaParser(br);
            } else {
                JavaParser.ReInit(br);
            }
            SimpleNode sn = JavaParser.CompilationUnit();
            org.w3c.dom.Node  root = doc.createElement("unused");

            sn.toNode(doc, root,
                    getMap(nodesArl, DEFAULT_NODES_MAP, _nodesMapCache),
                    getMap(tokensArl, DEFAULT_TOKENS_MAP, _tokensMapCache),
                    tokensPos);
            return new org.apache.xpath.NodeSet(root.getChildNodes());
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
        finally {
            if (br != null) {
                try {
                    br.close();
                } catch (Exception  e) {}
                finally {
                    br = null;
                }
            }
        }
    }
    
    /**
     * Returns an <code>int</code> array based on the 'arl' expression. If the 'arl'
     * expression is already in the 'cache', just returns the existing int array, otherwise
     * creates a new one and puts it in the 'cache'.
     * <p>
     * @param arl {@link org.ejen.util.arl.ArlUtil arl} expression.
     * @param defaultMap int array used by default (if 'arl' is <code>null</code>).
     * @param cache cache used for this kind of arl (SimpleNode or Token).
     * @return the corresponding int array.
     */
    protected static int[] getMap(String  arl, int[] defaultMap, Hashtable  cache) {
        if (arl == null) {
            return defaultMap;
        }
        int[] map = (int[]) (cache.get(arl));

        if (map == null) {
            map = ArlUtil.process(arl, new int[defaultMap.length]);
            cache.put(arl, map);
        }
        return map;
    }
    
    /**
     * Returns a <code>NodeSet</code> that contains all non empty lines in a Javadoc comment.
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="jsx:parseJavadoc(tok/stok[@ki=10])"/&gt;
     * </pre></td></tr></table>
     * <p>
     * For example,
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &#047;**
     *   * Says "Hello &lt;msg&gt; !"
     *   *
     *   * @param msg the message to use in salutation.
     *   * @throws java.lang.IllegalArgumentException ...
     *   *&#047;
     * </pre></td></tr></table>
     * <p>
     * will be parsed into
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;doc-line&gt;
     *    &lt;![CDATA[ Says "Hello &lt;msg&gt; !"]]&gt;
     *  &lt;/doc-line&gt;
     *  &lt;doc-line&gt;
     *    &lt;![CDATA[ @param msg the message to use in salutations.]]&gt;
     *  &lt;/doc-line&gt;
     *  &lt;doc-line&gt;
     *    &lt;![CDATA[ @throws java.lang.IllegalArgumentException ...]]&gt;
     *  &lt;/doc-line&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory]</b> Javadoc comment to be parsed.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param comment the Javadoc comment to be parsed.
     * @return a <code>NodeSet</code> with parsed lines.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (DOM error).
     */
    public static org.apache.xpath.NodeSet parseJavadoc(ExpressionContext context, String  comment) {
        org.w3c.dom.Document  doc = XSLUtil.getContextDocument(context);

        try {
            org.apache.xpath.NodeSet ns = new org.apache.xpath.NodeSet();
            StringTokenizer  sTok = new StringTokenizer (comment, "\n\r");

            while (sTok.hasMoreTokens()) {
                String  line = sTok.nextToken();
                boolean skip = false;

                if (line.startsWith("/**")) {
                    line = line.substring(3);
                    skip = true;
                }
                if (line.endsWith("*/")) {
                    line = line.substring(0, line.length() - 2);
                }
                if (!skip) {
                    int iStar = -1;
                    boolean done = false;

                    for (int i = 0; !done && i < line.length(); i++) {
                        switch (line.charAt(i)) {
                        case ' ':
                        case '\t':
                            break;

                        case '*':
                            iStar = i;

                        default:
                            done = true;
                            break;
                        }
                    }
                    if (iStar != -1) {
                        line = line.substring(iStar + 1);
                    } else {
                        line = "";
                    }
                }
                if (line.length() > 0) {
                    org.w3c.dom.Element  elt = doc.createElement("doc-line");

                    elt.appendChild(doc.createCDATASection(line));
                    ns.addElement(elt);
                }
            }
            return ns;
        } catch (org.w3c.dom.DOMException  e) {
            throw new WrappedRuntimeException(e);
        }
    }
}
