
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext;

import org.ejen.util.DOMUtil;
import org.ejen.util.XSLUtil;
import org.w3c.dom.Node ;
import org.apache.xalan.extensions.ExpressionContext;
import org.apache.xml.utils.WrappedRuntimeException;

/**
 * XML file include utility (static methods).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:xin="org.ejen.ext.XMLInclude"</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="xml" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;xsl:copy-of select="xin:{@link #load(ExpressionContext,String) load}('{$name}.xml')"/&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class XMLInclude {

    /**
     * Protected constructor (prevents instanciation).
     */
    protected XMLInclude() {}

    /**
     * Returns the root <code>Node</code> of an XML file (with all descendants).
     * <p>
     * <table class="usage"><tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="xin:load('{$name}.xml')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> name of the XML file to be loaded.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param fileName name of the XML file.
     * @return the root <code>Node</code> of the XML file.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (file not found...).
     */
    public static Node  load(ExpressionContext context, String  fileName) {
        return load(XSLUtil.evaluate(context, fileName));
    }
    
    /**
     * Returns the root <code>Node</code> of an XML file (with all descendants).
     * <p>
     * @param fileName name of the XML file.
     * @return the root <code>Node</code> of the XML file.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (file not found...).
     */
    protected static Node  load(String  fileName) {
        try {
            return DOMUtil.parseXMLFile(fileName).getDocumentElement();
        } catch (WrappedRuntimeException e) {
            throw e;
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }
}
