
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen.ext;

import org.ejen.util.DOMUtil;
import java.lang.reflect.*;
import org.w3c.dom.Node ;
import org.w3c.dom.Element ;
import org.w3c.dom.Document ;
import org.apache.xalan.extensions.ExpressionContext;
import org.apache.xml.utils.WrappedRuntimeException;

/**
 * Class reflection utility (static methods).
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (XSL stylesheet)</th></tr>
 * <tr><td class="usage"><pre>
 *
 *  &lt;?xml version="1.0" encoding="iso-8859-1"?&gt;
 *
 *  &lt;xsl:stylesheet xmlns:xsl="http://www.w3.org/1999/XSL/Transform"
 *                  ...
 *                  <b>xmlns:jcx="org.ejen.ext.JavaClassToXML"</b>
 *                  version="1.0"&gt;
 *
 *    &lt;xsl:output method="xml" encoding="iso-8859-1"/&gt;
 *
 *    &lt;xsl:template match="ejen"&gt;
 *
 *      &lt;xsl:copy-of select="jcx:{@link #process(ExpressionContext,String) process}('java.lang.String')"/&gt;
 *      &lt;xsl:value-of select="jcx:{@link #stripType(ExpressionContext,String) stripType}('[LLjava.lang.Object;')"/&gt;
 *      &lt;xsl:value-of select="jcx:{@link #stripTypeDims(ExpressionContext,String) stripTypeDims}('[LLjava.lang.Object;')"/&gt;
 *      &lt;xsl:value-of select="jcx:{@link #dimsOfType(ExpressionContext,String) dimsOfType}('[LLjava.lang.Object;')"/&gt;
 *      &lt;xsl:value-of select="jcx:{@link #packageOfType(ExpressionContext,String) packageOfType}('[LLjava.lang.Object;')"/&gt;
 *
 *    &lt;/xsl:template&gt;
 *
 *  &lt;/xsl:stylesheet&gt;
 * </pre></td></tr></table>
 * @author F. Wolff
 * @version 1.0
 */
public class JavaClassToXML {

    /** String that represents one array dimension: <code>[]</code> */
    public static final String  DIM_STR = "[]";

    /**
     * Protected constructor (prevents instanciation).
     */
    protected JavaClassToXML() {}

    /**
     * Returns a <code>Node</code> that represents a Class definition.
     * <p>
     * <table class="usage">
     * <tr><td class="usage"><pre>
     *
     *  &lt;xsl:copy-of select="jcx:process('java.lang.String')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * The class definition returned is recursive (for each class/interface definition,
     * all extended or implemented class/interface definitions are also provided).
     * <p>
     * For "java.lang.String", the returned tree as the following structure:
     * <p>
     * <table class="usage">
     * <tr><td class="usage"><pre>
     *
     *  &lt;class-definition interface="false"&gt;
     *    &lt;package&gt;&lt;![CDATA[java.lang]]&gt;&lt;/package&gt;
     *    &lt;modifiers&gt;&lt;![CDATA[public final]]&gt;&lt;/modifiers&gt;
     *    &lt;name&gt;&lt;![CDATA[String]]&gt;&lt;/name&gt;
     *    &lt;extends&gt;
     *      &lt;class-definition interface="false"&gt;
     *        &lt;package&gt;&lt;![CDATA[java.lang]]&gt;&lt;/package&gt;
     *        &lt;modifiers&gt;&lt;![CDATA[public]]&gt;&lt;/modifiers&gt;
     *        &lt;name&gt;&lt;![CDATA[Object]]&gt;&lt;/name&gt;
     *        ...
     *      &lt;/class-definition&gt;
     *    &lt;/extends&gt;
     *    &lt;implements&gt;
     *      &lt;class-definition interface="true"&gt;
     *        &lt;package&gt;&lt;![CDATA[java.io]]&gt;&lt;/package&gt;
     *        &lt;modifiers&gt;&lt;![CDATA[public abstract interface]]&gt;&lt;/modifiers&gt;
     *        &lt;name&gt;&lt;![CDATA[Serializable]]&gt;&lt;/name&gt;
     *      &lt;/class-definition&gt;
     *      &lt;class-definition interface="true"&gt;
     *        &lt;package&gt;&lt;![CDATA[java.lang]]&gt;&lt;/package&gt;
     *        &lt;modifiers&gt;&lt;![CDATA[public abstract interface]]&gt;&lt;/modifiers&gt;
     *        &lt;name&gt;&lt;![CDATA[Comparable]]&gt;&lt;/name&gt;
     *        &lt;method&gt;
     *          &lt;modifiers&gt;&lt;![CDATA[public abstract]]&gt;&lt;/modifiers&gt;
     *          &lt;return-type&gt;&lt;![CDATA[int]]&gt;&lt;/return-type&gt;
     *          &lt;name&gt;&lt;![CDATA[compareTo]]&gt;&lt;/name&gt;
     *          &lt;parameter&gt;&lt;![CDATA[java.lang.Object]]&gt;&lt;/parameter&gt;
     *        &lt;/method&gt;
     *      &lt;/class-definition&gt;
     *    &lt;/implements&gt;
     *    &lt;field&gt;
     *      &lt;modifiers&gt;&lt;![CDATA[private]]&gt;&lt;/modifiers&gt;
     *      &lt;type&gt;&lt;![CDATA[[C]]&gt;&lt;/type&gt;
     *      &lt;name&gt;&lt;![CDATA[value]]&gt;&lt;/name&gt;
     *    &lt;/field&gt;
     *    ...
     *    &lt;constructor&gt;
     *      &lt;modifiers&gt;&lt;![CDATA[public]]&gt;&lt;/modifiers&gt;
     *    &lt;/constructor&gt;
     *    &lt;constructor&gt;
     *      &lt;modifiers&gt;&lt;![CDATA[public]]&gt;&lt;/modifiers&gt;
     *      &lt;parameter&gt;&lt;![CDATA[java.lang.String]]&gt;&lt;/parameter&gt;
     *    &lt;/constructor&gt;
     *    ...
     *    &lt;method&gt;
     *      &lt;modifiers&gt;&lt;![CDATA[public]]&gt;&lt;/modifiers&gt;
     *      &lt;return-type&gt;&lt;![CDATA[int]]&gt;&lt;/return-type&gt;
     *      &lt;name&gt;&lt;![CDATA[hashCode]]&gt;&lt;/name&gt;
     *    &lt;/method&gt;
     *    &lt;method&gt;
     *      &lt;modifiers&gt;&lt;![CDATA[public]]&gt;&lt;/modifiers&gt;
     *      &lt;return-type&gt;&lt;![CDATA[int]]&gt;&lt;/return-type&gt;
     *      &lt;name&gt;&lt;![CDATA[compareTo]]&gt;&lt;/name&gt;
     *      &lt;parameter&gt;&lt;![CDATA[java.lang.String]]&gt;&lt;/parameter&gt;
     *    &lt;/method&gt;
     *    ...
     *    &lt;class-definition interface="false"&gt;
     *      &lt;package&gt;&lt;![CDATA[java.lang]]&gt;&lt;/package&gt;
     *      &lt;modifiers/&gt;
     *      &lt;name&gt;&lt;![CDATA[String$1]]&gt;&lt;/name&gt;
     *      ...
     *    &lt;/class-definition&gt;
     *    &lt;class-definition interface="false"&gt;
     *      &lt;package&gt;&lt;![CDATA[java.lang]]&gt;&lt;/package&gt;
     *      &lt;modifiers&gt;&lt;![CDATA[private static]]&gt;&lt;/modifiers&gt;
     *      &lt;name&gt;&lt;![CDATA[String$CaseInsensitiveComparator]]&gt;&lt;/name&gt;
     *      ...
     *    &lt;/class-definition&gt;
     *      ...
     *  &lt;/class-definition&gt;
     * </pre></td></tr></table>
     * <p>
     * Note: the two last "class-definitions" are inner class definitions in
     * <code>java.lang.String</code>.
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> full Class name.
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param className Class name.
     * @return a <code>Node</code> that represents a Class definition.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (...).
     */
    public static Node  process(ExpressionContext context, String  className) {
        className = org.ejen.util.XSLUtil.evaluate(context, className);
        Document  doc = org.ejen.util.XSLUtil.getContextDocument(context);
        Class  clazz;

        try {
            clazz = Class.forName(className);
        } catch (Exception  e) {
            throw new WrappedRuntimeException("Invalid class name", e);
        }
        return process(doc, clazz);
    }
    
    /**
     * Returns a <code>Node</code> that represents a Class definition.
     * <p>
     * @param doc the current Document.
     * @param c the Class to get definition from..
     * @return a <code>Node</code> that represents a Class definition.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (...).
     */
    protected static Node  process(Document  doc, Class  c) {
        try {
            Element  root = doc.createElement("class-definition");

            root.setAttribute("interface", (c.isInterface()) ? "true" : "false");
            String  className = c.getName();
            int iLastDot = className.lastIndexOf('.');

            if (iLastDot != -1) {
                DOMUtil.createCDATANode(doc, root, "package",
                        className.substring(0, iLastDot));
                className = className.substring(iLastDot + 1);
            }
            DOMUtil.createCDATANode(doc, root, "modifiers",
                    Modifier.toString(c.getModifiers()));
            DOMUtil.createCDATANode(doc, root, "name", className);
            Class  sc = c.getSuperclass();

            if (sc != null) {
                DOMUtil.createNode(doc, root, "extends").appendChild(process(doc,
                        sc));
            }
            Class [] is = c.getInterfaces();

            if (is.length > 0) {
                Node  nis = DOMUtil.createNode(doc, root, "implements");

                for (int i = 0; i < is.length; i++) {
                    nis.appendChild(process(doc, is[i]));
                }
            }
            Field[] fs = c.getDeclaredFields();

            for (int i = 0; i < fs.length; i++) {
                Field f = fs[i];
                int mods = f.getModifiers();
                Node  nf = DOMUtil.createNode(doc, root, "field");

                DOMUtil.createCDATANode(doc, nf, "modifiers",
                        Modifier.toString(mods));
                DOMUtil.createCDATANode(doc, nf, "type", f.getType().getName());
                DOMUtil.createCDATANode(doc, nf, "name", f.getName());
                if (Modifier.isStatic(mods) && Modifier.isPublic(mods)) {
                    DOMUtil.createCDATANode(doc, nf, "init",
                            f.get(null).toString());
                }
            }
            Constructor[] cos = c.getConstructors();

            for (int i = 0; i < cos.length; i++) {
                Constructor co = cos[i];
                Node  nco = DOMUtil.createNode(doc, root, "constructor");

                DOMUtil.createCDATANode(doc, nco, "modifiers",
                        Modifier.toString(co.getModifiers()));
                Class [] ps = co.getParameterTypes();

                for (int j = 0; j < ps.length; j++) {
                    DOMUtil.createCDATANode(doc, nco, "parameter",
                            ps[j].getName());
                }
                Class [] es = co.getExceptionTypes();

                for (int j = 0; j < es.length; j++) {
                    DOMUtil.createCDATANode(doc, nco, "exception",
                            es[j].getName());
                }
            }
            Method[] ms = c.getDeclaredMethods();

            for (int i = 0; i < ms.length; i++) {
                Method m = ms[i];
                Node  nm = DOMUtil.createNode(doc, root, "method");

                DOMUtil.createCDATANode(doc, nm, "modifiers",
                        Modifier.toString(m.getModifiers()));
                DOMUtil.createCDATANode(doc, nm, "return-type",
                        m.getReturnType().getName());
                DOMUtil.createCDATANode(doc, nm, "name", m.getName());
                Class [] ps = m.getParameterTypes();

                for (int j = 0; j < ps.length; j++) {
                    DOMUtil.createCDATANode(doc, nm, "parameter",
                            ps[j].getName());
                }
                Class [] es = m.getExceptionTypes();

                for (int j = 0; j < es.length; j++) {
                    DOMUtil.createCDATANode(doc, nm, "exception",
                            es[j].getName());
                }
            }
            Class [] cs = c.getDeclaredClasses();

            for (int i = 0; i < cs.length; i++) {
                root.appendChild(process(doc, cs[i]));
            }
            return root;
        } catch (WrappedRuntimeException e) {
            throw e;
        } catch (Exception  e) {
            throw new WrappedRuntimeException(e);
        }
    }

    /**
     * Strips a type definition (see {@link java.lang.Class#getName()
     * Java signature format}), removing all array dimensions and package information.
     * <p>
     * <table class="usage">
     * <tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="jcx:stripType('[[Ljava.lang.String;')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * For example, the function will return "<code>Object</code>" for
     * "<code>[LLjava.lang.Object;</code>" (ie: <code>java.lang.Object[][]</code>)
     * or "<code>int</code>" for "<code>[[[I</code>" (ie: <code>int[][][]</code>).
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> the Java type in the "Java signature format".
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param type the Java type in the "Java signature format".
     * @return the stripped type.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (...).
     */
    public static String  stripType(ExpressionContext context, String  type) {
        type = stripTypeDims(context, type);
        int iLastDot = type.lastIndexOf('.');

        return (iLastDot != -1) ? type.substring(iLastDot + 1) : type;
    }

    /**
     * Strips a type definition (see {@link java.lang.Class#getName()
     * Java signature format}), removing only array dimensions information.
     * <p>
     * <table class="usage">
     * <tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="jcx:stripTypeDims('[[I')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * For example, the function will return "<code>java.lang.Object</code>" for
     * "<code>[LLjava.lang.Object;</code>" (ie: <code>java.lang.Object[][]</code>)
     * or "<code>int</code>" for "<code>[[[I</code>" (ie: <code>int[][][]</code>).
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> the Java type in the "Java signature format".
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param type <b>[AVT]</b> - a type definition in the "Java signature format".
     * @return the stripped type.
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (...).
     */
    public static String  stripTypeDims(ExpressionContext context, String  type) {
        type = org.ejen.util.XSLUtil.evaluate(context, type);
        if (!type.startsWith("[")) {
            return type;
        }
        char c = type.charAt(1);

        switch (c) {
        case 'B':
            return "byte";

        case 'C':
            return "char";

        case 'D':
            return "double";

        case 'F':
            return "float";

        case 'I':
            return "int";

        case 'J':
            return "long";

        case 'S':
            return "short";

        case 'Z':
            return "boolean";

        case 'L':
            break;

        default:
            throw new RuntimeException ("Unknown type signature: " + c);
        }
        
        int b = 2;
        int e = type.length() - 1;

        while (b < e && type.charAt(b) == c) {
            ++b;
        }
        return type.substring(b, e);
    }

    /**
     * Returns type array dimensions in the form of "<code>[][][]...</code>".
     * <p>
     * <table class="usage">
     * <tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="jcx:stripTypeDims('[[I')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * For example, the function will return "<code>[][]</code>" for
     * "<code>[LLjava.lang.Object;</code>" (ie: <code>java.lang.Object[][]</code>)
     * or "<code>[][][]</code>" for "<code>[[[I</code>" (ie: <code>int[][][]</code>).
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> the Java type in the "Java signature format".
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param type <b>[AVT]</b> - a type definition in the "Java signature format".
     * @return the type array dimensions (may be "").
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (...).
     */
    public static String  dimsOfType(ExpressionContext context, String  type) {
        type = org.ejen.util.XSLUtil.evaluate(context, type);
        if (!type.startsWith("[")) {
            return "";
        }
        StringBuffer  dims = new StringBuffer (DIM_STR);
        char c = type.charAt(1);
        int l = type.length();

        for (int i = 2; i < l && type.charAt(i) == c; i++) {
            dims.append(DIM_STR);
        }
        return dims.toString();
    }

    /**
     * Returns type package information.
     * <p>
     * <table class="usage">
     * <tr><td class="usage"><pre>
     *
     *  &lt;xsl:value-of select="jcx:stripTypeDims('[[java.lang.Object;')"/&gt;
     * </pre></td></tr></table>
     * <p>
     * For example, the function will return "<code>java.lang</code>" for
     * "<code>[LLjava.lang.Object;</code>" (ie: <code>java.lang.Object[][]</code>)
     * or "" for "<code>[[[I</code>" (ie: <code>int[][][]</code>).
     * <p>
     * <dd><dl><dt><b>XSLT parameters:</b>
     *   <dd><b>[Mandatory/AVT]</b> the Java type in the "Java signature format".
     * </dl></dd>
     * <p>
     * @param context automatically passed by the xalan extension mechanism.
     * @param type <b>[AVT]</b> - a type definition in the "Java signature format".
     * @return the type package (may be "").
     * @throws org.apache.xml.utils.WrappedRuntimeException errors (...).
     */
    public static String  packageOfType(ExpressionContext context, String  type) {
        type = stripTypeDims(context, type);
        int iLastDot = type.lastIndexOf('.');

        return (iLastDot != -1) ? type.substring(0, iLastDot) : "";
    }
}
