
//
// Ejen (code generation system)
// Copyright (C) 2001, 2002 François Wolff (ejen@noos.fr).
//
// This file is part of Ejen.
//
// Ejen is free software; you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation; either version 2 of the License, or
// (at your option) any later version.
//
// Ejen is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with Ejen; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
package org.ejen;

import org.ejen.util.DOMUtil;
import java.util.Properties ;
import javax.xml.transform.dom.DOMSource ;
import org.w3c.dom.Node ;
import org.w3c.dom.NodeList ;
import org.apache.xpath.XPathAPI;
import org.w3c.dom.Document ;

/**
 * Merge node class.
 * <p>
 * A merge node merges an XML file into the current in memory DOM tree.
 * Attributes give a basic way to control the merge origin and destination.
 * If this is not sufficient, it is still possible to use an
 * {@link org.ejen.ext.XMLInclude} extension.
 * <p>
 * <table class="usage">
 * <tr><th class="usage">Usage (ant build file)</th></tr>
 * <tr><td class="usage"><pre><code>
 *  &lt;?xml version="1.0" encoding="UTF-8"?&gt;
 *
 *  &lt;project name="generate" default="build"&gt;
 *
 *    &lt;taskdef name="ejen" classname="org.ejen.EjenTask"/&gt;
 *
 *    &lt;target name="build"&gt;
 *      &lt;{@link org.ejen.EjenTask ejen} ...&gt;
 *        ...
 *        <b>&lt;merge {@link #setFile(String) file}="merged.xml"
 *              [{@link #setSelect(String) select}="/ejen/entity-bean"]
 *              [{@link #setTo(String) to}="/ejen/others"]
 *        /&gt;</b>
 *        ...
 *      &lt;/ejen&gt;
 *    &lt;/target&gt;
 *
 *  &lt;/project&gt;
 * </code></pre></td></tr></table>
 * <p>
 * <b>Parent nodes</b>:
 * <ul>
 *   <li>{@link org.ejen.EjenTask ejen}
 * </ul>
 * @author F. Wolff
 * @version 1.0
 * @see org.ejen.EjenSourceNode
 * @see org.ejen.EjenSaveNode
 * @see org.ejen.ext.XMLInclude
 */
public class EjenMergeNode extends EjenChildNode {
    protected String  _file = null;
    protected String  _select = null;
    protected String  _to = null;

    /**
     * Returns the name of this EjenMergeNode (always "merge").
     * @return the name of this EjenMergeNode.
     */
    public String  nodeName() {
        return "source";
    }

    /**
     * Returns all non null attributes of this EjenSaveNode.
     * @return non null attributes of this EjenSaveNode.
     */
    public Properties  getAttributes() {
        Properties  attrs = super.getAttributes();

        if (_file != null) {
            attrs.setProperty("file", _file);
        }
        if (_select != null) {
            attrs.setProperty("select", _select);
        }
        if (_to != null) {
            attrs.setProperty("to", _to);
        }
        return attrs;
    }
        
    /**
     * <b>[mandatory/AVT]</b> - sets the file attribute.
     * @param file name of the XML file to be merged into the current in
     *        memory DOM tree.
     */
    public void setFile(String  file) {
        _file = file;
    }

    /**
     * <b>[optional/AVT]</b> - sets the select attribute.
     * @param select may be used to select only a sub-nodes set in the
     *        XML file to be merged. Default is root node.
     */
    public void setSelect(String  select) {
        _select = select;
    }

    /**
     * <b>[optional/AVT]</b> - sets the to attribute.
     * @param to may be used to change the destination node (parent
     *        node of the nodes to be merged). Default is root node.
     */
    public void setTo(String  to) {
        _to = to;
    }

    /**
     * Checks this EjenMergeNode for mandatory attributes.
     * @throws org.ejen.EjenException if file attribute is not set.
     */
    public void check() {
        super.check();
        if (_file == null) {
            throw new EjenException(this, "No 'file' attribute");
        }
    }
    
    /**
     * Executes this EjenMergeNode.
     * @throws org.ejen.EjenException if something goes wrong...
     */
    public void process() {
        super.process();
        DOMSource  src = null;

        try {
            src = (DOMSource ) (getFromGlobalContext(CTX_DOM_SOURCE));
        } catch (Exception  e) {
            throw new EjenException(this, null, e);
        }
        if (src == null) {
            throw new EjenException(this,
                    "no '" + CTX_DOM_SOURCE + "' in global context");
        }
        try {
            Document  srcDoc = (Document ) (src.getNode());
            Node  to = srcDoc.getDocumentElement();

            if (_to != null) {
                to = XPathAPI.selectSingleNode(srcDoc, evaluateAVT(_to));
            }
            Document  mergeDoc = DOMUtil.parseXMLFile(evaluateAVT(_file));

            if (_select == null) {
                to.appendChild(srcDoc.importNode(mergeDoc.getDocumentElement().cloneNode(true),
                        true));
            } else {
                NodeList  nl = XPathAPI.selectNodeList(mergeDoc,
                        evaluateAVT(_select));

                for (int i = 0; i < nl.getLength(); i++) {
                    to.appendChild(srcDoc.importNode(nl.item(i), true));
                }
            }
        } catch (EjenException e) {
            throw e;
        } catch (Exception  e) {
            throw new EjenException(this, "merge error", e);
        }
    }
}
